(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 3.0,
MathReader 3.0, or any compatible application. The data for the notebook 
starts with the line of stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[    601796,      16794]*)
(*NotebookOutlinePosition[    603002,      16833]*)
(*  CellTagsIndexPosition[    602958,      16829]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["\<\
Chapter 8
Continuous Systems with Quadratic and Cubic Nonlinearities\
\>", "Title",
  TextAlignment->Left,
  TextJustification->0],

Cell[TextData[{
  "In this chapter, we use the method of multiple scales to determine \
second-order uniform asymptotic solutions of continuous or \
distributed-parameter systems with quadratic and cubic nonlinearities. We \
consider some of the internal resonances arising from the cubic \
nonlinearities, namely, ",
  StyleBox["one-to-one",
    FontColor->RGBColor[0, 0, 1]],
  " and ",
  StyleBox["three-to-one internal resonances",
    FontColor->RGBColor[0, 0, 1]],
  ". ",
  StyleBox["Two-to-one",
    FontColor->RGBColor[0, 0, 1]],
  " internal resonances are treated in Chapter 9. We directly attack these \
continuous systems because treatment of reduced-order models obtained with \
the Galerkin procedure might lead to erroneous results. We consider buckled \
beams in Section 8.1, circular cylindrical shells in Section 8.2, and \
near-square plates in Section 8.3."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell["Preliminaries", "Section",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(Off[General::spell1, Integrate::gener]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(Needs["\<Utilities`Notation`\>"]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "To determine second-order uniform asymptotic expansions of the solutions \
of  buckled beams, circular cylindrical shells, and near-square plates by \
using the method of multiple scales, we introduce the three time scales ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0 = t\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`T\_1 = \[Epsilon]\ t\)]],
  ", and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_2 = \(\[Epsilon]\^2\) t\)]],
  ", where \[Epsilon] is a small nondimensional parameter and the order of \
the amplitude of oscillations. Moreover, we symbolize these scales according \
to"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{"Symbolize", "[", 
        TagBox[\(T\_0\),
          NotationBoxTag,
          Editable->True], "]"}], ";", 
      RowBox[{"Symbolize", "[", 
        TagBox[\(T\_1\),
          NotationBoxTag,
          Editable->True], "]"}], ";", 
      RowBox[{"Symbolize", "[", 
        TagBox[\(T\_2\),
          NotationBoxTag,
          Editable->True], "]"}], ";"}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(timeScales = {T\_0, T\_1, T\_2}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "In terms of the time scales ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`T\_1\)]],
  ", and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_2\)]],
  ", the time derivatives can be expressed as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(dt[0]\)[expr_] := expr; 
    \(dt[1]\)[expr_] := 
      Sum[\[Epsilon]\^i\ D[expr, timeScales[\([i + 1]\)]], {i, 0, 2}]; 
    \(dt[2]\)[expr_] := 
      \((\(dt[1]\)[\(dt[1]\)[expr]] // Expand)\) /. 
        \[Epsilon]\^\(i_ /; i > 3\) -> 0; \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "In the course of the analysis, we need the complex conjugates of ",
  Cell[BoxData[
      \(TraditionalForm\`A\)]],
  " and other variables. We define them by using the following rule:"
}], "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(conjugateRule = {A -> A\&_, A\&_ -> A, 
        Complex[m_, n_] -> Complex[m, \(-n\)]}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "To speed up manipulations of the integrals involved in the governing \
equations with ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  ", we introduce the rules"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(intRule1 = {int[fun_, arg2__] :> int[Expand[fun], arg2]}; \n
    intRule2 = {int[a_ + b_, arg2__] :> int[a, arg2] + int[b, arg2], 
        int[\[Epsilon]\^n_. \ fun_, arg2__] :> \[Epsilon]\^n\ int[fun, arg2], 
        int[a_\ fun_, a1_, b1___] :> 
          a\ int[fun, a1, b1] /; FreeQ[a, First[a1]], 
        int[int[a1_, a2_]\ fun_. , a3__] :> 
          int[a1, a2]\ int[fun, a3] /; FreeQ[Rest[a2], First[a2]]}; \)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
To display the outputs in easily read expressions, we introduce the rules\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(displayRule = {
        \(\(Derivative[a_, b__]\)[w_\_i_]\)[_, T\_0, __] :> 
          SequenceForm[
            If[arg1 = 
                Times\ @@\ MapIndexed[D\_\(#2[\([1]\)] - 1\)\%#1\ &, {b}]; 
              arg1 =!= 1, arg1, "\<\>"], 
            w\_i\^SequenceForm\ @@\ Table["\<'\>", {a}]], 
        \(\(Derivative[a__]\)[A\_i__]\)[__] :> 
          SequenceForm[Times\ @@\ MapIndexed[D\_\(#2[\([1]\)]\)\%#1\ &, {a}], 
            A\_i], \(\(Derivative[a__]\)[A\&_\_i__]\)[__] :> 
          SequenceForm[Times\ @@\ MapIndexed[D\_\(#2[\([1]\)]\)\%#1\ &, {a}], 
            A\&_\_i], w_\_i_[_, T\_0, __] -> w\_i, A\_i__[__] -> A\_i, 
        A\&_\_i__[__] -> A\&_\_i, 
        Exp[a_.  + b_. \ Complex[0, m_]\ T\_0 + 
              c_. \ Complex[0, n_]\ T\_0] -> 
          Exp[a + \((m*b + n*c)\) I\ T\_0]}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["8.1 Buckled Beams", "Section",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
In this section, we consider the nonlinear transverse response of a buckled \
beam possessing either a one-to-one or a three-to-one internal resonance to a \
principal parametric resonance of the higher mode. The analysis assumes a \
unimodal  buckled deflection.\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
The equation governing nonlinear transverse vibrations of a parametrically \
excited homogeneous beam is (Nayfeh and Mook, 1979)\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(eq81a = 
      m\ w\_\(t, t\) + c\ w\_t + E\_s\ I\_s\ w\_\(x, x, x, x\) + 
            \((P - f\_p\ Cos[\[CapitalOmega]\ t] - 
                  \(\(E\_s\ A\)\/\(2\ \[ScriptL]\)\) 
                    \(\[Integral]\_0\%\[ScriptL]\(\((w\_x)\)\^2\) 
                        \[DifferentialD]x\))\)\ w\_\(x, x\) == 0 /. 
        w\_m__ :> \[PartialD]\_m w[x, t]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`P\)]],
  " is a static axial load, ",
  Cell[BoxData[
      \(TraditionalForm\`f\_p\ Cos[\[CapitalOmega]\ t]\)]],
  " is an axial harmonic excitation, ",
  Cell[BoxData[
      \(TraditionalForm\`w[x, t]\)]],
  " is the transverse deflection of the beam, ",
  Cell[BoxData[
      \(TraditionalForm\`x\)]],
  " is the distance along the undeflected beam, and ",
  Cell[BoxData[
      \(TraditionalForm\`t\)]],
  " is time. Other dimensional parameters are the damping coefficient ",
  Cell[BoxData[
      \(TraditionalForm\`c\)]],
  " and the beam length \[ScriptL], mass per unit length ",
  Cell[BoxData[
      \(TraditionalForm\`m\)]],
  ", cross-sectional area ",
  Cell[BoxData[
      \(TraditionalForm\`A\)]],
  ", modulus of elasticity ",
  Cell[BoxData[
      \(TraditionalForm\`E\_s\)]],
  ", and cross-sectional area moment of inertia ",
  Cell[BoxData[
      \(TraditionalForm\`I\_s\)]],
  ". We note that the subscript s is used",
  StyleBox[" ",
    FontSlant->"Italic"],
  "with",
  StyleBox[" I ",
    FontSlant->"Italic"],
  "and",
  StyleBox[" E ",
    FontSlant->"Italic"],
  "so that",
  StyleBox[" Mathematica ",
    FontSlant->"Italic"],
  "will not confuse them with",
  StyleBox[" ",
    FontSlant->"Italic"],
  Cell[BoxData[
      \(TraditionalForm\`\@\(-1\)\)]],
  "and the exponential function ",
  StyleBox[". ",
    FontSlant->"Italic"],
  "Equation",
  StyleBox[" ",
    FontSlant->"Italic"],
  StyleBox["eq81a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " is supplemented with the four linear homogeneous boundary conditions"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(bc81a = {\[ScriptCapitalB]\_1[w[0, t]] == 0, 
        \[ScriptCapitalB]\_2[w[0, t]] == 0, 
        \[ScriptCapitalB]\_3[w[\[ScriptL], t]] == 0, 
        \[ScriptCapitalB]\_4[w[\[ScriptL], t]] == 0}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where the ",
  Cell[BoxData[
      \(\[ScriptCapitalB]\_i\)]],
  " are linear homogeneous operators."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
It is good practice to cast the governing equations and boundary conditions \
in nondimensional form. Thus, we introduce the following nondimensional \
variables:\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(T = \@\(\(m\ \[ScriptL]\^4\)\/\(\ E\_s\ I\_s\)\); \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(nondimRule = {w -> \((\[ScriptL]\ w\&~[#1/\[ScriptL], #2/T]\ &)\), 
        x -> \[ScriptL]\ x\&~, t -> T\ t\&~, 
        \[CapitalOmega] -> \[CapitalOmega]\&~/T, 
        P -> P\&~\ E\_s\ I\_s/\[ScriptL]\^2, 
        f\_p -> f\&~\ E\_s\ I\_s/\[ScriptL]\^2, 
        c -> c\&~\ \@\(E\_s\ I\_s\ m\)/\[ScriptL]\^2, 
        A -> 2\ \[Alpha]\ I\_s/\[ScriptL]\^2}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where the tilde indicates a nondimensional variable. Substituting ",
  StyleBox["nondimRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["eq81a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and dropping the tildes, we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq81b = 
      \((\(\(\(\(\(\[ScriptL]\^3\/\(E\_s\ I\_s\)\) eq81a[\([1]\)] /. 
                      Integrate -> int\) /. nondimRule\) /. s_\&~ -> s\) /. 
                int[a_, {x\ \[ScriptL], __}] -> 
                  \[ScriptL]\ int[a, {x, 0, 1}] // Expand\) // PowerExpand)
          \) == 0; \neq81b /. int -> Integrate\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{"c", " ", 
          RowBox[{
            SuperscriptBox["w", 
              TagBox[\((0, 1)\),
                Derivative],
              MultilineFunction->None], "[", \(x, t\), "]"}]}], "+", 
        RowBox[{
          SuperscriptBox["w", 
            TagBox[\((0, 2)\),
              Derivative],
            MultilineFunction->None], "[", \(x, t\), "]"}], "+", 
        RowBox[{"P", " ", 
          RowBox[{
            SuperscriptBox["w", 
              TagBox[\((2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(x, t\), "]"}]}], "-", 
        RowBox[{"f", " ", \(Cos[t\ \[CapitalOmega]]\), " ", 
          RowBox[{
            SuperscriptBox["w", 
              TagBox[\((2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(x, t\), "]"}]}], "-", 
        RowBox[{"\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SuperscriptBox["w", 
                      TagBox[\((1, 0)\),
                        Derivative],
                      MultilineFunction->None], "[", \(x, t\), "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", 
          RowBox[{
            SuperscriptBox["w", 
              TagBox[\((2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(x, t\), "]"}]}], "+", 
        RowBox[{
          SuperscriptBox["w", 
            TagBox[\((4, 0)\),
              Derivative],
            MultilineFunction->None], "[", \(x, t\), "]"}]}], "==", "0"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["8.1.1 Postbuckling Deflection", "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "The first step in analyzing the nonlinear vibrations of a buckled beam is \
the determination of the critical Euler buckling loads ",
  Cell[BoxData[
      \(TraditionalForm\`P\_k\)]],
  " and their corresponding mode shapes ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_k[x]\)]],
  ". We symbolize the ",
  Cell[BoxData[
      \(TraditionalForm\`P\_k\)]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{"Symbolize", "[", 
        TagBox[\(P\_k\),
          NotationBoxTag,
          Editable->True], "]"}], ";"}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Dropping the time derivative and neglecting the forcing and nonlinear \
terms in ",
  StyleBox["eq81b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we define the linear buckling problem as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(buckEq = 
      \(eq81b /. int -> \((0\ &)\)\) /. {P -> P\_k, f -> 0, 
          w -> \((\[Phi]\_k[#1]\ &)\)}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{\(P\_k\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
          SubsuperscriptBox["\[Phi]", "k", 
            TagBox[\((4)\),
              Derivative],
            MultilineFunction->None], "[", "x", "]"}]}], "==", "0"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Solving ",
  StyleBox["buckEq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " for ",
  Cell[BoxData[
      FormBox[
        RowBox[{
          SubsuperscriptBox["\[Phi]", "k", 
            TagBox[\((4)\),
              Derivative],
            MultilineFunction->None], "[", "x", "]"}], TraditionalForm]]],
  " yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"buckRule", "=", 
      RowBox[{
        RowBox[{"Solve", "[", 
          RowBox[{"buckEq", ",", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "k", 
                TagBox[\((4)\),
                  Derivative],
                MultilineFunction->None], "[", "x", "]"}]}], "]"}], "[", 
        \([1]\), "]"}]}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          SubsuperscriptBox["\[Phi]", "k", 
            TagBox[\((4)\),
              Derivative],
            MultilineFunction->None], "[", "x", "]"}], "\[Rule]", 
        RowBox[{\(-P\_k\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}]}], "}"}]], "Output",\

  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Next, we increase the axial load ",
  Cell[BoxData[
      \(TraditionalForm\`P\)]],
  " beyond ",
  Cell[BoxData[
      \(TraditionalForm\`P\_k\)]],
  " and assume that the postbuckling displacement is ",
  Cell[BoxData[
      \(TraditionalForm\`b\ \[Phi]\_k\)]],
  ", where ",
  Cell[BoxData[
      \(TraditionalForm\`b\)]],
  " is a nondimensional measure of the buckling level. Substituting this \
displacement into ",
  StyleBox["eq81b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", dropping the forcing term and the time derivative, and using ",
  StyleBox["intRule2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["buckRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(bEq = 
      Factor[\(\(eq81b[\([1]\)] /. {f -> 0, w -> \((b\ \[Phi]\_k[#1]\ &)\)}
                  \) /. intRule2\) /. buckRule] == 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{\(-b\), " ", 
        RowBox[{"(", 
          RowBox[{\(-P\), "+", \(P\_k\), "+", 
            RowBox[{\(b\^2\), " ", "\[Alpha]", " ", 
              RowBox[{"int", "[", 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{
                      SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                        MultilineFunction->None], "[", "x", "]"}], "2"], ",", 
                  \({x, 0, 1}\)}], "]"}]}]}], ")"}], " ", 
        RowBox[{
          SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
            MultilineFunction->None], "[", "x", "]"}]}], "==", "0"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "It follows from ",
  StyleBox["bEq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " that either ",
  Cell[BoxData[
      \(TraditionalForm\`b = 0\)]],
  " (unbuckled case) or ",
  Cell[BoxData[
      \(TraditionalForm\`b\)]],
  " is given by"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((bRule = b\^2 -> \((b\^2 /. \(Solve[bEq, b]\)[\([2]\)])\))\) /. 
      int -> Integrate\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{\(b\^2\), "\[Rule]", 
      FractionBox[\(P - P\_k\), 
        RowBox[{"\[Alpha]", " ", 
          RowBox[{\(\[Integral]\_0\%1\), 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], 
              \(\[DifferentialD]x\)}]}]}]]}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "And solving ",
  StyleBox["bEq ",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  "for ",
  Cell[BoxData[
      \(TraditionalForm\`P\)]],
  " yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((PRule = \(Solve[bEq, P]\)[\([1]\)])\) /. int -> Integrate\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"P", "\[Rule]", 
        RowBox[{\(P\_k\), "+", 
          RowBox[{\(b\^2\), " ", "\[Alpha]", " ", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], "2"], 
                \(\[DifferentialD]x\)}]}]}]}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Because ",
  Cell[BoxData[
      \(TraditionalForm\`P\)]],
  " is greater than ",
  Cell[BoxData[
      \(TraditionalForm\`P\_k\)]],
  ", ",
  StyleBox["PRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " yields the amplitude of the ",
  StyleBox["k",
    FontSlant->"Italic"],
  "th buckling mode."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Having solved the postbuckling problem, we assume that the beam deflection \
is the sum of the static buckled displacement ",
  Cell[BoxData[
      \(TraditionalForm\`b\ \[Phi]\_k[x]\)]],
  ", corresponding to the ",
  StyleBox["k",
    FontSlant->"Italic"],
  "th buckled mode, and a time-dependent relative deflection ",
  Cell[BoxData[
      \(TraditionalForm\`u[x, t]\)]],
  "; that is,"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(wRule = {w -> \((b\ \[Phi]\_k[#1] + u[#1, #2]\ &)\)}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting ",
  StyleBox["wRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["eq81b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and using ",
  StyleBox["intRule1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", ",
  StyleBox["intRule2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", ",
  StyleBox["buckRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", and ",
  StyleBox["PRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we find that the nonlinear transverse vibrations of the beam around its \
buckled configuration are given by"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(EOM = 
      \(\(\(\(eq81b /. wRule\) /. intRule1\) //. intRule2\) /. buckRule\) /. 
          PRule // ExpandAll; EOM /. int -> Integrate\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{\(-b\), " ", "f", " ", \(Cos[t\ \[CapitalOmega]]\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "-", 
        RowBox[{"2", " ", \(b\^2\), " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SuperscriptBox["u", 
                      TagBox[\((1, 0)\),
                        Derivative],
                      MultilineFunction->None], "[", \(x, t\), "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "-", 
        RowBox[{"b", " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SuperscriptBox["u", 
                      TagBox[\((1, 0)\),
                        Derivative],
                      MultilineFunction->None], "[", \(x, t\), "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{"c", " ", 
          RowBox[{
            SuperscriptBox["u", 
              TagBox[\((0, 1)\),
                Derivative],
              MultilineFunction->None], "[", \(x, t\), "]"}]}], "+", 
        RowBox[{
          SuperscriptBox["u", 
            TagBox[\((0, 2)\),
              Derivative],
            MultilineFunction->None], "[", \(x, t\), "]"}], "+", 
        RowBox[{\(P\_k\), " ", 
          RowBox[{
            SuperscriptBox["u", 
              TagBox[\((2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(x, t\), "]"}]}], "-", 
        RowBox[{"f", " ", \(Cos[t\ \[CapitalOmega]]\), " ", 
          RowBox[{
            SuperscriptBox["u", 
              TagBox[\((2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(x, t\), "]"}]}], "-", 
        RowBox[{"2", " ", "b", " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SuperscriptBox["u", 
                      TagBox[\((1, 0)\),
                        Derivative],
                      MultilineFunction->None], "[", \(x, t\), "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", 
          RowBox[{
            SuperscriptBox["u", 
              TagBox[\((2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(x, t\), "]"}]}], "-", 
        RowBox[{"\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                SuperscriptBox[
                  RowBox[{
                    SuperscriptBox["u", 
                      TagBox[\((1, 0)\),
                        Derivative],
                      MultilineFunction->None], "[", \(x, t\), "]"}], "2"], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", 
          RowBox[{
            SuperscriptBox["u", 
              TagBox[\((2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(x, t\), "]"}]}], "+", 
        RowBox[{
          SuperscriptBox["u", 
            TagBox[\((4, 0)\),
              Derivative],
            MultilineFunction->None], "[", \(x, t\), "]"}]}], "==", "0"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(BC = bc81a /. {w -> u, \[ScriptL] -> 1}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[ScriptCapitalB]\_1[u[0, t]] == 0, 
      \[ScriptCapitalB]\_2[u[0, t]] == 0, \[ScriptCapitalB]\_3[u[1, t]] == 0, 
      \[ScriptCapitalB]\_4[u[1, t]] == 0}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "In the following analysis, we focus on ",
  StyleBox["hinged-hinged",
    FontColor->RGBColor[0, 0, 1]],
  ", ",
  StyleBox["first mode-buckled beams",
    FontColor->RGBColor[0, 0, 1]],
  ". Then, the  boundary conditions for the static deflection ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_k[x]\)]],
  " can be written as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{"bBC", "=", 
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
                SubscriptBox["\[Phi]", "k",
                  MultilineFunction->None], "[", "0", "]"}], "==", "0"}], 
            ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
                  MultilineFunction->None], "[", "0", "]"}], "==", "0"}], 
            ",", 
            RowBox[{
              RowBox[{
                SubscriptBox["\[Phi]", "k",
                  MultilineFunction->None], "[", "1", "]"}], "==", "0"}], 
            ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
                  MultilineFunction->None], "[", "1", "]"}], "==", "0"}]}], 
          "}"}]}], ";"}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "and the boundary conditions for the dynamic deflection ",
  Cell[BoxData[
      \(TraditionalForm\`u[x, t]\)]],
  " can be written as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(BC1 = {u[0, t] == 0, \[PartialD]\_\(x, x\)u[x, t] == 0 /. x -> 0, 
        u[1, t] == 0, \[PartialD]\_\(x, x\)u[x, t] == 0 /. x -> 1}\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(u[0, t] == 0\), ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["u", 
              TagBox[\((2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(0, t\), "]"}], "==", "0"}], 
        ",", \(u[1, t] == 0\), ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["u", 
              TagBox[\((2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(1, t\), "]"}], "==", "0"}]}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "For the case of first buckled mode, it follows from ",
  StyleBox["buckEq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["bBC",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  StyleBox[" ",
    FontColor->RGBColor[1, 0, 1]],
  "that the first critical buckling load and associated modeshape are"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(bucklingLoad = P\_k -> \[Pi]\^2; \n
    modeshape = {\[Phi]\_k -> \((Sin[\[Pi]\ #]\ &)\)}; \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where the buckling modeshape was normalized so that ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_k[1/2] = 1\)]],
  "."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell["Linear Vibrations", "Subsubsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "The next step in analyzing the nonlinear vibrations of a buckled beam is \
the calculation of its linear natural frequencies and corresponding \
modeshapes. To this end, we linearize ",
  StyleBox["EOM ",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  "in ",
  Cell[BoxData[
      \(TraditionalForm\`u[x, t]\)]],
  ", drop the forcing, damping, and nonlinear terms, and obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(linearEq = 
      \(\(\(\(EOM /. {c -> 0, f -> 0}\) /. 
              u -> \((\[Epsilon]\ u[#1, #2]\ &)\)\) /. intRule2\) /. 
          \[Epsilon]\^\(n_ /; n > 1\) -> 0\) /. \[Epsilon] -> 1; \n
    linearEq /. int -> Integrate\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{\(-2\), " ", \(b\^2\), " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SuperscriptBox["u", 
                      TagBox[\((1, 0)\),
                        Derivative],
                      MultilineFunction->None], "[", \(x, t\), "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
          SuperscriptBox["u", 
            TagBox[\((0, 2)\),
              Derivative],
            MultilineFunction->None], "[", \(x, t\), "]"}], "+", 
        RowBox[{\(P\_k\), " ", 
          RowBox[{
            SuperscriptBox["u", 
              TagBox[\((2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(x, t\), "]"}]}], "+", 
        RowBox[{
          SuperscriptBox["u", 
            TagBox[\((4, 0)\),
              Derivative],
            MultilineFunction->None], "[", \(x, t\), "]"}]}], "==", "0"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The boundary conditions are the same as ",
  StyleBox["BC1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", which we list below as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(linearBC = BC1\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(u[0, t] == 0\), ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["u", 
              TagBox[\((2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(0, t\), "]"}], "==", "0"}], 
        ",", \(u[1, t] == 0\), ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["u", 
              TagBox[\((2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(1, t\), "]"}], "==", "0"}]}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Then, using separation of variables, we seek the solution of ",
  StyleBox["linearEq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["linearBC",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in the form"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(uSol = u -> Function[{x, t}, \[CapitalPhi][x]\ Exp[I\ \[Omega]\ t]]; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`\[Omega]\)]],
  " is the nondimensional linear natural frequency and ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalPhi][x]\)]],
  " is the corresponding modeshape. Substituting ",
  StyleBox["uSol",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["linearEq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["linearBC",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  StyleBox[" ",
    FontColor->RGBColor[1, 0, 1]],
  "and using ",
  StyleBox["intRule2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["Thread",
    FontWeight->"Bold"],
  ", we obtain the equation governing ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalPhi][x]\)]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\[CapitalPhi]Eq = 
      \(\(linearEq /. uSol\) /. intRule2 // 
          Thread[E\^\(\(-I\)\ \[Omega]\ t\)\ #, Equal]\ &\) // ExpandAll; \n
    \[CapitalPhi]Eq /. int -> Integrate\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{\(\(-\[Omega]\^2\)\ \[CapitalPhi][x]\), "+", 
        RowBox[{\(P\_k\), " ", 
          RowBox[{
            SuperscriptBox["\[CapitalPhi]", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "-", 
        RowBox[{"2", " ", \(b\^2\), " ", "\[Alpha]", " ", 
          RowBox[{"(", 
            RowBox[{\(\[Integral]\_0\%1\), 
              RowBox[{
                RowBox[{
                  RowBox[{
                    SuperscriptBox["\[CapitalPhi]", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                      MultilineFunction->None], "[", "x", "]"}]}], 
                \(\[DifferentialD]x\)}]}], ")"}], " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
          SuperscriptBox["\[CapitalPhi]", 
            TagBox[\((4)\),
              Derivative],
            MultilineFunction->None], "[", "x", "]"}]}], "==", "0"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["and the boundary conditions as", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\[CapitalPhi]BC = linearBC /. u -> \((\[CapitalPhi][#1]\ &)\)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(\[CapitalPhi][0] == 0\), ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["\[CapitalPhi]", "\[DoublePrime]",
              MultilineFunction->None], "[", "0", "]"}], "==", "0"}], ",", 
        \(\[CapitalPhi][1] == 0\), ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["\[CapitalPhi]", "\[DoublePrime]",
              MultilineFunction->None], "[", "1", "]"}], "==", "0"}]}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To find the general solution of ",
  StyleBox["\[CapitalPhi]Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["\[CapitalPhi]BC",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we follow Nayfeh, Kreider, and Anderson (1995) and treat the definite \
integral as a constant ",
  StyleBox["B",
    FontSlant->"Italic"],
  "; thus, we have a nonhomogeneous linear ordinary-differential equation. \
Hence, its general solution can be expressed as a linear combination of \
homogeneous and particular solutions. Because ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_k[x]\)]],
  " is a solution of ",
  StyleBox["buckEq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", the particular solution is proportional to ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[Phi]\_k\^\[DoublePrime]\)[x]\)]],
  ". Consequently, we express the general solution of ",
  StyleBox["\[CapitalPhi]Eq ",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  "and ",
  StyleBox["\[CapitalPhi]BC",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(cList = Table[c\_i, {i, 5}]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(shapeFunc = {Cos[\[Lambda]\_1\ x], Sin[\[Lambda]\_1\ x], 
        Cosh[\[Lambda]\_2\ x], Sinh[\[Lambda]\_2\ x], 
        \(\[Phi]\_k\^\[DoublePrime]\)[x]}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\[CapitalPhi]Sol = 
      \[CapitalPhi] -> 
        Function[x, cList . shapeFunc /. modeshape // Evaluate]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\[CapitalPhi] \[Rule] 
      Function[x, 
        Cos[x\ \[Lambda]\_1]\ c\_1 + Sin[x\ \[Lambda]\_1]\ c\_2 + 
          Cosh[x\ \[Lambda]\_2]\ c\_3 + Sinh[x\ \[Lambda]\_2]\ c\_4 - 
          \[Pi]\^2\ Sin[\[Pi]\ x]\ c\_5]\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where the ",
  Cell[BoxData[
      \(TraditionalForm\`\[Lambda]\_i\)]],
  " satisfy the characteristic equation"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(charEq = 
      \(\(\(\[CapitalPhi]Eq /. b -> 0\) /. bucklingLoad\) /. 
            \[CapitalPhi] -> \((Exp[\[Lambda]\ #]\ &)\) // 
          Thread[\(E\^\(\(-\[Lambda]\)\ x\)\) #, Equal]\ &\) // ExpandAll\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\[Pi]\^2\ \[Lambda]\^2 + \[Lambda]\^4 - \[Omega]\^2 == 0\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(\[Lambda]Rule = {\(-\[Lambda]\_1\%2\), \[Lambda]\_2\%2} -> 
          \((\[Lambda]\^2 /. Solve[charEq, \[Lambda]] // Union)\) // Thread
        \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\(-\[Lambda]\_1\%2\) \[Rule] 
        1\/2\ \((\(-\[Pi]\^2\) - \@\(\[Pi]\^4 + 4\ \[Omega]\^2\))\), 
      \[Lambda]\_2\%2 \[Rule] 
        1\/2\ \((\(-\[Pi]\^2\) + \@\(\[Pi]\^4 + 4\ \[Omega]\^2\))\)}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Substituting ",
  StyleBox["bucklingLoad",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", ",
  StyleBox["modeshape",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", and ",
  StyleBox["\[CapitalPhi]Sol",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into the left-hand sides of ",
  StyleBox["\[CapitalPhi]BC",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["\[CapitalPhi]Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and using ",
  StyleBox["intRule1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["intRule2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(expr1 = 
      Append[\(#[\([1]\)]\ &\)\ /@\ \[CapitalPhi]BC /. \[CapitalPhi]Sol, 
        \(\(\(\(\(\[CapitalPhi]Eq[\([1]\)] /. bucklingLoad\) /. modeshape
                  \) /. \[CapitalPhi]Sol\) /. intRule1\) //. intRule2\) /. 
          int -> Integrate]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({c\_1 + c\_3, \(-c\_1\)\ \[Lambda]\_1\%2 + c\_3\ \[Lambda]\_2\%2, 
      Cos[\[Lambda]\_1]\ c\_1 + Sin[\[Lambda]\_1]\ c\_2 + 
        Cosh[\[Lambda]\_2]\ c\_3 + Sinh[\[Lambda]\_2]\ c\_4, 
      \(-Cos[\[Lambda]\_1]\)\ c\_1\ \[Lambda]\_1\%2 - 
        Sin[\[Lambda]\_1]\ c\_2\ \[Lambda]\_1\%2 + 
        Cosh[\[Lambda]\_2]\ c\_3\ \[Lambda]\_2\%2 + 
        Sinh[\[Lambda]\_2]\ c\_4\ \[Lambda]\_2\%2, 
      \(-\[Pi]\^6\)\ Sin[\[Pi]\ x]\ c\_5 - 
        \[Omega]\^2\ 
          \((Cos[x\ \[Lambda]\_1]\ c\_1 + Sin[x\ \[Lambda]\_1]\ c\_2 + 
              Cosh[x\ \[Lambda]\_2]\ c\_3 + Sinh[x\ \[Lambda]\_2]\ c\_4 - 
              \[Pi]\^2\ Sin[\[Pi]\ x]\ c\_5)\) + 
        Cos[x\ \[Lambda]\_1]\ c\_1\ \[Lambda]\_1\%4 + 
        Sin[x\ \[Lambda]\_1]\ c\_2\ \[Lambda]\_1\%4 + 
        Cosh[x\ \[Lambda]\_2]\ c\_3\ \[Lambda]\_2\%4 + 
        Sinh[x\ \[Lambda]\_2]\ c\_4\ \[Lambda]\_2\%4 + 
        \[Pi]\^2\ 
          \((\[Pi]\^4\ Sin[\[Pi]\ x]\ c\_5 - 
              Cos[x\ \[Lambda]\_1]\ c\_1\ \[Lambda]\_1\%2 - 
              Sin[x\ \[Lambda]\_1]\ c\_2\ \[Lambda]\_1\%2 + 
              Cosh[x\ \[Lambda]\_2]\ c\_3\ \[Lambda]\_2\%2 + 
              Sinh[x\ \[Lambda]\_2]\ c\_4\ \[Lambda]\_2\%2)\) + 
        2\ b\^2\ \[Pi]\^2\ \[Alpha]\ Sin[\[Pi]\ x]\ 
          \((\(-\(1\/2\)\)\ \[Pi]\^4\ c\_5 + 
              \(\[Pi]\ Sin[\[Lambda]\_1]\ c\_2\ 
                  \[Lambda]\_1\%2\)\/\(\((\[Pi] - \[Lambda]\_1)\)\ 
                  \((\[Pi] + \[Lambda]\_1)\)\) - 
              \[Pi]\ c\_1\ \[Lambda]\_1\ 
                \((\(-\(\[Lambda]\_1\/\(\((\[Pi] - \[Lambda]\_1)\)\ 
                            \((\[Pi] + \[Lambda]\_1)\)\)\)\) - 
                    \(Cos[\[Lambda]\_1]\ 
                        \[Lambda]\_1\)\/\(\((\[Pi] - \[Lambda]\_1)\)\ 
                        \((\[Pi] + \[Lambda]\_1)\)\))\) - 
              \(\[Pi]\ Sinh[\[Lambda]\_2]\ c\_4\ 
                  \[Lambda]\_2\%2\)\/\(\((\[Pi] - I\ \[Lambda]\_2)\)\ 
                  \((\[Pi] + I\ \[Lambda]\_2)\)\) + 
              \[Pi]\ c\_3\ \[Lambda]\_2\ 
                \((\(-\(\[Lambda]\_2\/\(\((\[Pi] - I\ \[Lambda]\_2)\)\ 
                            \((\[Pi] + I\ \[Lambda]\_2)\)\)\)\) - 
                    \(Cosh[\[Lambda]\_2]\ 
                        \[Lambda]\_2\)\/\(\((\[Pi] - I\ \[Lambda]\_2)\)\ 
                        \((\[Pi] + I\ \[Lambda]\_2)\)\))\))\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The coefficient matrix of ",
  StyleBox["cList",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in ",
  StyleBox["expr1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " can be obtained as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(coefMat = Outer[Coefficient, expr1, cList]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({{1, 0, 1, 0, 0}, {\(-\[Lambda]\_1\%2\), 0, \[Lambda]\_2\%2, 0, 0}, {
        Cos[\[Lambda]\_1], Sin[\[Lambda]\_1], Cosh[\[Lambda]\_2], 
        Sinh[\[Lambda]\_2], 0}, {\(-Cos[\[Lambda]\_1]\)\ \[Lambda]\_1\%2, 
        \(-Sin[\[Lambda]\_1]\)\ \[Lambda]\_1\%2, 
        Cosh[\[Lambda]\_2]\ \[Lambda]\_2\%2, 
        Sinh[\[Lambda]\_2]\ \[Lambda]\_2\%2, 0}, {
        \(-\[Omega]\^2\)\ Cos[x\ \[Lambda]\_1] - 
          \[Pi]\^2\ Cos[x\ \[Lambda]\_1]\ \[Lambda]\_1\%2 + 
          Cos[x\ \[Lambda]\_1]\ \[Lambda]\_1\%4 + 
          \(2\ b\^2\ \[Pi]\^3\ \[Alpha]\ Sin[\[Pi]\ x]\ 
              \[Lambda]\_1\%2\)\/\(\((\[Pi] - \[Lambda]\_1)\)\ 
              \((\[Pi] + \[Lambda]\_1)\)\) + 
          \(2\ b\^2\ \[Pi]\^3\ \[Alpha]\ Cos[\[Lambda]\_1]\ Sin[\[Pi]\ x]\ 
              \[Lambda]\_1\%2\)\/\(\((\[Pi] - \[Lambda]\_1)\)\ 
              \((\[Pi] + \[Lambda]\_1)\)\), 
        \(-\[Omega]\^2\)\ Sin[x\ \[Lambda]\_1] - 
          \[Pi]\^2\ Sin[x\ \[Lambda]\_1]\ \[Lambda]\_1\%2 + 
          Sin[x\ \[Lambda]\_1]\ \[Lambda]\_1\%4 + 
          \(2\ b\^2\ \[Pi]\^3\ \[Alpha]\ Sin[\[Pi]\ x]\ Sin[\[Lambda]\_1]\ 
              \[Lambda]\_1\%2\)\/\(\((\[Pi] - \[Lambda]\_1)\)\ 
              \((\[Pi] + \[Lambda]\_1)\)\), 
        \(-\[Omega]\^2\)\ Cosh[x\ \[Lambda]\_2] + 
          \[Pi]\^2\ Cosh[x\ \[Lambda]\_2]\ \[Lambda]\_2\%2 - 
          \(2\ b\^2\ \[Pi]\^3\ \[Alpha]\ Sin[\[Pi]\ x]\ 
              \[Lambda]\_2\%2\)\/\(\((\[Pi] - I\ \[Lambda]\_2)\)\ 
              \((\[Pi] + I\ \[Lambda]\_2)\)\) - 
          \(2\ b\^2\ \[Pi]\^3\ \[Alpha]\ Cosh[\[Lambda]\_2]\ Sin[\[Pi]\ x]\ 
              \[Lambda]\_2\%2\)\/\(\((\[Pi] - I\ \[Lambda]\_2)\)\ 
              \((\[Pi] + I\ \[Lambda]\_2)\)\) + 
          Cosh[x\ \[Lambda]\_2]\ \[Lambda]\_2\%4, 
        \(-\[Omega]\^2\)\ Sinh[x\ \[Lambda]\_2] + 
          \[Pi]\^2\ Sinh[x\ \[Lambda]\_2]\ \[Lambda]\_2\%2 - 
          \(2\ b\^2\ \[Pi]\^3\ \[Alpha]\ Sin[\[Pi]\ x]\ Sinh[\[Lambda]\_2]\ 
              \[Lambda]\_2\%2\)\/\(\((\[Pi] - I\ \[Lambda]\_2)\)\ 
              \((\[Pi] + I\ \[Lambda]\_2)\)\) + 
          Sinh[x\ \[Lambda]\_2]\ \[Lambda]\_2\%4, 
        \(-b\^2\)\ \[Pi]\^6\ \[Alpha]\ Sin[\[Pi]\ x] + 
          \[Pi]\^2\ \[Omega]\^2\ Sin[\[Pi]\ x]}}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Substituting ",
  StyleBox["charEq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["coefMat[[5]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and eliminating the common factor ",
  Cell[BoxData[
      \(TraditionalForm\`Sin[\[Pi]\ x]\)]],
  ", we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(coefMat[\([5]\)] = Coefficient[coefMat[\([5]\)], Sin[\[Pi]\ x]]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\(2\ b\^2\ \[Pi]\^3\ \[Alpha]\ 
            \[Lambda]\_1\%2\)\/\(\((\[Pi] - \[Lambda]\_1)\)\ 
            \((\[Pi] + \[Lambda]\_1)\)\) + 
        \(2\ b\^2\ \[Pi]\^3\ \[Alpha]\ Cos[\[Lambda]\_1]\ 
            \[Lambda]\_1\%2\)\/\(\((\[Pi] - \[Lambda]\_1)\)\ 
            \((\[Pi] + \[Lambda]\_1)\)\), 
      \(2\ b\^2\ \[Pi]\^3\ \[Alpha]\ Sin[\[Lambda]\_1]\ 
          \[Lambda]\_1\%2\)\/\(\((\[Pi] - \[Lambda]\_1)\)\ 
          \((\[Pi] + \[Lambda]\_1)\)\), 
      \(-\(\(2\ b\^2\ \[Pi]\^3\ \[Alpha]\ 
                \[Lambda]\_2\%2\)\/\(\((\[Pi] - I\ \[Lambda]\_2)\)\ 
                \((\[Pi] + I\ \[Lambda]\_2)\)\)\)\) - 
        \(2\ b\^2\ \[Pi]\^3\ \[Alpha]\ Cosh[\[Lambda]\_2]\ 
            \[Lambda]\_2\%2\)\/\(\((\[Pi] - I\ \[Lambda]\_2)\)\ 
            \((\[Pi] + I\ \[Lambda]\_2)\)\), 
      \(-\(\(2\ b\^2\ \[Pi]\^3\ \[Alpha]\ Sinh[\[Lambda]\_2]\ 
              \[Lambda]\_2\%2\)\/\(\((\[Pi] - I\ \[Lambda]\_2)\)\ 
              \((\[Pi] + I\ \[Lambda]\_2)\)\)\)\), 
      \(-b\^2\)\ \[Pi]\^6\ \[Alpha] + \[Pi]\^2\ \[Omega]\^2}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Setting the determinant of ",
  StyleBox["coefMat",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " equal to zero yields the following characteristic equation for the \
natural frequencies:"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(Factor[Det[coefMat]] == 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\[Pi]\^2\ \((b\^2\ \[Pi]\^4\ \[Alpha] - \[Omega]\^2)\)\ 
        Sin[\[Lambda]\_1]\ Sinh[\[Lambda]\_2]\ 
        \((\[Lambda]\_1\%2 + \[Lambda]\_2\%2)\)\^2 == 0\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "There are two possibilities: ",
  Cell[BoxData[
      \(TraditionalForm\`b\^2\ \[Pi]\^4\ \[Alpha] - \[Omega]\^2 = 0\)]],
  " and ",
  Cell[BoxData[
      FormBox[
        RowBox[{
          RowBox[{
            StyleBox["Sin",
              FontWeight->"Bold"], "[", \(\[Lambda]\_1\), "]"}], "=", "0"}], 
        TraditionalForm]]],
  ". Hence, either"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\[Omega]Sol1 = 
      \(Solve[b\^2\ \[Pi]\^4\ \[Alpha] - \[Omega]\^2 == 0, \[Omega]]\)[
        \([2]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Omega] \[Rule] b\ \[Pi]\^2\ \@\[Alpha]}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["or", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\[Omega]Sol2 = 
      \(Solve[\(-\[Lambda]\_1\%2\) + \((n\ \[Pi])\)\^2 == 0 /. \[Lambda]Rule, 
          \[Omega]]\)[\([2]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Omega] \[Rule] n\ \@\(\(-1\) + n\^2\)\ \[Pi]\^2}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "for ",
  Cell[BoxData[
      \(TraditionalForm\`n > 1\)]],
  ". The lowest five natural frequencies are"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(frequencies = {\[Omega]\_1 -> \((\[Omega] /. \[Omega]Sol1)\), 
        \[Omega]\_2 -> \((\(\[Omega] /. \[Omega]Sol2\) /. n -> 2)\), 
        \[Omega]\_3 -> \((\(\[Omega] /. \[Omega]Sol2\) /. n -> 3)\), 
        \[Omega]\_4 -> \((\(\[Omega] /. \[Omega]Sol2\) /. n -> 4)\), 
        \[Omega]\_5 -> \((\(\[Omega] /. \[Omega]Sol2\) /. n -> 5)\)}\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Omega]\_1 \[Rule] b\ \[Pi]\^2\ \@\[Alpha], 
      \[Omega]\_2 \[Rule] 2\ \@3\ \[Pi]\^2, 
      \[Omega]\_3 \[Rule] 6\ \@2\ \[Pi]\^2, 
      \[Omega]\_4 \[Rule] 4\ \@15\ \[Pi]\^2, 
      \[Omega]\_5 \[Rule] 10\ \@6\ \[Pi]\^2}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
The first two frequencies vs. the buckling level are plotted as follows:\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(Plot[
      \[Omega] /. {\[Omega]Sol1 /. b\ \@\[Alpha] -> s, 
            \[Omega]Sol2 /. n -> 2} // Evaluate, {s, 0, 15}, 
      PlotStyle -> {RGBColor[1, 0, 0], RGBColor[0, 0, 1]}, Frame -> True, 
      FrameLabel -> {\*"\"\<b\!\(\@\[Alpha]\)\>\"", "\<\[Omega]\>"}, 
      RotateLabel -> False, 
      Epilog -> {Text["\<C\>", {2 \@ 3, 2 \(\@ 3\) \[Pi]\^2 + 5}], 
          Text[\*"\"\<\!\(b\^\(1 : 3\)\)\>\"", {2/\@3, 3}], 
          Text[\*"\"\<\!\(b\^\(1 : 1\)\)\>\"", {2 \@ 3, 3}], 
          Text[\*"\"\<\!\(b\^\(3 : 1\)\)\>\"", {6 \@ 3, 3}]}]; \)\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.0634921 0.0147151 0.00397587 [
[.02381 -0.0125 -3 -9 ]
[.02381 -0.0125 3 0 ]
[.15079 -0.0125 -3 -9 ]
[.15079 -0.0125 3 0 ]
[.27778 -0.0125 -3 -9 ]
[.27778 -0.0125 3 0 ]
[.40476 -0.0125 -3 -9 ]
[.40476 -0.0125 3 0 ]
[.53175 -0.0125 -3 -9 ]
[.53175 -0.0125 3 0 ]
[.65873 -0.0125 -6 -9 ]
[.65873 -0.0125 6 0 ]
[.78571 -0.0125 -6 -9 ]
[.78571 -0.0125 6 0 ]
[.9127 -0.0125 -6 -9 ]
[.9127 -0.0125 6 0 ]
[ 0 0 -0.125 0 ]
[.5 -0.0125 -14.5 -24.5 ]
[.5 -0.0125 14.5 -9 ]
[-0.0125 .01472 -6 -4.5 ]
[-0.0125 .01472 0 4.5 ]
[-0.0125 .09423 -12 -4.5 ]
[-0.0125 .09423 0 4.5 ]
[-0.0125 .17375 -12 -4.5 ]
[-0.0125 .17375 0 4.5 ]
[-0.0125 .25327 -12 -4.5 ]
[-0.0125 .25327 0 4.5 ]
[-0.0125 .33278 -12 -4.5 ]
[-0.0125 .33278 0 4.5 ]
[-0.0125 .4123 -18 -4.5 ]
[-0.0125 .4123 0 4.5 ]
[-0.0125 .49182 -18 -4.5 ]
[-0.0125 .49182 0 4.5 ]
[-0.0125 .57134 -18 -4.5 ]
[-0.0125 .57134 0 4.5 ]
[ 0 0 -0.125 0 ]
[-0.01875 .30902 -28 -7 ]
[-0.01875 .30902 -18 7 ]
[ 0 .61803 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
.02381 0 m
.02381 .00625 L
s
[(0)] .02381 -0.0125 0 1 Mshowa
.15079 0 m
.15079 .00625 L
s
[(2)] .15079 -0.0125 0 1 Mshowa
.27778 0 m
.27778 .00625 L
s
[(4)] .27778 -0.0125 0 1 Mshowa
.40476 0 m
.40476 .00625 L
s
[(6)] .40476 -0.0125 0 1 Mshowa
.53175 0 m
.53175 .00625 L
s
[(8)] .53175 -0.0125 0 1 Mshowa
.65873 0 m
.65873 .00625 L
s
[(10)] .65873 -0.0125 0 1 Mshowa
.78571 0 m
.78571 .00625 L
s
[(12)] .78571 -0.0125 0 1 Mshowa
.9127 0 m
.9127 .00625 L
s
[(14)] .9127 -0.0125 0 1 Mshowa
.125 Mabswid
.05556 0 m
.05556 .00375 L
s
.0873 0 m
.0873 .00375 L
s
.11905 0 m
.11905 .00375 L
s
.18254 0 m
.18254 .00375 L
s
.21429 0 m
.21429 .00375 L
s
.24603 0 m
.24603 .00375 L
s
.30952 0 m
.30952 .00375 L
s
.34127 0 m
.34127 .00375 L
s
.37302 0 m
.37302 .00375 L
s
.43651 0 m
.43651 .00375 L
s
.46825 0 m
.46825 .00375 L
s
.5 0 m
.5 .00375 L
s
.56349 0 m
.56349 .00375 L
s
.59524 0 m
.59524 .00375 L
s
.62698 0 m
.62698 .00375 L
s
.69048 0 m
.69048 .00375 L
s
.72222 0 m
.72222 .00375 L
s
.75397 0 m
.75397 .00375 L
s
.81746 0 m
.81746 .00375 L
s
.84921 0 m
.84921 .00375 L
s
.88095 0 m
.88095 .00375 L
s
.94444 0 m
.94444 .00375 L
s
.97619 0 m
.97619 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
gsave
.5 -0.0125 -75.5 -28.5 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 23.5 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 19.500000 moveto
 460.000000 19.500000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 19.500000 lineto
closepath
clip newpath
63.000000 15.500000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 15.500000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(b) show
0.000000 0.000000 0.000000 setrgbcolor
%%IncludeResource: font Math2Mono
%%IncludeFont: Math2Mono
/Math2Mono findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
68.000000 10.000000 moveto
(\\217) show
78.000000 10.000000 moveto
(!) show
82.000000 10.000000 moveto
(!) show
83.000000 10.000000 moveto
(!) show
79.000000 15.500000 moveto
%%IncludeResource: font Math1Mono
%%IncludeFont: Math1Mono
/Math1Mono findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(a) show
88.000000 15.500000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1Mono
%%+ font Math2Mono
%%+ font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1Mono
%%+ Math2Mono
%%+ Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1Mono
%%+ font Math2Mono
%%+ font Courier
grestore
0 .01472 m
.00625 .01472 L
s
[(0)] -0.0125 .01472 1 0 Mshowa
0 .09423 m
.00625 .09423 L
s
[(20)] -0.0125 .09423 1 0 Mshowa
0 .17375 m
.00625 .17375 L
s
[(40)] -0.0125 .17375 1 0 Mshowa
0 .25327 m
.00625 .25327 L
s
[(60)] -0.0125 .25327 1 0 Mshowa
0 .33278 m
.00625 .33278 L
s
[(80)] -0.0125 .33278 1 0 Mshowa
0 .4123 m
.00625 .4123 L
s
[(100)] -0.0125 .4123 1 0 Mshowa
0 .49182 m
.00625 .49182 L
s
[(120)] -0.0125 .49182 1 0 Mshowa
0 .57134 m
.00625 .57134 L
s
[(140)] -0.0125 .57134 1 0 Mshowa
.125 Mabswid
0 .03459 m
.00375 .03459 L
s
0 .05447 m
.00375 .05447 L
s
0 .07435 m
.00375 .07435 L
s
0 .11411 m
.00375 .11411 L
s
0 .13399 m
.00375 .13399 L
s
0 .15387 m
.00375 .15387 L
s
0 .19363 m
.00375 .19363 L
s
0 .21351 m
.00375 .21351 L
s
0 .23339 m
.00375 .23339 L
s
0 .27315 m
.00375 .27315 L
s
0 .29303 m
.00375 .29303 L
s
0 .31291 m
.00375 .31291 L
s
0 .35266 m
.00375 .35266 L
s
0 .37254 m
.00375 .37254 L
s
0 .39242 m
.00375 .39242 L
s
0 .43218 m
.00375 .43218 L
s
0 .45206 m
.00375 .45206 L
s
0 .47194 m
.00375 .47194 L
s
0 .5117 m
.00375 .5117 L
s
0 .53158 m
.00375 .53158 L
s
0 .55146 m
.00375 .55146 L
s
0 .59122 m
.00375 .59122 L
s
0 .6111 m
.00375 .6111 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
gsave
-0.01875 .30902 -89 -11 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 22 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 18.000000 moveto
 460.000000 18.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 18.000000 lineto
closepath
clip newpath
63.000000 14.000000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 14.000000 moveto
%%IncludeResource: font Math1Mono
%%IncludeFont: Math1Mono
/Math1Mono findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(w) show
69.000000 14.000000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1Mono
%%+ font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1Mono
%%+ Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1Mono
%%+ font Courier
grestore
.02381 .61178 m
.02381 .61803 L
s
.15079 .61178 m
.15079 .61803 L
s
.27778 .61178 m
.27778 .61803 L
s
.40476 .61178 m
.40476 .61803 L
s
.53175 .61178 m
.53175 .61803 L
s
.65873 .61178 m
.65873 .61803 L
s
.78571 .61178 m
.78571 .61803 L
s
.9127 .61178 m
.9127 .61803 L
s
.125 Mabswid
.05556 .61428 m
.05556 .61803 L
s
.0873 .61428 m
.0873 .61803 L
s
.11905 .61428 m
.11905 .61803 L
s
.18254 .61428 m
.18254 .61803 L
s
.21429 .61428 m
.21429 .61803 L
s
.24603 .61428 m
.24603 .61803 L
s
.30952 .61428 m
.30952 .61803 L
s
.34127 .61428 m
.34127 .61803 L
s
.37302 .61428 m
.37302 .61803 L
s
.43651 .61428 m
.43651 .61803 L
s
.46825 .61428 m
.46825 .61803 L
s
.5 .61428 m
.5 .61803 L
s
.56349 .61428 m
.56349 .61803 L
s
.59524 .61428 m
.59524 .61803 L
s
.62698 .61428 m
.62698 .61803 L
s
.69048 .61428 m
.69048 .61803 L
s
.72222 .61428 m
.72222 .61803 L
s
.75397 .61428 m
.75397 .61803 L
s
.81746 .61428 m
.81746 .61803 L
s
.84921 .61428 m
.84921 .61803 L
s
.88095 .61428 m
.88095 .61803 L
s
.94444 .61428 m
.94444 .61803 L
s
.97619 .61428 m
.97619 .61803 L
s
.25 Mabswid
0 .61803 m
1 .61803 L
s
.99375 .01472 m
1 .01472 L
s
.99375 .09423 m
1 .09423 L
s
.99375 .17375 m
1 .17375 L
s
.99375 .25327 m
1 .25327 L
s
.99375 .33278 m
1 .33278 L
s
.99375 .4123 m
1 .4123 L
s
.99375 .49182 m
1 .49182 L
s
.99375 .57134 m
1 .57134 L
s
.125 Mabswid
.99625 .03459 m
1 .03459 L
s
.99625 .05447 m
1 .05447 L
s
.99625 .07435 m
1 .07435 L
s
.99625 .11411 m
1 .11411 L
s
.99625 .13399 m
1 .13399 L
s
.99625 .15387 m
1 .15387 L
s
.99625 .19363 m
1 .19363 L
s
.99625 .21351 m
1 .21351 L
s
.99625 .23339 m
1 .23339 L
s
.99625 .27315 m
1 .27315 L
s
.99625 .29303 m
1 .29303 L
s
.99625 .31291 m
1 .31291 L
s
.99625 .35266 m
1 .35266 L
s
.99625 .37254 m
1 .37254 L
s
.99625 .39242 m
1 .39242 L
s
.99625 .43218 m
1 .43218 L
s
.99625 .45206 m
1 .45206 L
s
.99625 .47194 m
1 .47194 L
s
.99625 .5117 m
1 .5117 L
s
.99625 .53158 m
1 .53158 L
s
.99625 .55146 m
1 .55146 L
s
.99625 .59122 m
1 .59122 L
s
.99625 .6111 m
1 .6111 L
s
.25 Mabswid
1 0 m
1 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
1 0 0 r
.5 Mabswid
.02381 .01472 m
.06244 .03859 L
.10458 .06463 L
.14415 .08909 L
.18221 .11261 L
.22272 .13765 L
.26171 .16175 L
.30316 .18736 L
.34309 .21204 L
.3815 .23578 L
.42237 .26104 L
.46172 .28536 L
.49955 .30874 L
.53984 .33364 L
.57861 .3576 L
.61984 .38308 L
.65954 .40762 L
.69774 .43123 L
.73838 .45635 L
.77751 .48053 L
.81909 .50623 L
.85916 .53099 L
.89771 .55482 L
.93871 .58016 L
.97619 .60332 L
s
0 0 1 r
.02381 .15065 m
.06244 .15065 L
.10458 .15065 L
.14415 .15065 L
.18221 .15065 L
.22272 .15065 L
.26171 .15065 L
.30316 .15065 L
.34309 .15065 L
.3815 .15065 L
.42237 .15065 L
.46172 .15065 L
.49955 .15065 L
.53984 .15065 L
.57861 .15065 L
.61984 .15065 L
.65954 .15065 L
.69774 .15065 L
.73838 .15065 L
.77751 .15065 L
.81909 .15065 L
.85916 .15065 L
.89771 .15065 L
.93871 .15065 L
.97619 .15065 L
s
0 g
gsave
.24375 .17053 -65.5 -11 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 22 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 18.000000 moveto
 460.000000 18.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 18.000000 lineto
closepath
clip newpath
63.000000 14.000000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 14.000000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(C) show
68.000000 14.000000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.09712 .02664 -73.25 -11 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 22 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 18.000000 moveto
 460.000000 18.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 18.000000 lineto
closepath
clip newpath
63.000000 14.000000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 14.000000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(b) show
68.000000 10.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
73.000000 10.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(:) show
78.000000 10.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(3) show
83.500000 14.000000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.24375 .02664 -73.25 -11 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 22 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 18.000000 moveto
 460.000000 18.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 18.000000 lineto
closepath
clip newpath
63.000000 14.000000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 14.000000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(b) show
68.000000 10.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
73.000000 10.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(:) show
78.000000 10.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
83.500000 14.000000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.68364 .02664 -73.25 -11 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 22 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 18.000000 moveto
 460.000000 18.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 18.000000 lineto
closepath
clip newpath
63.000000 14.000000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 14.000000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(b) show
68.000000 10.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(3) show
73.000000 10.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(:) show
78.000000 10.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
83.500000 14.000000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  TextAlignment->Left,
  TextJustification->1,
  ImageSize->{288, 177.938},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/AP00`40o`000?l0003oooooob7oool0
0?oooolQoooo003ooooo8Oooo`00ooooob7oool009coool20000P_ooo`00Toooo`@00005oooo0P00
00Koool2000000?oool0003oool0Moooo`00U?ooo`040000oooooooo00000oooo`030000oooo0000
00Goool010000?ooooooo`0007Woool009Coool02P000?ooooooo`000?ooo`000?ooo`000?ooo`00
00Goool010000?ooooooo`0007Woool009Coool300000oooo`050000oooooooooooo000000Coool0
10000?ooooooo`0007Woool009Coool00`000?ooooooo`07oooo00<0003oooooool00oooo`800000
0oooo`000?ooo`1goooo002Coooo0P0000Woool00`000?ooooooo`1ooooo002Ooooo00<0003ooooo
ool0O_ooo`00Woooo`030000oooooooo07koool009ooool;0000M_ooo`00ooooob7oool00?oooolQ
oooo000/oooo00<0003oooooool06_ooo`<0000Moooo0`0001_oool200007Oooo`030000oooooooo
01Soool500000_ooo`030000oooooooo01Coool5000000Coool000000000000Foooo1@0000;oool3
00006?ooo`00:oooo`030000oooo000001_oool00`000?ooooooo`0Noooo00<0003oooooool06_oo
o`030000oooo000001_oool00`000?ooo`00000Koooo00L0003oooooooooool0003oool000005ooo
o`050000oooooooooooo000001[oool00`000?ooooooo`03oooo00<0003oooooool05oooo`00:ooo
o`030000oooo000001coool00`000?ooooooo`0Koooo100001_oool00`000?ooo`00000Koooo00<0
003oool000006oooo`070000oooooooooooo0000oooo000001Ooool00`000?ooooooo`02oooo00<0
003oooooool05oooo`040000oooooooooooo100001Soool002_oool00`000?ooo`00000Loooo00<0
003oooooool06oooo`030000oooo000001coool200007Oooo`030000oooooooo01[oool01`000?oo
ooooooooo`000?ooo`00000Goooo00<0003oooooool00_ooo`030000oooooooo01Ooool01`000?oo
ooooooooo`000?ooo`00000Ioooo000[oooo00<0003oool000006oooo`030000oooo000001goool2
00007?ooo`030000oooooooo01_oool00`000?ooo`00000Koooo00L0003oooooooooool0003oool0
00005oooo`070000oooooooooooo0000oooo000001Soool00`000?ooooooo`02oooo0P0001Woool0
02coool00`000?ooooooo`0Koooo00<0003oooooool07Oooo`030000oooooooo01_oool200007?oo
o`030000oooooooo01Woool200001?ooo`030000oooooooo01Goool200001?ooo`030000oooooooo
01Koool200001Oooo`030000oooooooo01Ooool00?oooolQoooo003ooooo8Oooo`00ooooob7oool0
0?oooolQoooo003ooooo8Oooo`009_oooo<00007oooo000Voooo00<0003oooooool00oooo`030000
oooooooo00Goool00`000?ooooooo`04oooo00<0003oooooool01Oooo`030000oooooooo00Goool0
0`000?ooooooo`04oooo00<0003oooooool01Oooo`030000oooooooo00Goool00`000?ooooooo`04
oooo00<0003oooooool01Oooo`030000oooooooo00Goool00`000?ooooooo`05oooo00<0003ooooo
ool01?ooo`030000oooooooo00Goool00`000?ooooooo`05oooo00<0003oooooool01?ooo`030000
oooooooo00Goool00`000?ooooooo`05oooo00<0003oooooool01?ooo`030000oooooooo00Goool0
0`000?ooooooo`05oooo00<0003oooooool01?ooo`030000oooooooo00Goool00`000?ooooooo`05
oooo00<0003oooooool01?ooo`030000oooooooo00Goool00`000?ooooooo`05oooo00<0003ooooo
ool01?ooo`030000oooooooo00Goool00`000?ooooooo`05oooo00<0003oooooool01?ooo`030000
oooooooo00?oool00`000?ooooooo`05oooo000Poooo00<0003oooooool00oooo`030000oooooooo
00?oool00`000?ooooooo`0Loooo00<0003oooooool06oooo`030000oooooooo01coool00`000?oo
ooooo`0Loooo00<0003oooooool07?ooo`030000oooooooo01_oool00`000?ooooooo`0Loooo00<0
003oooooool04_ooo`030000oooooooo00Goool001ooool00`000?ooo`000004oooo00<0003ooooo
ool0koooo`030000oooooooo00Goool001ooool00`000?ooo`000004oooo0`0000?oool00ol00?oo
ooooo`3Xoooo0P0000Ooool001ooool00`000?ooo`000004oooo00<0003oooooool01?ooo`;o0005
oooo100001ooool40000I_ooo`@0001Coooo00<0003oooooool01Oooo`007oooo`030000oooo0000
00Coool00`000?ooooooo`06oooo0_l000Coool010000?ooooooo`0001ooool010000?ooooooo`00
06Koool010000?ooooooo`0005;oool00`000?ooooooo`05oooo000Poooo00<0003oooooool00ooo
o`030000oooooooo00Soool2o`000_ooo`040000oooooooo00007oooo`040000oooooooo0000I_oo
o`040000oooooooo0000D_ooo`030000oooooooo00Goool002Koool200002oooo`;o000300000_oo
o`<00003oooo00@0003oooooooooool300004Oooo`<00002oooo0`0000?oool010000?oooooooooo
o`<0001Hoooo0`0000;oool300000oooo`040000oooooooooooo0`0004?oool200001oooo`009_oo
o`030000oooooooo00coool00`000?ooooooo`03oooo00<0003oooooool02?ooo`030000oooooooo
00ooool00`000?ooooooo`03oooo00<0003oooooool01oooo`030000oooooooo05Ooool00`000?oo
ooooo`04oooo00<0003oooooool01_ooo`030000oooooooo04?oool00`000?ooooooo`05oooo000V
oooo00<0003oooooool02oooo`800002o`000oooo`030000oooooooo00;oool00`000?ooooooo`02
oooo00<0003oooooool03oooo`800005oooo00<0003oooooool00_ooo`030000oooooooo00;oool0
0`000?ooooooo`1Foooo0P0000Goool00`000?ooooooo`02oooo00<0003oooooool00_ooo`030000
oooooooo04?oool00`000?ooooooo`05oooo000Voooo00<0003oooooool03oooo`04o`00oooooooo
00002_ooo`030000oooooooo01Goool00`000?ooooooo`07oooo00<0003oooooool0G_ooo`030000
oooooooo00Koool00`000?ooooooo`13oooo00<0003oooooool01Oooo`009_ooo`030000oooooooo
013oool00ol0000000000008oooo0`0001Koool200002?ooo`80001Noooo0`0000Ooool20000AOoo
o`030000oooooooo00Goool002Koool200004oooo`;o003Joooo0P0000Ooool002Koool00`000?oo
ooooo`0Doooo00?o003oooooool0f?ooo`030000oooooooo00Goool002Koool00`000?ooooooo`0E
oooo0_l00=Soool00`000?ooooooo`05oooo000Voooo00<0003oooooool05oooo`03o`00oooooooo
0=Goool00`000?ooooooo`05oooo000Voooo00<0003oooooool06?ooo`;o003Eoooo00<0003ooooo
ool01Oooo`009_ooo`80000Koooo00?o003oooooool0dOooo`800007oooo000Voooo00<0003ooooo
ool06oooo`;o003Boooo00<0003oooooool01Oooo`009_ooo`030000oooooooo01goool00ol00?oo
ooooo`3?oooo00<0003oooooool01Oooo`006_ooo`<00003oooo00<0003oooooool00oooo`030000
oooooooo01koool2o`00coooo`030000oooooooo00Goool001[oool00`000?ooooooo`02oooo00<0
003oool000001?ooo`030000oooooooo023oool00ol00?ooooooo`3<oooo00<0003oooooool01Ooo
o`006oooo`070000oooooooooooo0000oooo000000Coool300008Oooo`;o003;oooo0P0000Ooool0
01_oool01`000?ooooooooooo`000?ooo`000004oooo00<0003oooooool08oooo`;o003:oooo00<0
003oooooool01Oooo`006_ooo`030000oooo000000;oool00`000?ooo`000004oooo00<0003ooooo
ool09Oooo`;o0038oooo00<0003oooooool01Oooo`006oooo`030000oooooooo00;oool00`000?oo
ooooo`03oooo00<0003oooooool09oooo`;o0036oooo00<0003oooooool01Oooo`009_ooo`030000
oooooooo02Woool00ol00?ooooooo`33oooo00<0003oooooool01Oooo`009_ooo`80000[oooo0_l0
0<;oool200001oooo`009_ooo`030000oooooooo02coool2o`00`Oooo`030000oooooooo00Goool0
02Koool00`000?ooooooo`0^oooo00?o003oooooool0__ooo`030000oooooooo00Goool002Koool0
0`000?ooooooo`0_oooo0_l00;koool00`000?ooooooo`05oooo000Voooo0P0003;oool2o`00^ooo
o`800007oooo000Voooo00<0003oooooool0<oooo`03o`00oooooooo0;Woool00`000?ooooooo`05
oooo000Voooo00<0003oooooool0=?ooo`;o002ioooo00<0003oooooool01Oooo`009_ooo`030000
oooooooo03Koool2o`00]oooo`030000oooooooo00Goool002Koool00`000?ooooooo`0hoooo00?o
003oooooool0]?ooo`030000oooooooo00Goool002Koool200001?ooonL00?l4oooo0P0000Ooool0
02Koool00`000?ooooooo`0koooo0_l00;;oool00`000?ooooooo`05oooo000Voooo00<0003ooooo
ool0>?ooo`800003oooo00?o003oooooool0[oooo`030000oooooooo00Goool001_oool300000_oo
o`030000oooooooo00?oool00`000?ooooooo`0goooo00<0003oooooool01?ooo`;o002_oooo00<0
003oooooool01Oooo`007?ooo`060000oooooooo0000oooo00001?ooo`030000oooooooo03Ooool0
0`000?ooooooo`06oooo0_l00:goool00`000?ooooooo`05oooo000Joooo10000004oooo0000oooo
00001?ooo`<0000goooo00<0003oooooool02?ooo`03o`00oooooooo0:Woool200001oooo`006_oo
o`030000oooo000000;oool00`000?ooo`000004oooo00<0003oooooool0=oooo`030000oooo0000
00Woool2o`00Z_ooo`030000oooooooo00Goool001_oool200000_ooo`030000oooo000000Coool0
0`000?ooooooo`0goooo0`0000_oool2o`00Z?ooo`030000oooooooo00Goool001coool01@000?oo
ooooooooo`000005oooo00<0003oooooool0Aoooo`03o`00oooooooo0:Goool00`000?ooooooo`05
oooo000Voooo00<0003oooooool0B?ooo`;o002Uoooo00<0003oooooool01Oooo`009_ooo`80001;
oooo00?o003oooooool0XOooo`800007oooo000Voooo00<0003oooooool0Boooo`;o002Roooo00<0
003oooooool01Oooo`009_ooo`030000oooooooo04goool00ol00?ooooooo`2Ooooo00<0003ooooo
ool01Oooo`009_ooo`030000oooooooo04koool2o`00Woooo`030000oooooooo00Goool002Koool0
0`000?ooooooo`1@oooo00?o003oooooool0W?ooo`030000oooooooo00Goool002Koool20000D_oo
o`;o002Koooo0P0000Ooool002Koool00`000?ooooooo`1Coooo0_l009[oool00`000?ooooooo`05
oooo000Voooo00<0003oooooool0EOooo`03o`00oooooooo09Ooool00`000?ooooooo`05oooo000V
oooo00<0003oooooool0E_ooo`;o002Goooo00<0003oooooool01Oooo`009_ooo`030000oooooooo
05Soool2o`00UOooo`030000oooooooo00Goool002Koool20000Foooo`03o`00oooooooo097oool2
00001oooo`009_ooo`030000oooooooo05_oool2o`00T_ooo`030000oooooooo00Goool001[oool2
00001?ooo`030000oooooooo00?oool00`000?ooooooo`1Moooo00?o003oooooool0Soooo`030000
oooooooo00Goool001[oool00`000?ooo`000002oooo00<0003oool000001?ooo`030000oooooooo
05koool2o`00Soooo`030000oooooooo00Goool001[oool00`000?ooo`000002oooo00<0003oool0
00001?ooo`<0001Poooo00?o003oooooool0Roooo`800007oooo000Joooo0P0000?oool00`000?oo
o`000004oooo00<0003oooooool0HOooo`;o002<oooo00<0003oooooool01Oooo`006_ooo`030000
oooooooo00;oool00`000?ooo`000004oooo00<0003oooooool0Hoooo`03o`00oooooooo08Woool0
0`000?ooooooo`05oooo000Koooo0P0000?oool00`000?ooooooo`03oooo00<0003oooooool0I?oo
o`;o0029oooo00<0003oooooool01Oooo`009_ooo`030000oooooooo06Koool2o`00Qoooo`030000
oooooooo00Goool002Koool20000JOooo`03o`00oooooooo08?oool200001oooo`009_ooo`030000
oooooooo06Woool2o`00Q?ooo`030000oooooooo00Goool002Koool00`000?ooooooo`1[oooo0_l0
08;oool00`000?ooooooo`05oooo000Voooo00<0003oooooool0KOooo`03o`00oooooooo07ooool0
0`000?ooooooo`05oooo000Voooo00<0003oooooool0K_ooo`;o001ooooo00<0003oooooool01Ooo
o`009_ooo`80001aoooo00?o003oooooool0Noooo`800007oooo000Voooo00<0003oooooool0LOoo
o`;o001loooo00<0003oooooool01Oooo`002Oooo`@0000Ioooo00<0003oooooool0Loooo`03o`00
oooooooo07Woool00`000?ooooooo`05oooo0008oooo00H0003oooooool0003oool0000Hoooo00<0
003oooooool0M?ooo`;o001ioooo00<0003oooooool01Oooo`002?ooo`060000oooooooo0000oooo
00006?ooo`030000oooooooo07Koool00ol00?ooooooo`1foooo00<0003oooooool01Oooo`002?oo
o`030000oooooooo00;oool00`000?ooooooo`0Foooo0P0007Soool2o`00MOooo`800007oooo0008
oooo00<0003oooooool00_ooo`030000oooooooo01Koool00`000?ooooooo`1ioooo0_l007Coool0
0`000?ooooooo`05oooo000Voooo00<0003oooooool0Noooo`03o`00oooooooo077oool00`000?oo
ooooo`05oooo000Koooo00<0003oooooool00_ooo`030000oooooooo00?oool00`000?ooooooo`1l
oooo0_l0077oool00`000?ooooooo`05oooo000Joooo00<0003oool000000_ooo`030000oooo0000
00Coool00`000?ooooooo`1noooo0_l006ooool00`000?ooooooo`05oooo000Joooo00<0003oool0
00000_ooo`030000oooo000000Coool30000P?ooo`03o`00oooooooo06_oool200001oooo`006ooo
o`070000oooooooooooo0000oooo000000Coool00`000?ooooooo`21oooo0_l006coool00`000?oo
ooooo`05oooo000Joooo00<0003oool000000_ooo`030000oooo000000Coool00`000?ooooooo`23
oooo00?o003oooooool0JOooo`030000oooooooo00Goool001_oool00`000?ooooooo`02oooo00<0
003oooooool00oooo`030000oooooooo08Coool2o`00JOooo`030000oooooooo00Goool002Koool2
0000Qoooo`03o`00oooooooo06Goool200001oooo`009_ooo`030000oooooooo08Ooool2o`00I_oo
o`030000oooooooo00Goool002Koool00`000?ooooooo`29oooo00?o003oooooool0Hoooo`030000
oooooooo00Goool002Koool00`000?ooooooo`2:oooo0_l006?oool00`000?ooooooo`05oooo000V
oooo00<0003oooooool0S?ooo`;o001Qoooo00<0003oooooool01Oooo`009_ooo`80002?oooo00?o
003oooooool0GOooo`800007oooo000Voooo00<0003oooooool0Soooo`;o001Noooo00<0003ooooo
ool01Oooo`009_ooo`030000oooooooo097oool2o`00G?ooo`030000oooooooo00Goool002Koool0
0`000?ooooooo`2Coooo00?o003oooooool0FOooo`030000oooooooo00Goool002Koool00`000?oo
ooooo`2Doooo0_l005Woool00`000?ooooooo`05oooo000Voooo0P0009Ooool2o`00E_ooo`800007
oooo000Voooo00<0003oooooool0V?ooo`03o`00oooooooo05Coool00`000?ooooooo`05oooo000V
oooo00<0003oooooool0VOooo`;o001Doooo00<0003oooooool01Oooo`005?ooo`D00002oooo00<0
003oooooool00_ooo`030000oooooooo00?oool00`000?ooooooo`2Koooo0_l005;oool00`000?oo
ooooo`05oooo000Foooo00L0003oooooooooool0003oool000000_ooo`030000oooo000000Coool0
0`000?ooooooo`2Moooo00?o003oooooool0Coooo`030000oooooooo00Goool001Koool01`000?oo
ooooooooo`000?ooo`000002oooo00<0003oool000001?ooo`<0002Noooo0_l004koool200001ooo
o`005_ooo`070000oooooooooooo0000oooo000000;oool00`000?ooo`000004oooo00<0003ooooo
ool0X?ooo`;o001=oooo00<0003oooooool01Oooo`005_ooo`070000oooooooooooo0000oooo0000
00;oool00`000?ooo`000004oooo00<0003oooooool0X_ooo`;o001;oooo00<0003oooooool01Ooo
o`005Oooo`800004oooo00<0003oooooool00_ooo`030000oooooooo00?oool00`000?ooooooo`2T
oooo0_l004Woool00`000?ooooooo`05oooo000Voooo00<0003oooooool0Y_ooo`03o`00oooooooo
04Koool00`000?ooooooo`05oooo000Voooo0P000:Soool2o`00AOooo`800007oooo000Voooo00<0
003oooooool0ZOooo`;o0014oooo00<0003oooooool01Oooo`009_ooo`030000oooooooo0:_oool0
0ol00?ooooooo`11oooo00<0003oooooool01Oooo`009_ooo`030000oooooooo0:coool2o`00@Ooo
o`030000oooooooo00Goool002Koool20000[oooo`;o000noooo0P0000Ooool002Koool00`000?oo
ooooo`2`oooo00?o003oooooool0??ooo`030000oooooooo00Goool002Koool00`000?ooooooo`2a
oooo0_l003coool00`000?ooooooo`05oooo000Voooo00<0003oooooool0/oooo`03o`00oooooooo
03Woool00`000?ooooooo`05oooo000Voooo00<0003oooooool0]?ooo`;o000ioooo00<0003ooooo
ool01Oooo`009_ooo`80002goooo00?o003oooooool0=Oooo`800007oooo000Voooo00<0003ooooo
ool0]oooo`;o000foooo00<0003oooooool01Oooo`009_ooo`030000oooooooo0;Woool00ol00?oo
ooooo`0coooo00<0003oooooool01Oooo`005?ooo`D000001?ooo`000000000000?oool00`000?oo
ooooo`03oooo00<0003oooooool0^_ooo`;o000coooo00<0003oooooool01Oooo`005_ooo`050000
oooooooooooo000000Coool00`000?ooo`000004oooo00<0003oooooool0_?ooo`;o000aoooo00<0
003oooooool01Oooo`005_ooo`030000oooooooo00;oool01`000?ooooooooooo`000?ooo`000004
oooo0`000;koool00ol00?ooooooo`0]oooo0P0000Ooool001Koool00`000?ooooooo`02oooo00L0
003oooooooooool0003oool000001?ooo`030000oooooooo0;ooool2o`00;_ooo`030000oooooooo
00Goool001Koool01`000?ooooooooooo`000?ooo`000002oooo00<0003oool000001?ooo`030000
oooooooo0<7oool2o`00;?ooo`030000oooooooo00Goool001Goool200001?ooo`030000oooooooo
00;oool00`000?ooooooo`03oooo00<0003oooooool0`oooo`03o`00oooooooo02Woool00`000?oo
ooooo`05oooo000Voooo00<0003oooooool0a?ooo`;o000Yoooo00<0003oooooool01Oooo`009_oo
o`800037oooo0_l002Koool200001oooo`009_ooo`030000oooooooo0<Soool00ol00?ooooooo`0T
oooo00<0003oooooool01Oooo`009_ooo`030000oooooooo0<Woool2o`009?ooo`030000oooooooo
00Goool002Koool00`000?ooooooo`3;oooo0_l002;oool00`000?ooooooo`05oooo000Voooo00<0
003oooooool0cOooo`03o`00oooooooo01ooool00`000?ooooooo`05oooo000Voooo0P000<ooool2
o`007_ooo`800007oooo000Voooo00<0003oooooool0d?ooo`03o`00oooooooo01coool00`000?oo
ooooo`05oooo000Voooo00<0003oooooool0dOooo`;o000Loooo00<0003oooooool01Oooo`009_oo
o`030000oooooooo0=?oool00ol00?ooooooo`0Ioooo00<0003oooooool01Oooo`009_ooo`80003E
oooo0_l001Soool200001oooo`009_ooo`030000oooooooo0=Koool00ol00?ooooooo`0Foooo00<0
003oooooool01Oooo`009_ooo`030000oooooooo0=Ooool2o`005_ooo`030000oooooooo00Goool0
01Coool500000_ooo`<00002oooo00<0003oooooool00oooo`030000oooooooo0=Woool2o`005?oo
o`030000oooooooo00Goool001Koool00`000?ooooooo`03oooo00H0003oooooool0003oool00004
oooo00<0003oooooool0foooo`03o`00oooooooo017oool00`000?ooooooo`05oooo000Foooo00@0
003oooooooooool4000000Coool0003oool00004oooo0`000=coool2o`004?ooo`800007oooo000F
oooo00L0003oooooooooool0003oool000000_ooo`030000oooo000000Coool00`000?ooooooo`3N
oooo0_l000ooool00`000?ooooooo`05oooo000Foooo00<0003oooooool00_ooo`800002oooo00<0
003oool000001?ooo`030000oooooooo0>3oool00ol00?ooooooo`0<oooo00<0003oooooool01Ooo
o`005Oooo`800005oooo00D0003oooooooooool000001Oooo`030000oooooooo0>7oool2o`003?oo
o`030000oooooooo00Goool002Koool00`000?ooooooo`3Soooo00?o003oooooool02Oooo`030000
oooooooo00Goool002Koool20000iOooo`;o0008oooo0P0000Ooool002Koool00`000?ooooooo`3V
oooo00?o003oooooool01_ooo`030000oooooooo00Goool002Koool00`000?ooooooo`3Woooo0_l0
00Koool00`000?ooooooo`05oooo000Voooo00<0003oooooool0jOooo`03o`00oooooooo00?oool0
0`000?ooooooo`05oooo000Voooo00<0003oooooool0koooo`030000oooooooo00Goool002Koool2
00001?ooo`030000oooooooo01coool00`000?ooooooo`0Koooo00<0003oooooool07?ooo`030000
oooooooo01coool00`000?ooooooo`0Loooo00<0003oooooool06oooo`030000oooooooo01coool0
0`000?ooooooo`0Aoooo0P0000Ooool002Kooooc00001oooo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-2.90434, -32.3348, 
  0.0651476, 1.04037}}]
}, Open  ]],

Cell[TextData[{
  "where the low-frequency mode (",
  StyleBox["red",
    FontColor->RGBColor[1, 0, 0]],
  " line) is symmetric and the second mode (",
  StyleBox["blue",
    FontColor->RGBColor[0, 0, 1]],
  " line) is antisymmetric. As the buckling level increases from zero, the \
frequency ",
  Cell[BoxData[
      \(TraditionalForm\`\[Omega]\_1\)]],
  " of the first mode increases from zero and crosses the frequency ",
  Cell[BoxData[
      \(TraditionalForm\`\[Omega]\_2\)]],
  " of the second mode at point C. Hence, there are two possible ",
  StyleBox["three-to-one",
    FontColor->RGBColor[0, 0, 1]],
  " internal resonances: ",
  Cell[BoxData[
      \(TraditionalForm\`\[Omega]\_2 = 3\ \[Omega]\_1\)]],
  " when ",
  Cell[BoxData[
      \(TraditionalForm\`b \@ \[Alpha] = b\^\(1 : 3\)\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Omega]\_1 = 3\ \[Omega]\_2\)]],
  " when ",
  Cell[BoxData[
      \(TraditionalForm\`b\^\(3 : 1\)\)]],
  ". There is also a possible ",
  StyleBox["one-to-one",
    FontColor->RGBColor[0, 0, 1]],
  " internal resonance ",
  Cell[BoxData[
      \(TraditionalForm\`\[Omega]\_2 = \[Omega]\_1\)]],
  " when ",
  Cell[BoxData[
      \(TraditionalForm\`b \@ \[Alpha] = b\^\(1 : 1\)\)]],
  ". Whether or not these candidates for internal resonances will be \
activated depends on the corresponding modeshapes. In this section, we \
consider the case of one-to-one internal resonance between the first and \
second modes. In Section 8.1.4, we consider three-to-one internal resonances \
between the first two modes. In Sections 8.1.5 and 8.1.6, we consider \
one-to-one and three-to-one internal resonances between the first and third \
modes and the first and fouth modes, respectively."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
In the next section, we consider a combination of a one-to-one internal \
resonance between the lowest two modes and a principal parametric resonance \
of the second mode; that is,\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(omgList = {\[Omega]\_1, \[Omega]\_2}; \n
    ResonanceConds = {
        \[Omega]\_2 == \[Omega]\_1 + \[Epsilon]\^2\ \[Sigma]\_1, 
        \[CapitalOmega] == 2  \[Omega]\_2 + \[Epsilon]\^2\ \[Sigma]\_2}; \)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where the ",
  Cell[BoxData[
      \(TraditionalForm\`\[Sigma]\_i\)]],
  " are detuning parameters that describe the nearness of the resonances. The \
buckling level for which the natural frequencies of the first and second \
modes are equal (i.e., ",
  Cell[BoxData[
      \(\[Omega]\_2 = \[Omega]\_1\)]],
  ") is given by"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(values1 = 
      \(Solve[\[Omega]\_1 == \[Omega]\_2 /. frequencies, b]\)[\([1]\)]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({b \[Rule] \(2\ \@3\)\/\@\[Alpha]}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The corresponding ",
  Cell[BoxData[
      \(TraditionalForm\`\[Lambda]\_m\)]],
  " are"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\[Lambda]RuleN = 
      \(Solve[\(\(\(\[Lambda]Rule /. \[Omega] -> \[Omega]\_1\) /. frequencies
                \) /. values1\) /. Rule -> Equal, {\[Lambda]\_1, 
            \[Lambda]\_2}]\)[\([4]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Lambda]\_1 \[Rule] 2\ \[Pi], \[Lambda]\_2 \[Rule] \@3\ \[Pi]}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
In order that the influence of the damping and nonlinearity balance the \
influence of the forcing, we scale the damping and forcing terms as\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(scaling = {c -> 2  \[Epsilon]\^2\ \[Mu]\ , f -> \[Epsilon]\^2\ f}; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting for the natural frequency ",
  Cell[BoxData[
      \(TraditionalForm\`\[Omega]\_1\)]],
  " back into ",
  StyleBox["coefMat",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we determine the modeshape as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(NullSpace[\(coefMat /. \[Omega] -> \[Omega]\_1\) /. frequencies]\)[
          \([1]\)] . shapeFunc /. modeshape\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(-\[Pi]\^2\)\ Sin[\[Pi]\ x]\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["which we normalize as", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(shape1 = 
      c\ Sin[\[Pi]\ x] /. 
        \(Solve[\[Integral]\_0\%1 c\^2\ \(Sin[\[Pi]\ x]\^2\) 
                  \[DifferentialD]x == 1, c]\)[\([2]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\@2\ Sin[\[Pi]\ x]\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Substituting ",
  Cell[BoxData[
      \(TraditionalForm\`\[Omega]\_2\)]],
  " into ",
  StyleBox["coefMat",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we determine the modeshape as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(NullSpace[
            \(\(coefMat /. \[Lambda]RuleN\) /. \[Omega] -> \[Omega]\_2\) /. 
              frequencies]\)[\([1]\)] . shapeFunc /. \[Lambda]RuleN\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(Sin[2\ \[Pi]\ x]\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["which we also normalize as", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(shape2 = 
      c\ Sin[2  \[Pi]\ x] /. 
        \(Solve[\[Integral]\_0\%1 c\^2\ \(Sin[2  \[Pi]\ x]\^2\) 
                  \[DifferentialD]x == 1, c]\)[\([2]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\@2\ Sin[2\ \[Pi]\ x]\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
In a similar fashion, we find that the next three normalized modeshapes are\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(shape3 = \@2\ Sin[3\ \[Pi]\ x]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(shape4 = \@2\ Sin[4\ \[Pi]\ x]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(shape5 = \@2\ Sin[5\ \[Pi]\ x]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "To treat a one-to-one internal resonance between the first and second \
modes, we define ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_k\)]],
  " and the first two eigenmodes of the buckled beam as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(modeshapes = 
      Join[modeshape, {\[CapitalPhi]\_1 -> Function[x, shape1 // Evaluate], 
          \[CapitalPhi]\_2 -> Function[x, shape2 // Evaluate]}]; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["8.1.2 Perturbation Analysis", "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "We use the method of multiple scales to directly attack ",
  StyleBox["EOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["BC1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ". To transform the time derivatives in ",
  StyleBox["EOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in terms of the scales ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`T\_1\)]],
  ", and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_2\)]],
  ", we let"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(multiScales = {u[x_, t] -> u[x, T\_0, T\_1, T\_2], 
        \(\(Derivative[m_, n_]\)[u_]\)[x_, t] -> 
          \(dt[n]\)[D[u[x, T\_0, T\_1, T\_2], {x, m}]], t -> T\_0}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
Then, we seek a second-order approximate solution in the form\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(solRule = 
      u -> \((Evaluate[Sum[\[Epsilon]\^j\ u\_j[#1, #2, #3, #4], {j, 3}]]\ &)
          \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(u \[Rule] 
      \((\[Epsilon]\ u\_1[#1, #2, #3, #4] + 
            \[Epsilon]\^2\ u\_2[#1, #2, #3, #4] + 
            \[Epsilon]\^3\ u\_3[#1, #2, #3, #4]&)\)\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`#1\)]],
  " stands for ",
  Cell[BoxData[
      \(TraditionalForm\`x\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`#2\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`#3\)]],
  ", and ",
  Cell[BoxData[
      \(TraditionalForm\`#4\)]],
  " stand for ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0, T\_1\)]],
  ", and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_2\)]],
  ", respectively."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting ",
  StyleBox["multiScales",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", ",
  StyleBox["solRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", and ",
  StyleBox["scaling",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["EOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["BC1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", expanding the result for small \[Epsilon], and discarding terms of order \
higher than ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\^3\)]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(eq81c = 
      \((\(\(\(\(\(Join[{EOM}, BC1] /. multiScales\) /. solRule\) /. scaling
                    \) /. intRule1\) //. intRule2 // TrigToExp\) // ExpandAll)
          \) /. \[Epsilon]\^\(n_ /; n > 3\) -> 0; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[
"Equating coefficients of like powers of \[Epsilon], we obtain"], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eqEps = 
      \(\(Thread[CoefficientList[Subtract\ @@\ #, \[Epsilon]] == 0]\ &\)\ /@\ 
            eq81c // Transpose\) // Rest; \neqEps /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
                RowBox[{\(P\_k\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["u", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ u, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "+", 
                InterpretationBox[
                  RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                    SubsuperscriptBox["u", "1", 
                      
                      InterpretationBox[
                        \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                        SequenceForm[ "'", "'", "'", "'"],
                        Editable->False]]}],
                  SequenceForm[ "", 
                    Power[ 
                      Subscript[ u, 1], 
                      SequenceForm[ "'", "'", "'", "'"]]],
                  Editable->False], "+", 
                InterpretationBox[
                  RowBox[{\(D\_0\%2\), "\[InvisibleSpace]", 
                    SubsuperscriptBox["u", "1", 
                      InterpretationBox["\[Null]",
                        SequenceForm[ ],
                        Editable->False]]}],
                  SequenceForm[ 
                    Power[ 
                      Subscript[ D, 0], 2], 
                    Power[ 
                      Subscript[ u, 1], 
                      SequenceForm[ ]]],
                  Editable->False], "-", 
                RowBox[{"2", " ", \(b\^2\), " ", "\[Alpha]", " ", 
                  RowBox[{"int", "[", 
                    RowBox[{
                      RowBox[{
                        RowBox[{"(", 
                          InterpretationBox[
                            RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                              SubsuperscriptBox["u", "1", 
                                InterpretationBox["\<\"'\"\>",
                                  SequenceForm[ "'"],
                                  Editable->False]]}],
                            SequenceForm[ "", 
                              Power[ 
                                Subscript[ u, 1], 
                                SequenceForm[ "'"]]],
                            Editable->False], ")"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}]}], ",", 
                      \({x, 0, 1}\)}], "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
                      MultilineFunction->None], "[", "x", "]"}]}]}], "==", 
              "0"}], ",", \(u\_1 == 0\), ",", 
            RowBox[{
              InterpretationBox[
                RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                  SubsuperscriptBox["u", "1", 
                    InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                      SequenceForm[ "'", "'"],
                      Editable->False]]}],
                SequenceForm[ "", 
                  Power[ 
                    Subscript[ u, 1], 
                    SequenceForm[ "'", "'"]]],
                Editable->False], "==", "0"}], ",", \(u\_1 == 0\), ",", 
            RowBox[{
              InterpretationBox[
                RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                  SubsuperscriptBox["u", "1", 
                    InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                      SequenceForm[ "'", "'"],
                      Editable->False]]}],
                SequenceForm[ "", 
                  Power[ 
                    Subscript[ u, 1], 
                    SequenceForm[ "'", "'"]]],
                Editable->False], "==", "0"}]}], "}"}], ",", 
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
                RowBox[{\(-2\), " ", "b", " ", "\[Alpha]", " ", 
                  RowBox[{"int", "[", 
                    RowBox[{
                      RowBox[{
                        RowBox[{"(", 
                          InterpretationBox[
                            RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                              SubsuperscriptBox["u", "1", 
                                InterpretationBox["\<\"'\"\>",
                                  SequenceForm[ "'"],
                                  Editable->False]]}],
                            SequenceForm[ "", 
                              Power[ 
                                Subscript[ u, 1], 
                                SequenceForm[ "'"]]],
                            Editable->False], ")"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}]}], ",", 
                      \({x, 0, 1}\)}], "]"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["u", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ u, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{\(P\_k\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["u", "2", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ u, 2], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "+", 
                InterpretationBox[
                  RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                    SubsuperscriptBox["u", "2", 
                      
                      InterpretationBox[
                        \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                        SequenceForm[ "'", "'", "'", "'"],
                        Editable->False]]}],
                  SequenceForm[ "", 
                    Power[ 
                      Subscript[ u, 2], 
                      SequenceForm[ "'", "'", "'", "'"]]],
                  Editable->False], "+", 
                InterpretationBox[
                  RowBox[{\(D\_0\%2\), "\[InvisibleSpace]", 
                    SubsuperscriptBox["u", "2", 
                      InterpretationBox["\[Null]",
                        SequenceForm[ ],
                        Editable->False]]}],
                  SequenceForm[ 
                    Power[ 
                      Subscript[ D, 0], 2], 
                    Power[ 
                      Subscript[ u, 2], 
                      SequenceForm[ ]]],
                  Editable->False], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\ D\_1\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["u", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Times[ 
                          Subscript[ D, 0], 
                          Subscript[ D, 1]], 
                        Power[ 
                          Subscript[ u, 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{
                \(1\/2\), " ", "b", " ", 
                  \(E\^\(\(-I\)\ T\_0\ \[CapitalOmega]\)\), " ", "f", " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
                      MultilineFunction->None], "[", "x", "]"}]}], "-", 
                RowBox[{
                \(1\/2\), " ", "b", " ", \(E\^\(I\ T\_0\ \[CapitalOmega]\)\), 
                  " ", "f", " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
                      MultilineFunction->None], "[", "x", "]"}]}], "-", 
                RowBox[{"b", " ", "\[Alpha]", " ", 
                  RowBox[{"int", "[", 
                    RowBox[{
                      SuperscriptBox[
                        RowBox[{"(", 
                          InterpretationBox[
                            RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                              SubsuperscriptBox["u", "1", 
                                InterpretationBox["\<\"'\"\>",
                                  SequenceForm[ "'"],
                                  Editable->False]]}],
                            SequenceForm[ "", 
                              Power[ 
                                Subscript[ u, 1], 
                                SequenceForm[ "'"]]],
                            Editable->False], ")"}], "2"], ",", 
                      \({x, 0, 1}\)}], "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
                      MultilineFunction->None], "[", "x", "]"}]}], "-", 
                RowBox[{"2", " ", \(b\^2\), " ", "\[Alpha]", " ", 
                  RowBox[{"int", "[", 
                    RowBox[{
                      RowBox[{
                        RowBox[{"(", 
                          InterpretationBox[
                            RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                              SubsuperscriptBox["u", "2", 
                                InterpretationBox["\<\"'\"\>",
                                  SequenceForm[ "'"],
                                  Editable->False]]}],
                            SequenceForm[ "", 
                              Power[ 
                                Subscript[ u, 2], 
                                SequenceForm[ "'"]]],
                            Editable->False], ")"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}]}], ",", 
                      \({x, 0, 1}\)}], "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
                      MultilineFunction->None], "[", "x", "]"}]}]}], "==", 
              "0"}], ",", \(u\_2 == 0\), ",", 
            RowBox[{
              InterpretationBox[
                RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                  SubsuperscriptBox["u", "2", 
                    InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                      SequenceForm[ "'", "'"],
                      Editable->False]]}],
                SequenceForm[ "", 
                  Power[ 
                    Subscript[ u, 2], 
                    SequenceForm[ "'", "'"]]],
                Editable->False], "==", "0"}], ",", \(u\_2 == 0\), ",", 
            RowBox[{
              InterpretationBox[
                RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                  SubsuperscriptBox["u", "2", 
                    InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                      SequenceForm[ "'", "'"],
                      Editable->False]]}],
                SequenceForm[ "", 
                  Power[ 
                    Subscript[ u, 2], 
                    SequenceForm[ "'", "'"]]],
                Editable->False], "==", "0"}]}], "}"}], ",", 
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
                RowBox[{
                \(-\(1\/2\)\), " ", \(E\^\(\(-I\)\ T\_0\ \[CapitalOmega]\)\), 
                  " ", "f", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["u", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ u, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{
                \(1\/2\), " ", \(E\^\(I\ T\_0\ \[CapitalOmega]\)\), " ", "f", 
                  " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["u", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ u, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"\[Alpha]", " ", 
                  RowBox[{"int", "[", 
                    RowBox[{
                      SuperscriptBox[
                        RowBox[{"(", 
                          InterpretationBox[
                            RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                              SubsuperscriptBox["u", "1", 
                                InterpretationBox["\<\"'\"\>",
                                  SequenceForm[ "'"],
                                  Editable->False]]}],
                            SequenceForm[ "", 
                              Power[ 
                                Subscript[ u, 1], 
                                SequenceForm[ "'"]]],
                            Editable->False], ")"}], "2"], ",", 
                      \({x, 0, 1}\)}], "]"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["u", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ u, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"2", " ", "b", " ", "\[Alpha]", " ", 
                  RowBox[{"int", "[", 
                    RowBox[{
                      RowBox[{
                        RowBox[{"(", 
                          InterpretationBox[
                            RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                              SubsuperscriptBox["u", "2", 
                                InterpretationBox["\<\"'\"\>",
                                  SequenceForm[ "'"],
                                  Editable->False]]}],
                            SequenceForm[ "", 
                              Power[ 
                                Subscript[ u, 2], 
                                SequenceForm[ "'"]]],
                            Editable->False], ")"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}]}], ",", 
                      \({x, 0, 1}\)}], "]"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["u", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ u, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"2", " ", "b", " ", "\[Alpha]", " ", 
                  RowBox[{"int", "[", 
                    RowBox[{
                      RowBox[{
                        RowBox[{"(", 
                          InterpretationBox[
                            RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                              SubsuperscriptBox["u", "1", 
                                InterpretationBox["\<\"'\"\>",
                                  SequenceForm[ "'"],
                                  Editable->False]]}],
                            SequenceForm[ "", 
                              Power[ 
                                Subscript[ u, 1], 
                                SequenceForm[ "'"]]],
                            Editable->False], ")"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}]}], ",", 
                      \({x, 0, 1}\)}], "]"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["u", "2", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ u, 2], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{\(P\_k\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["u", "3", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ u, 3], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "+", 
                InterpretationBox[
                  RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                    SubsuperscriptBox["u", "3", 
                      
                      InterpretationBox[
                        \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                        SequenceForm[ "'", "'", "'", "'"],
                        Editable->False]]}],
                  SequenceForm[ "", 
                    Power[ 
                      Subscript[ u, 3], 
                      SequenceForm[ "'", "'", "'", "'"]]],
                  Editable->False], "+", 
                RowBox[{"2", " ", "\[Mu]", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["u", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Subscript[ D, 0], 
                        Power[ 
                          Subscript[ u, 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}]}], "+", 
                InterpretationBox[
                  RowBox[{\(D\_0\%2\), "\[InvisibleSpace]", 
                    SubsuperscriptBox["u", "3", 
                      InterpretationBox["\[Null]",
                        SequenceForm[ ],
                        Editable->False]]}],
                  SequenceForm[ 
                    Power[ 
                      Subscript[ D, 0], 2], 
                    Power[ 
                      Subscript[ u, 3], 
                      SequenceForm[ ]]],
                  Editable->False], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\ D\_1\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["u", "2", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Times[ 
                          Subscript[ D, 0], 
                          Subscript[ D, 1]], 
                        Power[ 
                          Subscript[ u, 2], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}]}], "+", 
                InterpretationBox[
                  RowBox[{\(D\_1\%2\), "\[InvisibleSpace]", 
                    SubsuperscriptBox["u", "1", 
                      InterpretationBox["\[Null]",
                        SequenceForm[ ],
                        Editable->False]]}],
                  SequenceForm[ 
                    Power[ 
                      Subscript[ D, 1], 2], 
                    Power[ 
                      Subscript[ u, 1], 
                      SequenceForm[ ]]],
                  Editable->False], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\ D\_2\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["u", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Times[ 
                          Subscript[ D, 0], 
                          Subscript[ D, 2]], 
                        Power[ 
                          Subscript[ u, 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"2", " ", "b", " ", "\[Alpha]", " ", 
                  RowBox[{"int", "[", 
                    RowBox[{
                      RowBox[{
                        RowBox[{"(", 
                          InterpretationBox[
                            RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                              SubsuperscriptBox["u", "1", 
                                InterpretationBox["\<\"'\"\>",
                                  SequenceForm[ "'"],
                                  Editable->False]]}],
                            SequenceForm[ "", 
                              Power[ 
                                Subscript[ u, 1], 
                                SequenceForm[ "'"]]],
                            Editable->False], ")"}], " ", 
                        RowBox[{"(", 
                          InterpretationBox[
                            RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                              SubsuperscriptBox["u", "2", 
                                InterpretationBox["\<\"'\"\>",
                                  SequenceForm[ "'"],
                                  Editable->False]]}],
                            SequenceForm[ "", 
                              Power[ 
                                Subscript[ u, 2], 
                                SequenceForm[ "'"]]],
                            Editable->False], ")"}]}], ",", \({x, 0, 1}\)}], 
                    "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
                      MultilineFunction->None], "[", "x", "]"}]}], "-", 
                RowBox[{"2", " ", \(b\^2\), " ", "\[Alpha]", " ", 
                  RowBox[{"int", "[", 
                    RowBox[{
                      RowBox[{
                        RowBox[{"(", 
                          InterpretationBox[
                            RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                              SubsuperscriptBox["u", "3", 
                                InterpretationBox["\<\"'\"\>",
                                  SequenceForm[ "'"],
                                  Editable->False]]}],
                            SequenceForm[ "", 
                              Power[ 
                                Subscript[ u, 3], 
                                SequenceForm[ "'"]]],
                            Editable->False], ")"}], " ", 
                        RowBox[{
                          SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                            MultilineFunction->None], "[", "x", "]"}]}], ",", 
                      \({x, 0, 1}\)}], "]"}], " ", 
                  RowBox[{
                    SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
                      MultilineFunction->None], "[", "x", "]"}]}]}], "==", 
              "0"}], ",", \(u\_3 == 0\), ",", 
            RowBox[{
              InterpretationBox[
                RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                  SubsuperscriptBox["u", "3", 
                    InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                      SequenceForm[ "'", "'"],
                      Editable->False]]}],
                SequenceForm[ "", 
                  Power[ 
                    Subscript[ u, 3], 
                    SequenceForm[ "'", "'"]]],
                Editable->False], "==", "0"}], ",", \(u\_3 == 0\), ",", 
            RowBox[{
              InterpretationBox[
                RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                  SubsuperscriptBox["u", "3", 
                    InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                      SequenceForm[ "'", "'"],
                      Editable->False]]}],
                SequenceForm[ "", 
                  Power[ 
                    Subscript[ u, 3], 
                    SequenceForm[ "'", "'"]]],
                Editable->False], "==", "0"}]}], "}"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["First-Order Solution", "Subsubsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Because in the presence of damping all of the modes that are not directly \
or indirectly excited decay with time, the solution of ",
  StyleBox["eqEps[[1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " is taken to consist of the two modes that might be involved in the \
internal resonance; that is,"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(sol1 = {
        u\_1 -> Function[{x, T\_0, T\_1, T\_2}, 
            Sum[A\_i[T\_1, T\_2]\ \[CapitalPhi]\_i[x]\ 
                    Exp[I\ \[Omega]\_i\ T\_0] + 
                  A\&_\_i[T\_1, T\_2]\ \[CapitalPhi]\_i[x]\ 
                    Exp[\(-I\)\ \[Omega]\_i\ T\_0], {i, 2}] // Evaluate]}; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where the ",
  Cell[BoxData[
      \(\[CapitalPhi]\_i\)]],
  " are the eigenmodes, which satisfy the orthonormality condition ",
  Cell[BoxData[
      \(TraditionalForm
      \`\[Integral]\_0\%1 \[CapitalPhi]\_n[x]\ \[CapitalPhi]\_m[x]\ 
            \[DifferentialD]x = \[Delta]\_\(n, m\)\)]],
  "."
}], "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["Second-Order Solution", "Subsubsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting ",
  StyleBox["sol1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["eqEps[[2,1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and using ",
  StyleBox["intRule1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["intRule2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order2Eq = 
      \(\(\((eqEps[\([1, 1, 1]\)] /. u\_1 -> u\_2)\) == 
                \((eqEps[\([1, 1, 1]\)] /. u\_1 -> u\_2)\) - 
                  eqEps[\([2, 1, 1]\)] /. sol1\) /. intRule1\) //. 
          intRule2 // ExpandAll; order2Eq /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{\(P\_k\), " ", 
          RowBox[{"(", 
            InterpretationBox[
              RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                SubsuperscriptBox["u", "2", 
                  InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                    SequenceForm[ "'", "'"],
                    Editable->False]]}],
              SequenceForm[ "", 
                Power[ 
                  Subscript[ u, 2], 
                  SequenceForm[ "'", "'"]]],
              Editable->False], ")"}]}], "+", 
        InterpretationBox[
          RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
            SubsuperscriptBox["u", "2", 
              
              InterpretationBox[
                \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\[InvisibleSpace]\
"'"\),
                SequenceForm[ "'", "'", "'", "'"],
                Editable->False]]}],
          SequenceForm[ "", 
            Power[ 
              Subscript[ u, 2], 
              SequenceForm[ "'", "'", "'", "'"]]],
          Editable->False], "+", 
        InterpretationBox[
          RowBox[{\(D\_0\%2\), "\[InvisibleSpace]", 
            SubsuperscriptBox["u", "2", 
              InterpretationBox["\[Null]",
                SequenceForm[ ],
                Editable->False]]}],
          SequenceForm[ 
            Power[ 
              Subscript[ D, 0], 2], 
            Power[ 
              Subscript[ u, 2], 
              SequenceForm[ ]]],
          Editable->False], "-", 
        RowBox[{"2", " ", \(b\^2\), " ", "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              RowBox[{
                RowBox[{"(", 
                  InterpretationBox[
                    RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                      SubsuperscriptBox["u", "2", 
                        InterpretationBox["\<\"'\"\>",
                          SequenceForm[ "'"],
                          Editable->False]]}],
                    SequenceForm[ "", 
                      Power[ 
                        Subscript[ u, 2], 
                        SequenceForm[ "'"]]],
                    Editable->False], ")"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], ",", 
              \({x, 0, 1}\)}], "]"}], " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}]}], "==", 
      RowBox[{
        RowBox[{\(-2\), " ", "I", " ", \(E\^\(I\ T\_0\ \[Omega]\_1\)\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_1\[InvisibleSpace]A\_1\),
              SequenceForm[ 
                Subscript[ D, 1], 
                Subscript[ A, 1]],
              Editable->False], ")"}], " ", \(\[Omega]\_1\), " ", 
          \(\[CapitalPhi]\_1[x]\)}], "+", 
        RowBox[{
        "2", " ", "I", " ", \(E\^\(\(-I\)\ T\_0\ \[Omega]\_1\)\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_1\[InvisibleSpace]A\&_\_1\),
              SequenceForm[ 
                Subscript[ D, 1], 
                Subscript[ 
                  OverBar[ A], 1]],
              Editable->False], ")"}], " ", \(\[Omega]\_1\), " ", 
          \(\[CapitalPhi]\_1[x]\)}], "-", 
        RowBox[{"2", " ", "I", " ", \(E\^\(I\ T\_0\ \[Omega]\_2\)\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_1\[InvisibleSpace]A\_2\),
              SequenceForm[ 
                Subscript[ D, 1], 
                Subscript[ A, 2]],
              Editable->False], ")"}], " ", \(\[Omega]\_2\), " ", 
          \(\[CapitalPhi]\_2[x]\)}], "+", 
        RowBox[{
        "2", " ", "I", " ", \(E\^\(\(-I\)\ T\_0\ \[Omega]\_2\)\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_1\[InvisibleSpace]A\&_\_2\),
              SequenceForm[ 
                Subscript[ D, 1], 
                Subscript[ 
                  OverBar[ A], 2]],
              Editable->False], ")"}], " ", \(\[Omega]\_2\), " ", 
          \(\[CapitalPhi]\_2[x]\)}], "+", 
        RowBox[{
        \(1\/2\), " ", "b", " ", \(E\^\(\(-I\)\ T\_0\ \[CapitalOmega]\)\), 
          " ", "f", " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        \(1\/2\), " ", "b", " ", \(E\^\(I\ T\_0\ \[CapitalOmega]\)\), " ", 
          "f", " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "b", " ", \(E\^\(2\ I\ T\_0\ \[Omega]\_1\)\), " ", "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\_1\%2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", "b", " ", 
          \(E\^\(I\ T\_0\ \((\[Omega]\_1 + \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\_1\), " ", \(A\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "b", " ", \(E\^\(2\ I\ T\_0\ \[Omega]\_2\)\), " ", "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\_2\%2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{"2", " ", "b", " ", "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\_1\), " ", \(A\&_\_1\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", "b", " ", 
          \(E\^\(I\ T\_0\ \((\(-\[Omega]\_1\) + \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\_2\), " ", \(A\&_\_1\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "b", " ", \(E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_1\)\), " ", "\[Alpha]", 
          " ", 
          RowBox[{"int", "[", 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\&_\_1\%2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", "b", " ", 
          \(E\^\(I\ T\_0\ \((\[Omega]\_1 - \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\_1\), " ", \(A\&_\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{"2", " ", "b", " ", "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\_2\), " ", \(A\&_\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", "b", " ", 
          \(E\^\(I\ T\_0\ \((\(-\[Omega]\_1\) - \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\&_\_1\), " ", \(A\&_\_2\), 
          " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "b", " ", \(E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_2\)\), " ", "\[Alpha]", 
          " ", 
          RowBox[{"int", "[", 
            RowBox[{
              SuperscriptBox[
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\&_\_2\%2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "k", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", "b", " ", \(E\^\(2\ I\ T\_0\ \[Omega]\_1\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\_1\%2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[CapitalPhi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", "b", " ", 
          \(E\^\(I\ T\_0\ \((\[Omega]\_1 + \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\_1\), " ", \(A\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[CapitalPhi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{"4", " ", "b", " ", "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\_1\), " ", \(A\&_\_1\), " ", 
          RowBox[{
            SubsuperscriptBox["\[CapitalPhi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", "b", " ", 
          \(E\^\(I\ T\_0\ \((\(-\[Omega]\_1\) + \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\_2\), " ", \(A\&_\_1\), " ", 
          RowBox[{
            SubsuperscriptBox["\[CapitalPhi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", "b", " ", \(E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_1\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\&_\_1\%2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[CapitalPhi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", "b", " ", 
          \(E\^\(I\ T\_0\ \((\[Omega]\_1 - \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\_1\), " ", \(A\&_\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[CapitalPhi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", "b", " ", 
          \(E\^\(I\ T\_0\ \((\(-\[Omega]\_1\) - \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\&_\_1\), " ", \(A\&_\_2\), 
          " ", 
          RowBox[{
            SubsuperscriptBox["\[CapitalPhi]", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", "b", " ", 
          \(E\^\(I\ T\_0\ \((\[Omega]\_1 + \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\_1\), " ", \(A\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[CapitalPhi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", "b", " ", \(E\^\(2\ I\ T\_0\ \[Omega]\_2\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\_2\%2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[CapitalPhi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", "b", " ", 
          \(E\^\(I\ T\_0\ \((\(-\[Omega]\_1\) + \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\_2\), " ", \(A\&_\_1\), " ", 
          RowBox[{
            SubsuperscriptBox["\[CapitalPhi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", "b", " ", 
          \(E\^\(I\ T\_0\ \((\[Omega]\_1 - \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\_1\), " ", \(A\&_\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[CapitalPhi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{"4", " ", "b", " ", "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\_2\), " ", \(A\&_\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[CapitalPhi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", "b", " ", 
          \(E\^\(I\ T\_0\ \((\(-\[Omega]\_1\) - \[Omega]\_2)\)\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "1", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\&_\_1\), " ", \(A\&_\_2\), 
          " ", 
          RowBox[{
            SubsuperscriptBox["\[CapitalPhi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}], "+", 
        RowBox[{
        "2", " ", "b", " ", \(E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_2\)\), " ", 
          "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["\[Phi]", "k", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], " ", 
                RowBox[{
                  SubsuperscriptBox["\[CapitalPhi]", "2", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}]}], ",", 
              \({x, 0, 1}\)}], "]"}], " ", \(A\&_\_2\%2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[CapitalPhi]", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "x", "]"}]}]}]}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To collect the terms that might lead to secular terms from the right-hand \
side of ",
  StyleBox["order2Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we define the rules:"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(OmgRule = 
      \(\(Solve[ResonanceConds, 
              Complement[omgList, {#}]~Join~{\[CapitalOmega]}]\)[\([1]\)]\ &
          \)\ /@\ omgList\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({{\[Omega]\_2 \[Rule] \[Epsilon]\^2\ \[Sigma]\_1 + \[Omega]\_1, 
        \[CapitalOmega] \[Rule] 
          2\ \[Epsilon]\^2\ \[Sigma]\_1 + \[Epsilon]\^2\ \[Sigma]\_2 + 
            2\ \[Omega]\_1}, {
        \[Omega]\_1 \[Rule] \(-\[Epsilon]\^2\)\ \[Sigma]\_1 + \[Omega]\_2, 
        \[CapitalOmega] \[Rule] \[Epsilon]\^2\ \[Sigma]\_2 + 2\ \[Omega]\_2}}
      \)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[BoxData[
    \(expRule1[i_] := 
      Exp[arg_] :> 
        Exp[Expand[arg /. OmgRule[\([i]\)]] /. \[Epsilon]\^2\ T\_0 -> T\_2]
          \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Collecting the terms that may lead to secular terms, the terms \
proportional to ",
  Cell[BoxData[
      \(E\^\(I\ \[Omega]\_i\ T\_0\)\)]],
  ", we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(ST1 = 
      \(Coefficient[order2Eq[\([2]\)] /. expRule1[#], 
            E\^\(I\ \[Omega]\_#\ T\_0\)]\ &\)\ /@\ {1, 2}; \n
    ST1 /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{\(-2\), " ", "I", " ", 
            RowBox[{"(", 
              InterpretationBox[\(D\_1\[InvisibleSpace]A\_1\),
                SequenceForm[ 
                  Subscript[ D, 1], 
                  Subscript[ A, 1]],
                Editable->False], ")"}], " ", \(\[Omega]\_1\), " ", 
            \(\[CapitalPhi]\_1[x]\)}], "-", 
          RowBox[{"2", " ", "I", " ", \(E\^\(I\ T\_2\ \[Sigma]\_1\)\), " ", 
            RowBox[{"(", 
              InterpretationBox[\(D\_1\[InvisibleSpace]A\_2\),
                SequenceForm[ 
                  Subscript[ D, 1], 
                  Subscript[ A, 2]],
                Editable->False], ")"}], " ", \(\[Omega]\_2\), " ", 
            \(\[CapitalPhi]\_2[x]\)}]}], ",", 
        RowBox[{
          RowBox[{
          \(-2\), " ", "I", " ", \(E\^\(\(-I\)\ T\_2\ \[Sigma]\_1\)\), " ", 
            RowBox[{"(", 
              InterpretationBox[\(D\_1\[InvisibleSpace]A\_1\),
                SequenceForm[ 
                  Subscript[ D, 1], 
                  Subscript[ A, 1]],
                Editable->False], ")"}], " ", \(\[Omega]\_1\), " ", 
            \(\[CapitalPhi]\_1[x]\)}], "-", 
          RowBox[{"2", " ", "I", " ", 
            RowBox[{"(", 
              InterpretationBox[\(D\_1\[InvisibleSpace]A\_2\),
                SequenceForm[ 
                  Subscript[ D, 1], 
                  Subscript[ A, 2]],
                Editable->False], ")"}], " ", \(\[Omega]\_2\), " ", 
            \(\[CapitalPhi]\_2[x]\)}]}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Because the homogeneous parts of ",
  StyleBox["order2Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  StyleBox[" ",
    FontColor->RGBColor[1, 0, 1]],
  "and corresponding boundary conditions",
  StyleBox[" ",
    FontColor->RGBColor[1, 0, 1]],
  "have a nontrivial solution, the corresponding nonhomogeneous problem has a \
solution only if solvability conditions are satisfied. These conditions \
demand that ",
  StyleBox["ST1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " be orthogonal to every solution of the adjoint homogeneous problem. In \
this case, the problem is self-adjoint and hence solutions of the adjoint \
problem are given by the ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalPhi]\_i[x]\)]],
  ". Demanding that ",
  StyleBox["ST1[[i]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  StyleBox[" ",
    FontColor->RGBColor[1, 0, 1]],
  "be orthogonal to the ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalPhi]\_i[x]\)]],
  ", we obtain the solvability conditions"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(SCond1 = 
      Table[\[Integral]\_0\%1 ST1[\([k]\)]\ \[CapitalPhi]\_k[x] 
                \[DifferentialD]x == 0, {k, 2}] /. modeshapes\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{\(-2\), " ", "I", " ", \(\[Omega]\_1\), " ", 
            RowBox[{
              SubsuperscriptBox["A", "1", 
                TagBox[\((1, 0)\),
                  Derivative],
                MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}]}], "==", 
          "0"}], ",", 
        RowBox[{
          RowBox[{\(-2\), " ", "I", " ", \(\[Omega]\_2\), " ", 
            RowBox[{
              SubsuperscriptBox["A", "2", 
                TagBox[\((1, 0)\),
                  Derivative],
                MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}]}], "==", 
          "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"SCond1Rule", "=", 
      RowBox[{
        RowBox[{"Solve", "[", 
          RowBox[{"SCond1", ",", 
            RowBox[{"{", 
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["A", "1", 
                    TagBox[\((1, 0)\),
                      Derivative],
                    MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}], 
                ",", 
                RowBox[{
                  SubsuperscriptBox["A", "2", 
                    TagBox[\((1, 0)\),
                      Derivative],
                    MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}]}], 
              "}"}]}], "]"}], "[", \([1]\), "]"}]}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            SubsuperscriptBox["A", "1", 
              TagBox[\((1, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}], 
          "\[Rule]", "0"}], ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["A", "2", 
              TagBox[\((1, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}], 
          "\[Rule]", "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["whose complex conjugate is", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(ccSCond1Rule = SCond1Rule /. conjugateRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            SubsuperscriptBox[\(A\&_\), "1", 
              TagBox[\((1, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}], 
          "\[Rule]", "0"}], ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox[\(A\&_\), "2", 
              TagBox[\((1, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}], 
          "\[Rule]", "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Substituting ",
  StyleBox["SCond1Rule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["ccSCond1Rule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["order2Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and using ",
  StyleBox["modeshapes",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order2Eqm = 
      \(\(\(\(order2Eq /. SCond1Rule\) /. ccSCond1Rule\) /. modeshapes\) /. 
          int -> Integrate\) /. Integrate -> int; \n
    order2Eqm /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{\(P\_k\), " ", 
          RowBox[{"(", 
            InterpretationBox[
              RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                SubsuperscriptBox["u", "2", 
                  InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                    SequenceForm[ "'", "'"],
                    Editable->False]]}],
              SequenceForm[ "", 
                Power[ 
                  Subscript[ u, 2], 
                  SequenceForm[ "'", "'"]]],
              Editable->False], ")"}]}], "+", 
        InterpretationBox[
          RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
            SubsuperscriptBox["u", "2", 
              
              InterpretationBox[
                \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\[InvisibleSpace]\
"'"\),
                SequenceForm[ "'", "'", "'", "'"],
                Editable->False]]}],
          SequenceForm[ "", 
            Power[ 
              Subscript[ u, 2], 
              SequenceForm[ "'", "'", "'", "'"]]],
          Editable->False], "+", 
        InterpretationBox[
          RowBox[{\(D\_0\%2\), "\[InvisibleSpace]", 
            SubsuperscriptBox["u", "2", 
              InterpretationBox["\[Null]",
                SequenceForm[ ],
                Editable->False]]}],
          SequenceForm[ 
            Power[ 
              Subscript[ D, 0], 2], 
            Power[ 
              Subscript[ u, 2], 
              SequenceForm[ ]]],
          Editable->False], "+", 
        RowBox[{"2", " ", \(b\^2\), " ", \(\[Pi]\^3\), " ", "\[Alpha]", " ", 
          RowBox[{"int", "[", 
            RowBox[{
              RowBox[{\(Cos[\[Pi]\ x]\), " ", 
                RowBox[{"(", 
                  InterpretationBox[
                    RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                      SubsuperscriptBox["u", "2", 
                        InterpretationBox["\<\"'\"\>",
                          SequenceForm[ "'"],
                          Editable->False]]}],
                    SequenceForm[ "", 
                      Power[ 
                        Subscript[ u, 2], 
                        SequenceForm[ "'"]]],
                    Editable->False], ")"}]}], ",", \({x, 0, 1}\)}], "]"}], 
          " ", \(Sin[\[Pi]\ x]\)}]}], "==", 
      \(\(-\(1\/2\)\)\ b\ E\^\(\(-I\)\ T\_0\ \[CapitalOmega]\)\ f\ \[Pi]\^2\ 
          Sin[\[Pi]\ x] - 
        1\/2\ b\ E\^\(I\ T\_0\ \[CapitalOmega]\)\ f\ \[Pi]\^2\ 
          Sin[\[Pi]\ x] - 
        3\ b\ E\^\(2\ I\ T\_0\ \[Omega]\_1\)\ \[Pi]\^4\ \[Alpha]\ 
          Sin[\[Pi]\ x]\ A\_1\%2 - 
        8\ b\ E\^\(I\ T\_0\ \((\[Omega]\_1 + \[Omega]\_2)\)\)\ \[Pi]\^4\ 
          \[Alpha]\ Sin[2\ \[Pi]\ x]\ A\_1\ A\_2 - 
        4\ b\ E\^\(2\ I\ T\_0\ \[Omega]\_2\)\ \[Pi]\^4\ \[Alpha]\ 
          Sin[\[Pi]\ x]\ A\_2\%2 - 
        6\ b\ \[Pi]\^4\ \[Alpha]\ Sin[\[Pi]\ x]\ A\_1\ A\&_\_1 - 
        8\ b\ E\^\(I\ T\_0\ \((\(-\[Omega]\_1\) + \[Omega]\_2)\)\)\ \[Pi]\^4\ 
          \[Alpha]\ Sin[2\ \[Pi]\ x]\ A\_2\ A\&_\_1 - 
        3\ b\ E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_1\)\ \[Pi]\^4\ \[Alpha]\ 
          Sin[\[Pi]\ x]\ A\&_\_1\%2 - 
        8\ b\ E\^\(I\ T\_0\ \((\[Omega]\_1 - \[Omega]\_2)\)\)\ \[Pi]\^4\ 
          \[Alpha]\ Sin[2\ \[Pi]\ x]\ A\_1\ A\&_\_2 - 
        8\ b\ \[Pi]\^4\ \[Alpha]\ Sin[\[Pi]\ x]\ A\_2\ A\&_\_2 - 
        8\ b\ E\^\(I\ T\_0\ \((\(-\[Omega]\_1\) - \[Omega]\_2)\)\)\ \[Pi]\^4\ 
          \[Alpha]\ Sin[2\ \[Pi]\ x]\ A\&_\_1\ A\&_\_2 - 
        4\ b\ E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_2\)\ \[Pi]\^4\ \[Alpha]\ 
          Sin[\[Pi]\ x]\ A\&_\_2\%2\)}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["The associated boundary conditions are", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order2BC = eqEps[\([2]\)] // Rest\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(u\_2[0, T\_0, T\_1, T\_2] == 0\), ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["u", "2", 
              TagBox[\((2, 0, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(0, T\_0, T\_1, T\_2\), "]"}], 
          "==", "0"}], ",", \(u\_2[1, T\_0, T\_1, T\_2] == 0\), ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["u", "2", 
              TagBox[\((2, 0, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(1, T\_0, T\_1, T\_2\), "]"}], 
          "==", "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "We use the method of undetermined coefficients to determine a particular \
solution of ",
  StyleBox["order2Eqm",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["order2BC",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ". To accomplish this, we first determine the forms of the nonhomogeneous \
terms (i.e., the terms on the right-hand side of ",
  StyleBox["order2Eqm",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ") as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(rhsTerms = 
      \(\(Cases[order2Eqm[\([2]\)], #]\ &\)\ /@
            \ {a_\ f_[b_\ x] :> 
                f[b\ x] /; FreeQ[a, T\_0], _\ E\^\(a_\ T\_0 + b_. \)\ 
                  f_[c_\ x] -> E\^\(a\ T\_0 + b\)\ f[c\ x]} // Flatten\) // 
        Union\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({Sin[\[Pi]\ x], E\^\(\(-I\)\ T\_0\ \[CapitalOmega]\)\ Sin[\[Pi]\ x], 
      E\^\(I\ T\_0\ \[CapitalOmega]\)\ Sin[\[Pi]\ x], 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_1\)\ Sin[\[Pi]\ x], 
      E\^\(2\ I\ T\_0\ \[Omega]\_1\)\ Sin[\[Pi]\ x], 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_2\)\ Sin[\[Pi]\ x], 
      E\^\(2\ I\ T\_0\ \[Omega]\_2\)\ Sin[\[Pi]\ x], 
      E\^\(\(-I\)\ T\_0\ \[Omega]\_1 - I\ T\_0\ \[Omega]\_2\)\ 
        Sin[2\ \[Pi]\ x], 
      E\^\(I\ T\_0\ \[Omega]\_1 - I\ T\_0\ \[Omega]\_2\)\ Sin[2\ \[Pi]\ x], 
      E\^\(\(-I\)\ T\_0\ \[Omega]\_1 + I\ T\_0\ \[Omega]\_2\)\ 
        Sin[2\ \[Pi]\ x], 
      E\^\(I\ T\_0\ \[Omega]\_1 + I\ T\_0\ \[Omega]\_2\)\ Sin[2\ \[Pi]\ x]}
      \)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "which all satisfy the ",
  StyleBox["order2BC",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ". Since we have only even spatial derivatives on the left-hand side of ",
  StyleBox["order2Eqm",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we seek a particular solution as a linear combination of the ",
  StyleBox["rhsTerms ",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  "as follows:"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(symbolList = Table[\[Alpha]\_i, {i, Length[rhsTerms]}]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Alpha]\_1, \[Alpha]\_2, \[Alpha]\_3, \[Alpha]\_4, \[Alpha]\_5, 
      \[Alpha]\_6, \[Alpha]\_7, \[Alpha]\_8, \[Alpha]\_9, \[Alpha]\_10, 
      \[Alpha]\_11}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[BoxData[
    \(\(sol2Form = 
      u\_2 -> Function[{x, T\_0, T\_1, T\_2}, 
          symbolList . rhsTerms // Evaluate]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting ",
  StyleBox["sol2Form",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["order2Eqm",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", equating coefficients of like terms, and solving for the ",
  StyleBox["symbolList",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(symbolRule = 
      \(Solve[Coefficient[
                  \(\(\(Subtract\ @@\ order2Eqm /. sol2Form\) /. intRule1
                        \) //. intRule2\) /. int -> Integrate, rhsTerms] == 
                0 /. Exp[_\ T\_0 + _. ] -> 0 // Thread, symbolList]\)[\([1]\)]
        \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Alpha]\_1 \[Rule] 
        \(-\(\(2\ 
                \((3\ b\ \[Pi]\^2\ \[Alpha]\ A\_1[T\_1, T\_2]\ 
                      A\&_\_1[T\_1, T\_2] + 
                    4\ b\ \[Pi]\^2\ \[Alpha]\ A\_2[T\_1, T\_2]\ 
                      A\&_\_2[T\_1, T\_2])\)\)\/\(\[Pi]\^2 - P\_k + 
                b\^2\ \[Pi]\^2\ \[Alpha]\)\)\), 
      \[Alpha]\_2 \[Rule] 
        \(-\(\(b\ f\ \[Pi]\^2\)\/\(2\ 
                \((\[Pi]\^4 - \[Pi]\^2\ P\_k + b\^2\ \[Pi]\^4\ \[Alpha] - 
                    \[CapitalOmega]\^2)\)\)\)\), 
      \[Alpha]\_3 \[Rule] 
        \(-\(\(b\ f\ \[Pi]\^2\)\/\(2\ 
                \((\[Pi]\^4 - \[Pi]\^2\ P\_k + b\^2\ \[Pi]\^4\ \[Alpha] - 
                    \[CapitalOmega]\^2)\)\)\)\), 
      \[Alpha]\_4 \[Rule] 
        \(-\(\(3\ b\ \[Pi]\^4\ \[Alpha]\ 
                A\&_\_1[T\_1, T\_2]\^2\)\/\(\[Pi]\^4 - \[Pi]\^2\ P\_k + 
                b\^2\ \[Pi]\^4\ \[Alpha] - 4\ \[Omega]\_1\%2\)\)\), 
      \[Alpha]\_5 \[Rule] 
        \(-\(\(3\ b\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_1, T\_2]\^2\)\/\(\[Pi]\^4 - 
                \[Pi]\^2\ P\_k + b\^2\ \[Pi]\^4\ \[Alpha] - 
                4\ \[Omega]\_1\%2\)\)\), 
      \[Alpha]\_6 \[Rule] 
        \(-\(\(4\ b\ \[Pi]\^4\ \[Alpha]\ 
                A\&_\_2[T\_1, T\_2]\^2\)\/\(\[Pi]\^4 - \[Pi]\^2\ P\_k + 
                b\^2\ \[Pi]\^4\ \[Alpha] - 4\ \[Omega]\_2\%2\)\)\), 
      \[Alpha]\_7 \[Rule] 
        \(-\(\(4\ b\ \[Pi]\^4\ \[Alpha]\ A\_2[T\_1, T\_2]\^2\)\/\(\[Pi]\^4 - 
                \[Pi]\^2\ P\_k + b\^2\ \[Pi]\^4\ \[Alpha] - 
                4\ \[Omega]\_2\%2\)\)\), 
      \[Alpha]\_8 \[Rule] 
        \(-\(\(8\ b\ \[Pi]\^4\ \[Alpha]\ A\&_\_1[T\_1, T\_2]\ 
                A\&_\_2[T\_1, T\_2]\)\/\(16\ \[Pi]\^4 - 4\ \[Pi]\^2\ P\_k - 
                \[Omega]\_1\%2 - 2\ \[Omega]\_1\ \[Omega]\_2 - 
                \[Omega]\_2\%2\)\)\), 
      \[Alpha]\_9 \[Rule] 
        \(-\(\(8\ b\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_1, T\_2]\ 
                A\&_\_2[T\_1, T\_2]\)\/\(16\ \[Pi]\^4 - 4\ \[Pi]\^2\ P\_k - 
                \[Omega]\_1\%2 + 2\ \[Omega]\_1\ \[Omega]\_2 - 
                \[Omega]\_2\%2\)\)\), 
      \[Alpha]\_10 \[Rule] 
        \(-\(\(8\ b\ \[Pi]\^4\ \[Alpha]\ A\_2[T\_1, T\_2]\ 
                A\&_\_1[T\_1, T\_2]\)\/\(16\ \[Pi]\^4 - 4\ \[Pi]\^2\ P\_k - 
                \[Omega]\_1\%2 + 2\ \[Omega]\_1\ \[Omega]\_2 - 
                \[Omega]\_2\%2\)\)\), 
      \[Alpha]\_11 \[Rule] 
        \(-\(\(8\ b\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_1, T\_2]\ 
                A\_2[T\_1, T\_2]\)\/\(16\ \[Pi]\^4 - 4\ \[Pi]\^2\ P\_k - 
                \[Omega]\_1\%2 - 2\ \[Omega]\_1\ \[Omega]\_2 - 
                \[Omega]\_2\%2\)\)\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Substituting for the parameter values in ",
  StyleBox["symbolRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(symbolRuleN = 
      \(\(\(symbolRule /. \[CapitalOmega] -> 2  \[Omega]\_2\) /. frequencies
            \) /. bucklingLoad\) /. values1\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Alpha]\_1 \[Rule] 
        \(-\(\(6\ \@3\ \[Pi]\^2\ \@\[Alpha]\ A\_1[T\_1, T\_2]\ 
                  A\&_\_1[T\_1, T\_2] + 
                8\ \@3\ \[Pi]\^2\ \@\[Alpha]\ A\_2[T\_1, T\_2]\ 
                  A\&_\_2[T\_1, T\_2]\)\/\(6\ \[Pi]\^2\)\)\), 
      \[Alpha]\_2 \[Rule] f\/\(12\ \@3\ \[Pi]\^2\ \@\[Alpha]\), 
      \[Alpha]\_3 \[Rule] f\/\(12\ \@3\ \[Pi]\^2\ \@\[Alpha]\), 
      \[Alpha]\_4 \[Rule] \(\@\[Alpha]\ A\&_\_1[T\_1, T\_2]\^2\)\/\(2\ \@3\), 
      \[Alpha]\_5 \[Rule] \(\@\[Alpha]\ A\_1[T\_1, T\_2]\^2\)\/\(2\ \@3\), 
      \[Alpha]\_6 \[Rule] 
        \(2\ \@\[Alpha]\ A\&_\_2[T\_1, T\_2]\^2\)\/\(3\ \@3\), 
      \[Alpha]\_7 \[Rule] \(2\ \@\[Alpha]\ A\_2[T\_1, T\_2]\^2\)\/\(3\ \@3\), 
      \[Alpha]\_8 \[Rule] 
        \(4\ \@\[Alpha]\ A\&_\_1[T\_1, T\_2]\ A\&_\_2[T\_1, T\_2]\)\/\(3\ 
            \@3\), \[Alpha]\_9 \[Rule] 
        \(-\(\(4\ \@\[Alpha]\ A\_1[T\_1, T\_2]\ A\&_\_2[T\_1, T\_2]\)\/\@3
                \)\), \[Alpha]\_10 \[Rule] 
        \(-\(\(4\ \@\[Alpha]\ A\_2[T\_1, T\_2]\ A\&_\_1[T\_1, T\_2]\)\/\@3
                \)\), \[Alpha]\_11 \[Rule] 
        \(4\ \@\[Alpha]\ A\_1[T\_1, T\_2]\ A\_2[T\_1, T\_2]\)\/\(3\ \@3\)}
      \)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Substituting ",
  StyleBox["symbolRuleN",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["sol2Form",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol2 = sol2Form /. symbolRuleN\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(u\_2 \[Rule] 
      Function[{x, T\_0, T\_1, T\_2}, 
        \(Sin[\[Pi]\ x]\ 
              \((\(-\((
                    6\ \@3\ \[Pi]\^2\ \@\[Alpha]\ A\_1[T\_1, T\_2]\ 
                        A\&_\_1[T\_1, T\_2] + 
                      8\ \@3\ \[Pi]\^2\ \@\[Alpha]\ A\_2[T\_1, T\_2]\ 
                        A\&_\_2[T\_1, T\_2])\)\))\)\)\/\(6\ \[Pi]\^2\) + 
          \(E\^\(\(-I\)\ T\_0\ \[CapitalOmega]\)\ Sin[\[Pi]\ x]\ f\)\/\(12\ 
              \@3\ \[Pi]\^2\ \@\[Alpha]\) + 
          \(E\^\(I\ T\_0\ \[CapitalOmega]\)\ Sin[\[Pi]\ x]\ f\)\/\(12\ \@3\ 
              \[Pi]\^2\ \@\[Alpha]\) + 
          \(E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_1\)\ Sin[\[Pi]\ x]\ 
              \((\@\[Alpha]\ A\&_\_1[T\_1, T\_2]\^2)\)\)\/\(2\ \@3\) + 
          \(E\^\(2\ I\ T\_0\ \[Omega]\_1\)\ Sin[\[Pi]\ x]\ 
              \((\@\[Alpha]\ A\_1[T\_1, T\_2]\^2)\)\)\/\(2\ \@3\) + 
          \(E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_2\)\ Sin[\[Pi]\ x]\ 
              \((2\ \@\[Alpha]\ A\&_\_2[T\_1, T\_2]\^2)\)\)\/\(3\ \@3\) + 
          \(E\^\(2\ I\ T\_0\ \[Omega]\_2\)\ Sin[\[Pi]\ x]\ 
              \((2\ \@\[Alpha]\ A\_2[T\_1, T\_2]\^2)\)\)\/\(3\ \@3\) + 
          \(E\^\(\(-I\)\ T\_0\ \[Omega]\_1 - I\ T\_0\ \[Omega]\_2\)\ 
              Sin[2\ \[Pi]\ x]\ 
              \((4\ \@\[Alpha]\ A\&_\_1[T\_1, T\_2]\ A\&_\_2[T\_1, T\_2])
                \)\)\/\(3\ \@3\) + 
          \(E\^\(I\ T\_0\ \[Omega]\_1 - I\ T\_0\ \[Omega]\_2\)\ 
              Sin[2\ \[Pi]\ x]\ 
              \((\(-4\)\ \@\[Alpha]\ A\_1[T\_1, T\_2]\ A\&_\_2[T\_1, T\_2])
                \)\)\/\@3 + 
          \(E\^\(\(-I\)\ T\_0\ \[Omega]\_1 + I\ T\_0\ \[Omega]\_2\)\ 
              Sin[2\ \[Pi]\ x]\ 
              \((\(-4\)\ \@\[Alpha]\ A\_2[T\_1, T\_2]\ A\&_\_1[T\_1, T\_2])
                \)\)\/\@3 + 
          \(E\^\(I\ T\_0\ \[Omega]\_1 + I\ T\_0\ \[Omega]\_2\)\ 
              Sin[2\ \[Pi]\ x]\ 
              \((4\ \@\[Alpha]\ A\_1[T\_1, T\_2]\ A\_2[T\_1, T\_2])\)\)\/\(3\ 
              \@3\)]\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Solvability Conditions", "Subsubsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting ",
  StyleBox["sol1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["sol2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["eqEps[[3,1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and using ",
  StyleBox["intRule1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["intRule2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(order3Eq = 
        \((eqEps[\([1, 1, 1]\)] /. u\_1 -> u\_3)\) == 
          \((\(\(\(\(\(\((eqEps[\([1, 1, 1]\)] /. u\_1 -> u\_3)\) - 
                            eqEps[\([3, 1, 1]\)] /. sol1\) /. sol2\) /. 
                      intRule1\) //. intRule2\) /. modeshapes\) /. 
                int -> Integrate // Expand)\); \) // Timing\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({10.4749999999994547`\ Second, Null}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Collecting the terms that may lead to secular terms, the terms \
proportional to ",
  Cell[BoxData[
      \(E\^\(I\ \[Omega]\_i\ T\_0\)\)]],
  ", we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(ST2 = 
      \(Coefficient[order3Eq[\([2]\)] /. expRule1[#], 
            E\^\(I\ \[Omega]\_#\ T\_0\)]\ &\)\ /@\ {1, 2}; \n
    ST2 /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{\(-\@2\), " ", 
            RowBox[{"(", 
              InterpretationBox[\(D\_1\%2\[InvisibleSpace]A\_1\),
                SequenceForm[ 
                  Power[ 
                    Subscript[ D, 1], 2], 
                  Subscript[ A, 1]],
                Editable->False], ")"}], " ", \(Sin[\[Pi]\ x]\)}], "-", 
          RowBox[{\(\@2\), " ", \(E\^\(I\ T\_2\ \[Sigma]\_1\)\), " ", 
            RowBox[{"(", 
              InterpretationBox[\(D\_1\%2\[InvisibleSpace]A\_2\),
                SequenceForm[ 
                  Power[ 
                    Subscript[ D, 1], 2], 
                  Subscript[ A, 2]],
                Editable->False], ")"}], " ", \(Sin[2\ \[Pi]\ x]\)}], "-", 
          RowBox[{"2", " ", "I", " ", \(\@2\), " ", 
            RowBox[{"(", 
              InterpretationBox[\(D\_2\[InvisibleSpace]A\_1\),
                SequenceForm[ 
                  Subscript[ D, 2], 
                  Subscript[ A, 1]],
                Editable->False], ")"}], " ", \(Sin[\[Pi]\ x]\), " ", 
            \(\[Omega]\_1\)}], "-", 
          \(2\ I\ \@2\ \[Mu]\ Sin[\[Pi]\ x]\ A\_1\ \[Omega]\_1\), "-", 
          RowBox[{
          "2", " ", "I", " ", \(\@2\), " ", \(E\^\(I\ T\_2\ \[Sigma]\_1\)\), 
            " ", 
            RowBox[{"(", 
              InterpretationBox[\(D\_2\[InvisibleSpace]A\_2\),
                SequenceForm[ 
                  Subscript[ D, 2], 
                  Subscript[ A, 2]],
                Editable->False], ")"}], " ", \(Sin[2\ \[Pi]\ x]\), " ", 
            \(\[Omega]\_2\)}], "-", 
          \(2\ I\ \@2\ E\^\(I\ T\_2\ \[Sigma]\_1\)\ \[Mu]\ Sin[2\ \[Pi]\ x]\ 
            A\_2\ \[Omega]\_2\), "-", 
          \(\(E\^\(2\ I\ T\_2\ \[Sigma]\_1 + I\ T\_2\ \[Sigma]\_2\)\ f\ 
                \[Pi]\^2\ Sin[\[Pi]\ x]\ A\&_\_1\)\/\@2\), "-", 
          \(\(b\ E\^\(2\ I\ T\_2\ \[Sigma]\_1 + I\ T\_2\ \[Sigma]\_2\)\ f\ 
                \[Pi]\^2\ \@\[Alpha]\ Sin[\[Pi]\ x]\ A\&_\_1\)\/\(2\ \@6\)\), 
          "-", 
          \(3\ \@2\ \[Pi]\^4\ \[Alpha]\ Sin[\[Pi]\ x]\ A\_1\%2\ A\&_\_1\), 
          "+", \(5\ \@\(3\/2\)\ b\ \[Pi]\^4\ \[Alpha]\^\(3/2\)\ 
            Sin[\[Pi]\ x]\ A\_1\%2\ A\&_\_1\), "-", 
          \(8\ \@2\ E\^\(I\ T\_2\ \[Sigma]\_1\)\ \[Pi]\^4\ \[Alpha]\ 
            Sin[2\ \[Pi]\ x]\ A\_1\ A\_2\ A\&_\_1\), "+", 
          \(68\/3\ \@\(2\/3\)\ b\ E\^\(I\ T\_2\ \[Sigma]\_1\)\ \[Pi]\^4\ 
            \[Alpha]\^\(3/2\)\ Sin[2\ \[Pi]\ x]\ A\_1\ A\_2\ A\&_\_1\), "-", 
          \(4\ \@2\ E\^\(2\ I\ T\_2\ \[Sigma]\_1\)\ \[Pi]\^4\ \[Alpha]\ 
            Sin[\[Pi]\ x]\ A\_2\%2\ A\&_\_1\), "+", 
          \(14\ \@\(2\/3\)\ b\ E\^\(2\ I\ T\_2\ \[Sigma]\_1\)\ \[Pi]\^4\ 
            \[Alpha]\^\(3/2\)\ Sin[\[Pi]\ x]\ A\_2\%2\ A\&_\_1\), "-", 
          \(2\ \@2\ E\^\(I\ T\_2\ \[Sigma]\_1 + I\ T\_2\ \[Sigma]\_2\)\ f\ 
            \[Pi]\^2\ Sin[2\ \[Pi]\ x]\ A\&_\_2\), "-", 
          \(1\/3\ \@\(2\/3\)\ b\ 
            E\^\(I\ T\_2\ \[Sigma]\_1 + I\ T\_2\ \[Sigma]\_2\)\ f\ \[Pi]\^2\ 
            \@\[Alpha]\ Sin[2\ \[Pi]\ x]\ A\&_\_2\), "-", 
          \(4\ \@2\ E\^\(\(-I\)\ T\_2\ \[Sigma]\_1\)\ \[Pi]\^4\ \[Alpha]\ 
            Sin[2\ \[Pi]\ x]\ A\_1\%2\ A\&_\_2\), "+", 
          \(14\ \@\(2\/3\)\ b\ E\^\(\(-I\)\ T\_2\ \[Sigma]\_1\)\ \[Pi]\^4\ 
            \[Alpha]\^\(3/2\)\ Sin[2\ \[Pi]\ x]\ A\_1\%2\ A\&_\_2\), "-", 
          \(8\ \@2\ \[Pi]\^4\ \[Alpha]\ Sin[\[Pi]\ x]\ A\_1\ A\_2\ A\&_\_2\), 
          "+", \(68\/3\ \@\(2\/3\)\ b\ \[Pi]\^4\ \[Alpha]\^\(3/2\)\ 
            Sin[\[Pi]\ x]\ A\_1\ A\_2\ A\&_\_2\), "-", 
          \(48\ \@2\ E\^\(I\ T\_2\ \[Sigma]\_1\)\ \[Pi]\^4\ \[Alpha]\ 
            Sin[2\ \[Pi]\ x]\ A\_2\%2\ A\&_\_2\), "+", 
          \(40\/3\ \@\(2\/3\)\ b\ E\^\(I\ T\_2\ \[Sigma]\_1\)\ \[Pi]\^4\ 
            \[Alpha]\^\(3/2\)\ Sin[2\ \[Pi]\ x]\ A\_2\%2\ A\&_\_2\)}], ",", 
        RowBox[{
          RowBox[{\(-\@2\), " ", \(E\^\(\(-I\)\ T\_2\ \[Sigma]\_1\)\), " ", 
            RowBox[{"(", 
              InterpretationBox[\(D\_1\%2\[InvisibleSpace]A\_1\),
                SequenceForm[ 
                  Power[ 
                    Subscript[ D, 1], 2], 
                  Subscript[ A, 1]],
                Editable->False], ")"}], " ", \(Sin[\[Pi]\ x]\)}], "-", 
          RowBox[{\(\@2\), " ", 
            RowBox[{"(", 
              InterpretationBox[\(D\_1\%2\[InvisibleSpace]A\_2\),
                SequenceForm[ 
                  Power[ 
                    Subscript[ D, 1], 2], 
                  Subscript[ A, 2]],
                Editable->False], ")"}], " ", \(Sin[2\ \[Pi]\ x]\)}], "-", 
          RowBox[{
          "2", " ", "I", " ", \(\@2\), " ", 
            \(E\^\(\(-I\)\ T\_2\ \[Sigma]\_1\)\), " ", 
            RowBox[{"(", 
              InterpretationBox[\(D\_2\[InvisibleSpace]A\_1\),
                SequenceForm[ 
                  Subscript[ D, 2], 
                  Subscript[ A, 1]],
                Editable->False], ")"}], " ", \(Sin[\[Pi]\ x]\), " ", 
            \(\[Omega]\_1\)}], "-", 
          \(2\ I\ \@2\ E\^\(\(-I\)\ T\_2\ \[Sigma]\_1\)\ \[Mu]\ 
            Sin[\[Pi]\ x]\ A\_1\ \[Omega]\_1\), "-", 
          RowBox[{"2", " ", "I", " ", \(\@2\), " ", 
            RowBox[{"(", 
              InterpretationBox[\(D\_2\[InvisibleSpace]A\_2\),
                SequenceForm[ 
                  Subscript[ D, 2], 
                  Subscript[ A, 2]],
                Editable->False], ")"}], " ", \(Sin[2\ \[Pi]\ x]\), " ", 
            \(\[Omega]\_2\)}], "-", 
          \(2\ I\ \@2\ \[Mu]\ Sin[2\ \[Pi]\ x]\ A\_2\ \[Omega]\_2\), "-", 
          \(\(E\^\(I\ T\_2\ \[Sigma]\_1 + I\ T\_2\ \[Sigma]\_2\)\ f\ 
                \[Pi]\^2\ Sin[\[Pi]\ x]\ A\&_\_1\)\/\@2\), "-", 
          \(\(b\ E\^\(I\ T\_2\ \[Sigma]\_1 + I\ T\_2\ \[Sigma]\_2\)\ f\ 
                \[Pi]\^2\ \@\[Alpha]\ Sin[\[Pi]\ x]\ A\&_\_1\)\/\(2\ \@6\)\), 
          "-", \(3\ \@2\ E\^\(\(-I\)\ T\_2\ \[Sigma]\_1\)\ \[Pi]\^4\ 
            \[Alpha]\ Sin[\[Pi]\ x]\ A\_1\%2\ A\&_\_1\), "+", 
          \(5\ \@\(3\/2\)\ b\ E\^\(\(-I\)\ T\_2\ \[Sigma]\_1\)\ \[Pi]\^4\ 
            \[Alpha]\^\(3/2\)\ Sin[\[Pi]\ x]\ A\_1\%2\ A\&_\_1\), "-", 
          \(8\ \@2\ \[Pi]\^4\ \[Alpha]\ Sin[2\ \[Pi]\ x]\ A\_1\ A\_2\ 
            A\&_\_1\), "+", 
          \(68\/3\ \@\(2\/3\)\ b\ \[Pi]\^4\ \[Alpha]\^\(3/2\)\ 
            Sin[2\ \[Pi]\ x]\ A\_1\ A\_2\ A\&_\_1\), "-", 
          \(4\ \@2\ E\^\(I\ T\_2\ \[Sigma]\_1\)\ \[Pi]\^4\ \[Alpha]\ 
            Sin[\[Pi]\ x]\ A\_2\%2\ A\&_\_1\), "+", 
          \(14\ \@\(2\/3\)\ b\ E\^\(I\ T\_2\ \[Sigma]\_1\)\ \[Pi]\^4\ 
            \[Alpha]\^\(3/2\)\ Sin[\[Pi]\ x]\ A\_2\%2\ A\&_\_1\), "-", 
          \(2\ \@2\ E\^\(I\ T\_2\ \[Sigma]\_2\)\ f\ \[Pi]\^2\ 
            Sin[2\ \[Pi]\ x]\ A\&_\_2\), "-", 
          \(1\/3\ \@\(2\/3\)\ b\ E\^\(I\ T\_2\ \[Sigma]\_2\)\ f\ \[Pi]\^2\ 
            \@\[Alpha]\ Sin[2\ \[Pi]\ x]\ A\&_\_2\), "-", 
          \(4\ \@2\ E\^\(\(-2\)\ I\ T\_2\ \[Sigma]\_1\)\ \[Pi]\^4\ \[Alpha]\ 
            Sin[2\ \[Pi]\ x]\ A\_1\%2\ A\&_\_2\), "+", 
          \(14\ \@\(2\/3\)\ b\ E\^\(\(-2\)\ I\ T\_2\ \[Sigma]\_1\)\ \[Pi]\^4\ 
            \[Alpha]\^\(3/2\)\ Sin[2\ \[Pi]\ x]\ A\_1\%2\ A\&_\_2\), "-", 
          \(8\ \@2\ E\^\(\(-I\)\ T\_2\ \[Sigma]\_1\)\ \[Pi]\^4\ \[Alpha]\ 
            Sin[\[Pi]\ x]\ A\_1\ A\_2\ A\&_\_2\), "+", 
          \(68\/3\ \@\(2\/3\)\ b\ E\^\(\(-I\)\ T\_2\ \[Sigma]\_1\)\ \[Pi]\^4\ 
            \[Alpha]\^\(3/2\)\ Sin[\[Pi]\ x]\ A\_1\ A\_2\ A\&_\_2\), "-", 
          \(48\ \@2\ \[Pi]\^4\ \[Alpha]\ Sin[2\ \[Pi]\ x]\ A\_2\%2\ 
            A\&_\_2\), "+", 
          \(40\/3\ \@\(2\/3\)\ b\ \[Pi]\^4\ \[Alpha]\^\(3/2\)\ 
            Sin[2\ \[Pi]\ x]\ A\_2\%2\ A\&_\_2\)}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Demanding that ",
  StyleBox["ST2[[i]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  StyleBox[" ",
    FontColor->RGBColor[1, 0, 1]],
  "be orthogonal to the ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalPhi]\_i[x]\)]],
  ", we obtain the solvability conditions"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(SCond2 = 
      \(\(\(\(\(Table[
                    int[ST2[\([k]\)]\ \[CapitalPhi]\_k[x], {x, 0, 1}] == 0, {
                      k, 2}] /. intRule1\) //. intRule2\) /. modeshapes\) /. 
            values1\) /. int -> Integrate\) /. D[SCond1Rule, T\_1]; \n
    SCond2 /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            RowBox[{\(-2\), " ", "I", " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_2\[InvisibleSpace]A\_1\),
                  SequenceForm[ 
                    Subscript[ D, 2], 
                    Subscript[ A, 1]],
                  Editable->False], ")"}], " ", \(\[Omega]\_1\)}], "-", 
            \(2\ I\ \[Mu]\ A\_1\ \[Omega]\_1\), "-", 
            \(E\^\(2\ I\ T\_2\ \[Sigma]\_1 + I\ T\_2\ \[Sigma]\_2\)\ f\ 
              \[Pi]\^2\ A\&_\_1\), "+", 
            \(12\ \[Pi]\^4\ \[Alpha]\ A\_1\%2\ A\&_\_1\), "+", 
            \(24\ E\^\(2\ I\ T\_2\ \[Sigma]\_1\)\ \[Pi]\^4\ \[Alpha]\ 
              A\_2\%2\ A\&_\_1\), "+", 
            \(112\/3\ \[Pi]\^4\ \[Alpha]\ A\_1\ A\_2\ A\&_\_2\)}], "==", 
          "0"}], ",", 
        RowBox[{
          RowBox[{
            RowBox[{\(-2\), " ", "I", " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_2\[InvisibleSpace]A\_2\),
                  SequenceForm[ 
                    Subscript[ D, 2], 
                    Subscript[ A, 2]],
                  Editable->False], ")"}], " ", \(\[Omega]\_2\)}], "-", 
            \(2\ I\ \[Mu]\ A\_2\ \[Omega]\_2\), "+", 
            \(112\/3\ \[Pi]\^4\ \[Alpha]\ A\_1\ A\_2\ A\&_\_1\), "-", 
            \(8\/3\ E\^\(I\ T\_2\ \[Sigma]\_2\)\ f\ \[Pi]\^2\ A\&_\_2\), "+", 
            \(24\ E\^\(\(-2\)\ I\ T\_2\ \[Sigma]\_1\)\ \[Pi]\^4\ \[Alpha]\ 
              A\_1\%2\ A\&_\_2\), "-", 
            \(64\/3\ \[Pi]\^4\ \[Alpha]\ A\_2\%2\ A\&_\_2\)}], "==", "0"}]}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  "8.1.3 The Function ",
  StyleBox["MMSDirect11",
    FontColor->RGBColor[1, 0, 1]]
}], "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Collecting the commands in the preceding section, we build a function ",
  StyleBox["MMSDirect11",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", specifically for the nonlinear transverse vibrations of a hinged-hinged \
beam around its first buckled mode (",
  StyleBox["EOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["BC1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  "), to determine the solvability conditions for different combinations of ",
  
  StyleBox["frequencies",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", ",
  StyleBox["modeshapes",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", and ",
  StyleBox["ResonanceConds",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in which the first mode might be involved in either a one-to-one or a \
three-to-one internal resonance with one of the other modes."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
    \(MMSDirect11[frequencies_, modeshapes_, ResonanceConds_]\), ":=", "\n", 
      "\t", 
      RowBox[{"Module", "[", 
        RowBox[{\({}\), ",", "\n", "\t\t", 
          RowBox[{
          \(multiScales = {u[x_, t] -> u[x, T\_0, T\_1, T\_2], 
                \(\(Derivative[m_, n_]\)[u_]\)[x_, t] -> 
                  \(dt[n]\)[D[u[x, T\_0, T\_1, T\_2], {x, m}]], t -> T\_0}\), 
            ";", "\n", "     ", 
            \(solRule = 
              u -> \((Evaluate[
                      Sum[\[Epsilon]\^j\ u\_j[#1, #2, #3, #4], {j, 3}]]\ &)
                  \)\), ";", "\n", "\t\t", 
            \(scaling = {c -> 2  \[Epsilon]\^2\ \[Mu]\ , 
                f -> \[Epsilon]\^2\ f}\), ";", "\n", "     ", 
            \(eqs = \((
                  \(\(\(\(\(Join[{EOM}, BC1] /. multiScales\) /. solRule\) /. 
                            scaling\) /. intRule1\) //. intRule2 // TrigToExp
                      \) // ExpandAll)\) /. 
                \[Epsilon]\^\(n_ /; n > 3\) -> 0\), ";", "\n", "     ", 
            \(eqEps = 
              \(\(Thread[CoefficientList[Subtract\ @@\ #, \[Epsilon]] == 0]\ &
                      \)\ /@\ eqs // Transpose\) // Rest\), ";", "\n", 
            "     ", 
            \(sol1 = {
                u\_1 -> 
                  Function[{x, T\_0, T\_1, T\_2}, 
                    Sum[A\_i[T\_1, T\_2]\ \[CapitalPhi]\_i[x]\ 
                            Exp[I\ \[Omega]\_i\ T\_0] + 
                          A\&_\_i[T\_1, T\_2]\ \[CapitalPhi]\_i[x]\ 
                            Exp[\(-I\)\ \[Omega]\_i\ T\_0], {i, 2}] // 
                      Evaluate]}\), ";", "\n", "     ", 
            \(order2Eq = 
              \(\(\((eqEps[\([1, 1, 1]\)] /. u\_1 -> u\_2)\) == 
                        \((eqEps[\([1, 1, 1]\)] /. u\_1 -> u\_2)\) - 
                          eqEps[\([2, 1, 1]\)] /. sol1\) /. intRule1\) //. 
                  intRule2 // ExpandAll\), ";", "\n", "\t\t", 
            \(omgList = {\[Omega]\_1, \[Omega]\_2}\), ";", "\n", "     ", 
            \(OmgRule = 
              \(\(Solve[ResonanceConds, 
                      Complement[omgList, {#}]~Join~{\[CapitalOmega]}]\)[
                    \([1]\)]\ &\)\ /@\ omgList\), ";", "\n", "     ", 
            \(expRule1[i_] := 
              Exp[arg_] :> 
                Exp[Expand[arg /. OmgRule[\([i]\)]] /. 
                    \[Epsilon]\^2\ T\_0 -> T\_2]\), ";", "\n", "     ", 
            \(ST1 = \(Coefficient[order2Eq[\([2]\)] /. expRule1[#], 
                    E\^\(I\ \[Omega]\_#\ T\_0\)]\ &\)\ /@\ {1, 2}\), ";", 
            "\n", "     ", 
            \(SCond1 = 
              Table[\[Integral]\_0\%1 ST1[\([k]\)]\ \[CapitalPhi]\_k[x] 
                        \[DifferentialD]x == 0, {k, 2}] /. modeshapes\), ";", 
            "\n", "     ", 
            RowBox[{"SCond1Rule", "=", 
              RowBox[{
                RowBox[{"Solve", "[", 
                  RowBox[{"SCond1", ",", 
                    RowBox[{"{", 
                      RowBox[{
                        RowBox[{
                          SubsuperscriptBox["A", "1", 
                            TagBox[\((1, 0)\),
                              Derivative],
                            MultilineFunction->None], "[", \(T\_1, T\_2\), 
                          "]"}], ",", 
                        RowBox[{
                          SubsuperscriptBox["A", "2", 
                            TagBox[\((1, 0)\),
                              Derivative],
                            MultilineFunction->None], "[", \(T\_1, T\_2\), 
                          "]"}]}], "}"}]}], "]"}], "[", \([1]\), "]"}]}], 
            ";", "\n", "     ", 
            \(ccSCond1Rule = SCond1Rule /. conjugateRule\), ";", "\n", 
            "     ", 
            \(order2Eqm = 
              \(\(\(\(order2Eq /. SCond1Rule\) /. ccSCond1Rule\) /. modeshapes
                    \) /. int -> Integrate\) /. Integrate -> int\), ";", 
            "\n", "     ", 
            \(rhsTerms = 
              \(\(Cases[order2Eqm[\([2]\)], #]\ &\)\ /@
                    \ {a_\ f_[b_\ x] :> 
                        f[b\ x] /; FreeQ[a, T\_0], _\ E\^\(a_\ T\_0 + b_. \)\ 
                          f_[c_\ x] -> E\^\(a\ T\_0 + b\)\ f[c\ x]} // Flatten
                  \) // Union\), ";", "\n", "     ", 
            \(symbolList = Table[\[Alpha]\_i, {i, Length[rhsTerms]}]\), ";", 
            "\n", "     ", 
            \(sol2Form = 
              u\_2 -> Function[{x, T\_0, T\_1, T\_2}, 
                  symbolList . rhsTerms // Evaluate]\), ";", "\n", "     ", 
            \(symbolRule = 
              \(Solve[Coefficient[
                          \(\(\(Subtract\ @@\ order2Eqm /. sol2Form\) /. 
                                intRule1\) //. intRule2\) /. 
                            int -> Integrate, rhsTerms] == 0 /. 
                      Exp[_\ T\_0 + _. ] -> 0 // Thread, symbolList]\)[
                \([1]\)]\), ";", "\n", " \t ", 
            \(values1 = 
              Solve[\(Select[ResonanceConds, FreeQ[#, \[CapitalOmega]]\ &] /. 
                      \[Epsilon] -> 0\) /. frequencies, b] // Flatten\), ";", 
            "\n", "     ", 
            \(symbolRuleN = 
              \(\(\(symbolRule /. \[CapitalOmega] -> 2  \[Omega]\_2\) /. 
                    frequencies\) /. bucklingLoad\) /. values1\), ";", "\n", 
            "     ", \(sol2 = sol2Form /. symbolRuleN\), ";", "\n", "     ", 
            \(order3Eq = 
              \((eqEps[\([1, 1, 1]\)] /. u\_1 -> u\_3)\) == 
                \((\(\(\(\(\(\((eqEps[\([1, 1, 1]\)] /. u\_1 -> u\_3)\) - 
                                  eqEps[\([3, 1, 1]\)] /. sol1\) /. sol2\) /. 
                            intRule1\) //. intRule2\) /. modeshapes\) /. 
                      int -> Integrate // Expand)\)\), ";", "\n", "     ", 
            \(ST2 = \(Coefficient[order3Eq[\([2]\)] /. expRule1[#], 
                    E\^\(I\ \[Omega]\_#\ T\_0\)]\ &\)\ /@\ {1, 2}\), ";", 
            "\n", "     ", 
            \(SCond2 = 
              \(\(\(\(\(Table[
                            int[ST2[\([k]\)]\ \[CapitalPhi]\_k[x], {x, 0, 
                                  1}] == 0, {k, 2}] /. intRule1\) //. intRule2
                        \) /. modeshapes\) /. values1\) /. 
                  int -> Integrate\) /. D[SCond1Rule, T\_1]\), ";", "\n", 
            " \t ", \(Return[
              \(SCond2 /. A\_i_ -> \((A\_i[#2]\ &)\)\) /. 
                f_[T\_1, T\_2] -> f[T\_2]]\)}]}], "\n", " ", "]"}]}]], "Input",\

  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
8.1.4 Three-to-One Internal Resonances Between the First Two Modes\
\>", "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "In this section, we consider the case ",
  Cell[BoxData[
      \(TraditionalForm\`\[Omega]\_2 \[TildeTilde] 3  \[Omega]\_1\)]],
  " and ",
  Cell[BoxData[
      \(\[CapitalOmega] \[TildeTilde] 2  \[Omega]\_2\)]],
  ". To describe the nearness of the resonances, we introduce the detuning \
parameters ",
  Cell[BoxData[
      \(TraditionalForm\`\[Sigma]\_i\)]],
  " defined by"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ResonanceConds1 = {
        \[Omega]\_2 == 3  \[Omega]\_1 + \[Epsilon]\^2\ \[Sigma]\_1, 
        \[CapitalOmega] == 2  \[Omega]\_2 + \[Epsilon]\^2\ \[Sigma]\_2}; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Using ",
  StyleBox["MMSDirect11",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain the solvability conditions"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(MMSDirect11[frequencies, modeshapes, ResonanceConds1] // Timing\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(21.5010000000002055`\ Second\), ",", 
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
              \(\(-2\)\ I\ \[Mu]\ \[Omega]\_1\ A\_1[T\_2]\), "+", 
                \(12\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\^2\ A\&_\_1[T\_2]\), 
                "+", \(624\/35\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\ A\_2[T\_2]\ 
                  A\&_\_2[T\_2]\), "-", 
                RowBox[{"2", " ", "I", " ", \(\[Omega]\_1\), " ", 
                  RowBox[{
                    SubsuperscriptBox["A", "1", "\[Prime]",
                      MultilineFunction->None], "[", \(T\_2\), "]"}]}]}], 
              "==", "0"}], ",", 
            RowBox[{
              RowBox[{
              \(\(-2\)\ I\ \[Mu]\ \[Omega]\_2\ A\_2[T\_2]\), "+", 
                \(624\/35\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\ A\_2[T\_2]\ 
                  A\&_\_1[T\_2]\), "-", 
                \(72\/35\ E\^\(I\ T\_2\ \[Sigma]\_2\)\ f\ \[Pi]\^2\ 
                  A\&_\_2[T\_2]\), "-", 
                \(576\/35\ \[Pi]\^4\ \[Alpha]\ A\_2[T\_2]\^2\ 
                  A\&_\_2[T\_2]\), "-", 
                RowBox[{"2", " ", "I", " ", \(\[Omega]\_2\), " ", 
                  RowBox[{
                    SubsuperscriptBox["A", "2", "\[Prime]",
                      MultilineFunction->None], "[", \(T\_2\), "]"}]}]}], 
              "==", "0"}]}], "}"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
which indicates that this three-to-one internal resonance is not activated.\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Next, we consider the case in which the frequency of the first mode is \
three times that of the second mode; that is, the buckling load is above the \
first crossover value. To accomplish this, we redefine the frequencies and \
denote the frequency of the first mode by ",
  Cell[BoxData[
      \(\[Omega]\_2\)]],
  " and that of the second mode by ",
  Cell[BoxData[
      \(\[Omega]\_1\)]],
  " as follows:"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(frequencies2 = {\[Omega]\_2 -> \((\[Omega] /. \[Omega]Sol1)\), 
        \[Omega]\_1 -> \((\(\[Omega] /. \[Omega]Sol2\) /. n -> 2)\)}\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Omega]\_2 \[Rule] b\ \[Pi]\^2\ \@\[Alpha], 
      \[Omega]\_1 \[Rule] 2\ \@3\ \[Pi]\^2}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["Moreover, we redefine the modeshapes according to", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(modeshapes2 = 
      Join[modeshape, {\[CapitalPhi]\_1 -> Function[x, shape2 // Evaluate], 
          \[CapitalPhi]\_2 -> Function[x, shape1 // Evaluate]}]; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "To describe the nearness of the resonances, we introduce the two detuning \
parameters ",
  Cell[BoxData[
      \(TraditionalForm\`\[Sigma]\_i\)]],
  " defined by"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ResonanceConds2 = {
        \[Omega]\_2 == 3  \[Omega]\_1 + \[Epsilon]\^2\ \[Sigma]\_1, 
        \[CapitalOmega] == 2  \[Omega]\_2 + \[Epsilon]\^2\ \[Sigma]\_2}; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Using ",
  StyleBox["MMSDirect11",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain the solvability conditions"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(MMSDirect11[frequencies2, modeshapes2, ResonanceConds2] // Timing\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(21.7209999999995506`\ Second\), ",", 
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
              \(\(-2\)\ I\ \[Mu]\ \[Omega]\_1\ A\_1[T\_2]\), "+", 
                \(64\/5\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\^2\ A\&_\_1[T\_2]\), 
                "-", \(496\/5\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\ A\_2[T\_2]\ 
                  A\&_\_2[T\_2]\), "-", 
                RowBox[{"2", " ", "I", " ", \(\[Omega]\_1\), " ", 
                  RowBox[{
                    SubsuperscriptBox["A", "1", "\[Prime]",
                      MultilineFunction->None], "[", \(T\_2\), "]"}]}]}], 
              "==", "0"}], ",", 
            RowBox[{
              RowBox[{
              \(\(-2\)\ I\ \[Mu]\ \[Omega]\_2\ A\_2[T\_2]\), "-", 
                \(496\/5\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\ A\_2[T\_2]\ 
                  A\&_\_1[T\_2]\), "-", 
                \(E\^\(I\ T\_2\ \[Sigma]\_2\)\ f\ \[Pi]\^2\ A\&_\_2[T\_2]\), 
                "+", \(12\ \[Pi]\^4\ \[Alpha]\ A\_2[T\_2]\^2\ 
                  A\&_\_2[T\_2]\), "-", 
                RowBox[{"2", " ", "I", " ", \(\[Omega]\_2\), " ", 
                  RowBox[{
                    SubsuperscriptBox["A", "2", "\[Prime]",
                      MultilineFunction->None], "[", \(T\_2\), "]"}]}]}], 
              "==", "0"}]}], "}"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
Again, this three-to-one internal resonance is not activated.\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
8.1.5 One-to-One and Three-to-One Internal Resonances Between the First and \
Third Modes\
\>", "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
Variations of the first and third natural frequencies with the buckling level \
are plotted as follows:\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.031746 0.0147151 0.00198793 [
[.02381 -0.0125 -3 -9 ]
[.02381 -0.0125 3 0 ]
[.18254 -0.0125 -3 -9 ]
[.18254 -0.0125 3 0 ]
[.34127 -0.0125 -6 -9 ]
[.34127 -0.0125 6 0 ]
[.5 -0.0125 -6 -9 ]
[.5 -0.0125 6 0 ]
[.65873 -0.0125 -6 -9 ]
[.65873 -0.0125 6 0 ]
[.81746 -0.0125 -6 -9 ]
[.81746 -0.0125 6 0 ]
[.97619 -0.0125 -6 -9 ]
[.97619 -0.0125 6 0 ]
[ 0 0 -0.125 0 ]
[.5 -0.0125 -13.0313 -22 ]
[.5 -0.0125 13.0313 -9 ]
[-0.0125 .01472 -6 -4.5 ]
[-0.0125 .01472 0 4.5 ]
[-0.0125 .11411 -12 -4.5 ]
[-0.0125 .11411 0 4.5 ]
[-0.0125 .21351 -18 -4.5 ]
[-0.0125 .21351 0 4.5 ]
[-0.0125 .31291 -18 -4.5 ]
[-0.0125 .31291 0 4.5 ]
[-0.0125 .4123 -18 -4.5 ]
[-0.0125 .4123 0 4.5 ]
[-0.0125 .5117 -18 -4.5 ]
[-0.0125 .5117 0 4.5 ]
[-0.0125 .6111 -18 -4.5 ]
[-0.0125 .6111 0 4.5 ]
[ 0 0 -0.125 0 ]
[-0.01875 .30902 -28 -5.59375 ]
[-0.01875 .30902 -18 5.59375 ]
[ 0 .61803 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
.02381 0 m
.02381 .00625 L
s
[(0)] .02381 -0.0125 0 1 Mshowa
.18254 0 m
.18254 .00625 L
s
[(5)] .18254 -0.0125 0 1 Mshowa
.34127 0 m
.34127 .00625 L
s
[(10)] .34127 -0.0125 0 1 Mshowa
.5 0 m
.5 .00625 L
s
[(15)] .5 -0.0125 0 1 Mshowa
.65873 0 m
.65873 .00625 L
s
[(20)] .65873 -0.0125 0 1 Mshowa
.81746 0 m
.81746 .00625 L
s
[(25)] .81746 -0.0125 0 1 Mshowa
.97619 0 m
.97619 .00625 L
s
[(30)] .97619 -0.0125 0 1 Mshowa
.125 Mabswid
.05556 0 m
.05556 .00375 L
s
.0873 0 m
.0873 .00375 L
s
.11905 0 m
.11905 .00375 L
s
.15079 0 m
.15079 .00375 L
s
.21429 0 m
.21429 .00375 L
s
.24603 0 m
.24603 .00375 L
s
.27778 0 m
.27778 .00375 L
s
.30952 0 m
.30952 .00375 L
s
.37302 0 m
.37302 .00375 L
s
.40476 0 m
.40476 .00375 L
s
.43651 0 m
.43651 .00375 L
s
.46825 0 m
.46825 .00375 L
s
.53175 0 m
.53175 .00375 L
s
.56349 0 m
.56349 .00375 L
s
.59524 0 m
.59524 .00375 L
s
.62698 0 m
.62698 .00375 L
s
.69048 0 m
.69048 .00375 L
s
.72222 0 m
.72222 .00375 L
s
.75397 0 m
.75397 .00375 L
s
.78571 0 m
.78571 .00375 L
s
.84921 0 m
.84921 .00375 L
s
.88095 0 m
.88095 .00375 L
s
.9127 0 m
.9127 .00375 L
s
.94444 0 m
.94444 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
gsave
.5 -0.0125 -74.0313 -26 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 21 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 17.000000 moveto
 461.000000 17.000000 lineto
 461.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 17.000000 lineto
closepath
clip newpath
63.000000 13.750000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 13.750000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(b) show
0.000000 0.000000 0.000000 setrgbcolor
%%IncludeResource: font Math2Mono
%%IncludeFont: Math2Mono
/Math2Mono findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
68.500000 9.062500 moveto
(\\217) show
75.687500 9.062500 moveto
(!) show
78.375000 9.062500 moveto
(!) show
80.437500 9.062500 moveto
(!) show
77.000000 13.750000 moveto
%%IncludeResource: font Math1Mono
%%IncludeFont: Math1Mono
/Math1Mono findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(a) show
85.062500 13.750000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1Mono
%%+ font Math2Mono
%%+ font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1Mono
%%+ Math2Mono
%%+ Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1Mono
%%+ font Math2Mono
%%+ font Courier
grestore
0 .01472 m
.00625 .01472 L
s
[(0)] -0.0125 .01472 1 0 Mshowa
0 .11411 m
.00625 .11411 L
s
[(50)] -0.0125 .11411 1 0 Mshowa
0 .21351 m
.00625 .21351 L
s
[(100)] -0.0125 .21351 1 0 Mshowa
0 .31291 m
.00625 .31291 L
s
[(150)] -0.0125 .31291 1 0 Mshowa
0 .4123 m
.00625 .4123 L
s
[(200)] -0.0125 .4123 1 0 Mshowa
0 .5117 m
.00625 .5117 L
s
[(250)] -0.0125 .5117 1 0 Mshowa
0 .6111 m
.00625 .6111 L
s
[(300)] -0.0125 .6111 1 0 Mshowa
.125 Mabswid
0 .03459 m
.00375 .03459 L
s
0 .05447 m
.00375 .05447 L
s
0 .07435 m
.00375 .07435 L
s
0 .09423 m
.00375 .09423 L
s
0 .13399 m
.00375 .13399 L
s
0 .15387 m
.00375 .15387 L
s
0 .17375 m
.00375 .17375 L
s
0 .19363 m
.00375 .19363 L
s
0 .23339 m
.00375 .23339 L
s
0 .25327 m
.00375 .25327 L
s
0 .27315 m
.00375 .27315 L
s
0 .29303 m
.00375 .29303 L
s
0 .33278 m
.00375 .33278 L
s
0 .35266 m
.00375 .35266 L
s
0 .37254 m
.00375 .37254 L
s
0 .39242 m
.00375 .39242 L
s
0 .43218 m
.00375 .43218 L
s
0 .45206 m
.00375 .45206 L
s
0 .47194 m
.00375 .47194 L
s
0 .49182 m
.00375 .49182 L
s
0 .53158 m
.00375 .53158 L
s
0 .55146 m
.00375 .55146 L
s
0 .57134 m
.00375 .57134 L
s
0 .59122 m
.00375 .59122 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
gsave
-0.01875 .30902 -89 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 15.187500 moveto
 461.000000 15.187500 lineto
 461.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 15.187500 lineto
closepath
clip newpath
63.000000 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 12.312500 moveto
%%IncludeResource: font Math1Mono
%%IncludeFont: Math1Mono
/Math1Mono findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(w) show
69.000000 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1Mono
%%+ font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1Mono
%%+ Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1Mono
%%+ font Courier
grestore
.02381 .61178 m
.02381 .61803 L
s
.18254 .61178 m
.18254 .61803 L
s
.34127 .61178 m
.34127 .61803 L
s
.5 .61178 m
.5 .61803 L
s
.65873 .61178 m
.65873 .61803 L
s
.81746 .61178 m
.81746 .61803 L
s
.97619 .61178 m
.97619 .61803 L
s
.125 Mabswid
.05556 .61428 m
.05556 .61803 L
s
.0873 .61428 m
.0873 .61803 L
s
.11905 .61428 m
.11905 .61803 L
s
.15079 .61428 m
.15079 .61803 L
s
.21429 .61428 m
.21429 .61803 L
s
.24603 .61428 m
.24603 .61803 L
s
.27778 .61428 m
.27778 .61803 L
s
.30952 .61428 m
.30952 .61803 L
s
.37302 .61428 m
.37302 .61803 L
s
.40476 .61428 m
.40476 .61803 L
s
.43651 .61428 m
.43651 .61803 L
s
.46825 .61428 m
.46825 .61803 L
s
.53175 .61428 m
.53175 .61803 L
s
.56349 .61428 m
.56349 .61803 L
s
.59524 .61428 m
.59524 .61803 L
s
.62698 .61428 m
.62698 .61803 L
s
.69048 .61428 m
.69048 .61803 L
s
.72222 .61428 m
.72222 .61803 L
s
.75397 .61428 m
.75397 .61803 L
s
.78571 .61428 m
.78571 .61803 L
s
.84921 .61428 m
.84921 .61803 L
s
.88095 .61428 m
.88095 .61803 L
s
.9127 .61428 m
.9127 .61803 L
s
.94444 .61428 m
.94444 .61803 L
s
.25 Mabswid
0 .61803 m
1 .61803 L
s
.99375 .01472 m
1 .01472 L
s
.99375 .11411 m
1 .11411 L
s
.99375 .21351 m
1 .21351 L
s
.99375 .31291 m
1 .31291 L
s
.99375 .4123 m
1 .4123 L
s
.99375 .5117 m
1 .5117 L
s
.99375 .6111 m
1 .6111 L
s
.125 Mabswid
.99625 .03459 m
1 .03459 L
s
.99625 .05447 m
1 .05447 L
s
.99625 .07435 m
1 .07435 L
s
.99625 .09423 m
1 .09423 L
s
.99625 .13399 m
1 .13399 L
s
.99625 .15387 m
1 .15387 L
s
.99625 .17375 m
1 .17375 L
s
.99625 .19363 m
1 .19363 L
s
.99625 .23339 m
1 .23339 L
s
.99625 .25327 m
1 .25327 L
s
.99625 .27315 m
1 .27315 L
s
.99625 .29303 m
1 .29303 L
s
.99625 .33278 m
1 .33278 L
s
.99625 .35266 m
1 .35266 L
s
.99625 .37254 m
1 .37254 L
s
.99625 .39242 m
1 .39242 L
s
.99625 .43218 m
1 .43218 L
s
.99625 .45206 m
1 .45206 L
s
.99625 .47194 m
1 .47194 L
s
.99625 .49182 m
1 .49182 L
s
.99625 .53158 m
1 .53158 L
s
.99625 .55146 m
1 .55146 L
s
.99625 .57134 m
1 .57134 L
s
.99625 .59122 m
1 .59122 L
s
.25 Mabswid
1 0 m
1 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
1 0 0 r
.5 Mabswid
.02381 .01472 m
.06244 .03859 L
.10458 .06463 L
.14415 .08909 L
.18221 .11261 L
.22272 .13765 L
.26171 .16175 L
.30316 .18736 L
.34309 .21204 L
.3815 .23578 L
.42237 .26104 L
.46172 .28536 L
.49955 .30874 L
.53984 .33364 L
.57861 .3576 L
.61984 .38308 L
.65954 .40762 L
.69774 .43123 L
.73838 .45635 L
.77751 .48053 L
.81909 .50623 L
.85916 .53099 L
.89771 .55482 L
.93871 .58016 L
.97619 .60332 L
s
0 0 1 r
.02381 .1812 m
.06244 .1812 L
.10458 .1812 L
.14415 .1812 L
.18221 .1812 L
.22272 .1812 L
.26171 .1812 L
.30316 .1812 L
.34309 .1812 L
.3815 .1812 L
.42237 .1812 L
.46172 .1812 L
.49955 .1812 L
.53984 .1812 L
.57861 .1812 L
.61984 .1812 L
.65954 .1812 L
.69774 .1812 L
.73838 .1812 L
.77751 .1812 L
.81909 .1812 L
.85916 .1812 L
.89771 .1812 L
.93871 .1812 L
.97619 .1812 L
s
0 g
gsave
.29318 .20505 -65.75 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 15.187500 moveto
 461.000000 15.187500 lineto
 461.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 15.187500 lineto
closepath
clip newpath
63.000000 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(C) show
68.500000 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.1136 .02465 -71.7188 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 15.187500 moveto
 461.000000 15.187500 lineto
 461.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 15.187500 lineto
closepath
clip newpath
63.000000 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(b) show
68.500000 8.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
72.312500 8.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(:) show
76.125000 8.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(3) show
80.437500 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.29318 .02465 -71.7188 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 15.187500 moveto
 461.000000 15.187500 lineto
 461.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 15.187500 lineto
closepath
clip newpath
63.000000 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(b) show
68.500000 8.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
72.312500 8.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(:) show
76.125000 8.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
80.437500 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.83193 .02465 -71.7188 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 15.187500 moveto
 461.000000 15.187500 lineto
 461.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 15.187500 lineto
closepath
clip newpath
63.000000 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(b) show
68.500000 8.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(3) show
72.312500 8.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(:) show
76.125000 8.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
80.437500 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  TextAlignment->Left,
  TextJustification->1,
  ImageSize->{288, 177.938},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/AP00`40o`000?l0003oooooob7oool0
0?oooolQoooo002Noooo0P00083oool009koool20000P?ooo`00U?ooo`@00005oooo00<0003oool0
00000oooo`8000000oooo`000?ooo`1hoooo002Eoooo00@0003oooooool00002oooo00D0003oool0
003oool000000_ooo`040000oooooooo0000N_ooo`00UOooo`040000oooooooo00000oooo`070000
oooooooooooo0000oooo000000;oool00`000?ooooooo`1hoooo002Eoooo0`0000Soool00`000?oo
o`000002oooo00<0003oooooool0N?ooo`00UOooo`030000oooooooo00Soool00`000?ooooooo`02
000000?oool0003oool0N?ooo`00U?ooo`80000;oooo00<0003oooooool0O?ooo`00XOooo`030000
oooooooo07coool00:3oool90000Moooo`00ooooob7oool00?oooolQoooo000/oooo00<0003ooooo
ool08oooo`030000oooooooo01ooool500000_ooo`030000oooooooo01coool500000_ooo`030000
oooooooo01goool300000oooo`030000oooooooo01koool300000oooo`030000oooooooo01goool2
00001?ooo`030000oooooooo00Woool002_oool00`000?ooo`00000Soooo00<0003oool000008_oo
o`070000oooooooooooo0000oooo000001ooool01`000?ooooooooooo`000?ooo`00000Noooo00<0
003oooooool00_ooo`030000oooo000001ooool00`000?ooooooo`02oooo00<0003oool000008?oo
o`060000oooooooo0000oooo00002_ooo`00:oooo`030000oooo000002Goool00`000?ooooooo`0P
oooo00L0003oooooooooool0003oool000007oooo`030000oooooooo00?oool00`000?ooooooo`0M
oooo00L0003oooooooooool0003oool000008?ooo`030000oooooooo00?oool00`000?ooooooo`0M
oooo00L0003oooooooooool0003oool000002_ooo`00:oooo`030000oooo000002?oool200008ooo
o`070000oooooooooooo0000oooo000001ooool010000?ooooooooooo`80000Poooo00L0003ooooo
ooooool0003oool000008?ooo`040000oooooooooooo0P00027oool01P000?ooooooo`000?ooo`00
00[oool002_oool00`000?ooo`00000Soooo00<0003oooooool08_ooo`070000oooooooooooo0000
oooo000001ooool01@000?ooooooooooo`00000Poooo00<0003oool000000_ooo`030000oooo0000
01ooool00`000?ooo`000002oooo00<0003oooooool07_ooo`030000oooo000000;oool00`000?oo
o`00000:oooo000/oooo00<0003oooooool08_ooo`<0000Qoooo0P0000Coool00`000?ooooooo`0M
oooo0P0000?oool300007oooo`030000oooooooo00;oool00`000?ooooooo`0Ooooo00@0003ooooo
ooooool300007oooo`030000oooooooo00;oool00`000?ooooooo`09oooo003ooooo8Oooo`00oooo
ob7oool00?oooolQoooo003ooooo8Oooo`00ooooob7oool002Ooooob00001oooo`008Oooo`030000
oooooooo00?oool00`000?ooooooo`02oooo00<0003oooooool09?ooo`030000oooooooo02?oool0
0`000?ooooooo`0Soooo00<0003oooooool08oooo`030000oooooooo02Coool00`000?ooooooo`0S
oooo00<0003oooooool00oooo`030000oooooooo00Goool0023oool00`000?ooo`000004oooo00<0
003oooooool0k_ooo`030000oooooooo00Goool0023oool00`000?ooo`000004oooo0P0000?oool0
0ol00?ooooooo`3Woooo0`0000Ooool0023oool00`000?ooo`000004oooo00<0003oooooool00ooo
o`;o000:oooo100002Soool40000OOooo`@0000^oooo00<0003oooooool01Oooo`008?ooo`030000
oooo000000Coool00`000?ooooooo`05oooo0_l000Woool010000?ooooooo`0002Soool010000?oo
ooooo`0007goool010000?ooooooo`0002goool00`000?ooooooo`05oooo000Qoooo00<0003ooooo
ool00oooo`030000oooooooo00Ooool00ol00?ooooooo`06oooo00@0003oooooool0000Xoooo00@0
003oooooool0001moooo00@0003oooooool0000]oooo00<0003oooooool01Oooo`009oooo`030000
oooooooo00Soool2o`001_ooo`<00003oooo0`0000;oool00`000?ooo`00000200007?ooo`<00002
oooo0`0000;oool00`000?ooooooo`030000LOooo`<00003oooo0`0000;oool00`000?ooooooo`03
00008?ooo`030000oooooooo00Goool002Ooool00`000?ooooooo`0:oooo0_l000Coool00`000?oo
ooooo`04oooo00<0003oooooool01Oooo`030000oooooooo01[oool00`000?ooooooo`03oooo00<0
003oooooool01Oooo`030000oooooooo073oool00`000?ooooooo`05oooo00<0003oooooool01?oo
o`030000oooooooo01koool200001oooo`009oooo`030000oooooooo00coool00ol00?ooooooo`02
00001_ooo`050000oooooooooooo000000;oool00`000?ooooooo`0Joooo0P0000Goool01@000?oo
ooooooooo`000003oooo00<0003oooooool0Koooo`800006oooo00D0003oooooooooool000000ooo
o`030000oooooooo01ooool00`000?ooooooo`05oooo000Woooo00<0003oooooool03Oooo`;o0008
oooo00<0003oooooool01Oooo`030000oooooooo023oool00`000?ooooooo`05oooo00<0003ooooo
ool0N?ooo`030000oooooooo00Coool00`000?ooooooo`0Ooooo00<0003oooooool01Oooo`009ooo
o`030000oooooooo00ooool2o`001Oooo`800005oooo0`00027oool200001_ooo`80001hoooo0`00
00Goool200008Oooo`030000oooooooo00Goool002Ooool00`000?ooooooo`0Aoooo00?o003ooooo
ool0f_ooo`030000oooooooo00Goool002Ooool00`000?ooooooo`0Boooo0_l00=Woool200001ooo
o`009oooo`030000oooooooo01Coool2o`00f?ooo`030000oooooooo00Goool002Ooool00`000?oo
ooooo`0Foooo00?o003oooooool0eOooo`030000oooooooo00Goool002Ooool00`000?ooooooo`0G
oooo0_l00=Goool00`000?ooooooo`05oooo000Woooo00<0003oooooool06Oooo`03o`00oooooooo
0=;oool00`000?ooooooo`05oooo000Woooo00<0003oooooool06_ooo`;o003Aoooo0P0000Ooool0
02Ooool00`000?ooooooo`0Loooo00?o003oooooool0coooo`030000oooooooo00Goool002Ooool0
0`000?ooooooo`0Moooo0_l00<ooool00`000?ooooooo`05oooo000Woooo00<0003oooooool07ooo
o`03o`00oooooooo0<coool00`000?ooooooo`05oooo000Woooo00<0003oooooool08?ooo`;o003;
oooo0P0000Ooool002Ooool00`000?ooooooo`0Roooo00?o003oooooool0bOooo`030000oooooooo
00Goool002Ooool00`000?ooooooo`0Soooo0_l00<Woool00`000?ooooooo`05oooo000Loooo00<0
003oooooool00_ooo`030000oooooooo00?oool00`000?ooooooo`0Uoooo00?o003oooooool0a_oo
o`030000oooooooo00Goool001_oool00`000?ooo`000002oooo00<0003oool000001?ooo`030000
oooooooo02Koool2o`00a_ooo`030000oooooooo00Goool001goool01P000?ooooooo`000?ooo`00
00Coool20000:Oooo`03o`00oooooooo0<7oool300001oooo`006oooo`800003oooo00<0003oool0
00001?ooo`030000oooooooo02Woool2o`00`oooo`030000oooooooo00Goool001_oool00`000?oo
ooooo`02oooo00<0003oool000001?ooo`030000oooooooo02_oool2o`00`Oooo`030000oooooooo
00Goool001_oool300000oooo`030000oooooooo00?oool00`000?ooooooo`0]oooo00?o003ooooo
ool0__ooo`030000oooooooo00Goool002Ooool00`000?ooooooo`0^oooo0_l00;koool00`000?oo
ooooo`05oooo000Woooo00<0003oooooool0<?ooo`;o002koooo0P0000Ooool002Ooool00`000?oo
ooooo`0boooo00?o003oooooool0^Oooo`030000oooooooo00Goool002Ooool00`000?ooooooo`0c
oooo0_l00;Woool00`000?ooooooo`05oooo000Woooo00<0003oooooool0=Oooo`;o002goooo00<0
003oooooool01Oooo`009oooo`030000oooooooo03Ooool00ol00?ooooooo`2doooo00<0003ooooo
ool01Oooo`009oooo`030000oooooooo03Soool2o`00/oooo`800007oooo000Woooo00<0003ooooo
ool0>_ooo`;o002boooo00<0003oooooool01Oooo`009oooo`030000oooooooo03coool00ol00?oo
ooooo`2_oooo00<0003oooooool01Oooo`009oooo`030000oooooooo03goool2o`00[oooo`030000
oooooooo00Goool002Ooool00`000?ooooooo`0ooooo0_l00:coool200001oooo`009oooo`030000
oooooooo047oool00ol00?ooooooo`2Zoooo00<0003oooooool01Oooo`009oooo`030000oooooooo
00;ooooW003o1Oooo`030000oooooooo00Goool002Ooool00`000?ooooooo`14oooo0_l00:Soool0
0`000?ooooooo`05oooo000Woooo00<0003oooooool0A_ooo`03o`00oooooooo0:Goool00`000?oo
ooooo`05oooo000Woooo00<0003oooooool0Aoooo`;o002Toooo0P0000Ooool002Ooool00`000?oo
ooooo`12oooo0P0000Goool00ol00?ooooooo`2Roooo00<0003oooooool01Oooo`009oooo`030000
oooooooo047oool00`000?ooooooo`06oooo0_l00:;oool00`000?ooooooo`05oooo000Eoooo1@00
00;oool00`000?ooooooo`02oooo00<0003oooooool00oooo`030000oooooooo047oool00`000?oo
ooooo`08oooo00?o003oooooool0Woooo`030000oooooooo00Goool001Ooool01`000?oooooooooo
o`000?ooo`000002oooo00<0003oool000001?ooo`030000oooooooo047oool00`000?ooooooo`09
oooo0_l009ooool00`000?ooooooo`05oooo000Goooo00L0003oooooooooool0003oool000000_oo
o`030000oooo000000Coool20000@_ooo`030000oooo000000_oool00ol00?ooooooo`2Joooo0`00
00Ooool001Ooool01`000?ooooooooooo`000?ooo`000002oooo00<0003oool000001?ooo`030000
oooooooo047oool300003?ooo`;o002Loooo00<0003oooooool01Oooo`005oooo`070000oooooooo
oooo0000oooo000000;oool00`000?ooo`000004oooo00<0003oooooool0D_ooo`;o002Joooo00<0
003oooooool01Oooo`005_ooo`800004oooo00<0003oooooool00_ooo`030000oooooooo00?oool0
0`000?ooooooo`1Doooo0_l009Soool00`000?ooooooo`05oooo000Woooo00<0003oooooool0E_oo
o`;o002Foooo00<0003oooooool01Oooo`009oooo`030000oooooooo05Soool2o`00Toooo`800007
oooo000Woooo00<0003oooooool0F_ooo`03o`00oooooooo097oool00`000?ooooooo`05oooo000W
oooo00<0003oooooool0Foooo`;o002Aoooo00<0003oooooool01Oooo`009oooo`030000oooooooo
05goool00ol00?ooooooo`2>oooo00<0003oooooool01Oooo`009oooo`030000oooooooo05koool0
0ol00?ooooooo`2=oooo00<0003oooooool01Oooo`009oooo`030000oooooooo05ooool2o`00S?oo
o`800007oooo000Woooo00<0003oooooool0HOooo`03o`00oooooooo08[oool00`000?ooooooo`05
oooo000Woooo00<0003oooooool0H_ooo`03o`00oooooooo08Woool00`000?ooooooo`05oooo000W
oooo00<0003oooooool0Hoooo`;o0029oooo00<0003oooooool01Oooo`009oooo`030000oooooooo
06Goool2o`00Q_ooo`800007oooo000Woooo00<0003oooooool0Ioooo`;o0025oooo00<0003ooooo
ool01Oooo`009oooo`030000oooooooo06Woool2o`00Poooo`030000oooooooo00Goool002Ooool0
0`000?ooooooo`1[oooo0_l0087oool00`000?ooooooo`05oooo000Woooo00<0003oooooool0KOoo
o`;o001ooooo00<0003oooooool01Oooo`009oooo`030000oooooooo06ooool00ol00?ooooooo`1k
oooo0P0000Ooool002Ooool00`000?ooooooo`1`oooo0_l007coool00`000?ooooooo`05oooo0009
oooo100001[oool00`000?ooooooo`1boooo00?o003oooooool0NOooo`030000oooooooo00Goool0
00Soool01P000?ooooooo`000?ooo`0000Ooool500000_ooo`030000oooooooo00;oool00`000?oo
ooooo`03oooo00<0003oooooool0Loooo`;o001ioooo00<0003oooooool01Oooo`002?ooo`060000
oooooooo0000oooo00002Oooo`070000oooooooooooo0000oooo000000;oool00`000?ooo`000004
oooo00<0003oooooool0MOooo`03o`00oooooooo07Koool00`000?ooooooo`05oooo0008oooo00<0
003oooooool00_ooo`030000oooooooo00Ooool00`000?ooooooo`03oooo00H0003oooooool0003o
ool00004oooo0P0007Ooool2o`00M?ooo`<00007oooo0008oooo00<0003oooooool00_ooo`030000
oooooooo00Ooool010000?ooooooooooo`800003oooo00<0003oool000001?ooo`030000oooooooo
07Soool2o`00M?ooo`030000oooooooo00Goool001Ooool01@000?ooooooooooo`000004oooo00<0
003oool000001?ooo`030000oooooooo07[oool00ol00?ooooooo`1aoooo00<0003oooooool01Ooo
o`005_ooo`800003oooo0`0000?oool00`000?ooooooo`03oooo00<0003oooooool0Noooo`;o001a
oooo00<0003oooooool01Oooo`009oooo`030000oooooooo07goool2o`00Koooo`030000oooooooo
00Goool002Ooool00`000?ooooooo`1ooooo00?o003oooooool0Joooo`800007oooo000Woooo00<0
003oooooool0P?ooo`;o001/oooo00<0003oooooool01Oooo`009oooo`030000oooooooo08;oool0
0ol00?ooooooo`1Yoooo00<0003oooooool01Oooo`009oooo`030000oooooooo08?oool2o`00JOoo
o`030000oooooooo00Goool002Ooool00`000?ooooooo`25oooo00?o003oooooool0I_ooo`030000
oooooooo00Goool002Ooool00`000?ooooooo`26oooo0_l006Goool200001oooo`009oooo`030000
oooooooo08Soool00ol00?ooooooo`1Soooo00<0003oooooool01Oooo`009oooo`030000oooooooo
08Woool2o`00Hoooo`030000oooooooo00Goool002Ooool00`000?ooooooo`2;oooo0_l0067oool0
0`000?ooooooo`05oooo000Woooo00<0003oooooool0SOooo`03o`00oooooooo05goool200001ooo
o`009oooo`030000oooooooo08koool2o`00G_ooo`030000oooooooo00Goool002Ooool00`000?oo
ooooo`2@oooo0_l005coool00`000?ooooooo`05oooo000Woooo00<0003oooooool0T_ooo`03o`00
oooooooo05Woool00`000?ooooooo`05oooo000Woooo00<0003oooooool0Toooo`;o001Ioooo00<0
003oooooool01Oooo`009oooo`030000oooooooo09Goool2o`00E_ooo`800007oooo000Woooo00<0
003oooooool0Uoooo`03o`00oooooooo05Coool00`000?ooooooo`05oooo000Woooo00<0003ooooo
ool0V?ooo`;o001Doooo00<0003oooooool01Oooo`005_ooo`<00003oooo00<0003oooooool00_oo
o`030000oooooooo00?oool00`000?ooooooo`2Joooo0_l005;oool00`000?ooooooo`05oooo000F
oooo00<0003oooooool00_ooo`030000oooo000000;oool00`000?ooo`000004oooo00<0003ooooo
ool0W?ooo`03o`00oooooooo04ooool00`000?ooooooo`05oooo000Goooo00L0003oooooooooool0
003oool000000_ooo`030000oooo000000Coool20000W_ooo`;o001=oooo0`0000Ooool001Ooool0
1`000?ooooooooooo`000?ooo`000002oooo00<0003oool000001?ooo`030000oooooooo09ooool0
0ol00?ooooooo`1<oooo00<0003oooooool01Oooo`005_ooo`030000oooo000000;oool00`000?oo
o`000002oooo00<0003oool000001?ooo`030000oooooooo0:3oool2o`00C?ooo`030000oooooooo
00Goool001Ooool00`000?ooooooo`02oooo00<0003oooooool00_ooo`030000oooooooo00?oool0
0`000?ooooooo`2Roooo00?o003oooooool0BOooo`030000oooooooo00Goool002Ooool00`000?oo
ooooo`2Soooo0_l004Woool00`000?ooooooo`05oooo000Woooo00<0003oooooool0YOooo`03o`00
oooooooo04Goool200001oooo`009oooo`030000oooooooo0:Koool2o`00A_ooo`030000oooooooo
00Goool002Ooool00`000?ooooooo`2Xoooo0_l004Coool00`000?ooooooo`05oooo000Woooo00<0
003oooooool0Z_ooo`03o`00oooooooo047oool00`000?ooooooo`05oooo000Woooo00<0003ooooo
ool0Zoooo`;o0011oooo00<0003oooooool01Oooo`009oooo`030000oooooooo0:goool2o`00?_oo
o`800007oooo000Woooo00<0003oooooool0[oooo`03o`00oooooooo03coool00`000?ooooooo`05
oooo000Woooo00<0003oooooool0/?ooo`;o000loooo00<0003oooooool01Oooo`009oooo`030000
oooooooo0;;oool2o`00>_ooo`030000oooooooo00Goool002Ooool00`000?ooooooo`2doooo0_l0
03Ooool200001oooo`009oooo`030000oooooooo0;Koool2o`00=_ooo`030000oooooooo00Goool0
02Ooool00`000?ooooooo`2hoooo00?o003oooooool0<oooo`030000oooooooo00Goool002Ooool0
0`000?ooooooo`2ioooo0_l003?oool00`000?ooooooo`05oooo000Woooo00<0003oooooool0^ooo
o`03o`00oooooooo033oool00`000?ooooooo`05oooo000Woooo00<0003oooooool0_?ooo`03o`00
oooooooo02koool200001oooo`009oooo`030000oooooooo0;goool2o`00;oooo`030000oooooooo
00Goool002Ooool00`000?ooooooo`2ooooo00?o003oooooool0;?ooo`030000oooooooo00Goool0
01Koool300000oooo`030000oooooooo00;oool00`000?ooooooo`03oooo00<0003oooooool0`?oo
o`;o000/oooo00<0003oooooool01Oooo`005_ooo`030000oooooooo00;oool00`000?ooo`000002
oooo00<0003oool000001?ooo`030000oooooooo0<;oool00ol00?ooooooo`0Yoooo00<0003ooooo
ool01Oooo`005oooo`030000oooooooo00?oool01P000?ooooooo`000?ooo`0000Coool20000a?oo
o`;o000Woooo0`0000Ooool001Ooool010000?ooooooooooo`800003oooo00<0003oool000001?oo
o`030000oooooooo0<Goool2o`009oooo`030000oooooooo00Goool001Koool00`000?ooo`000002
oooo00<0003oooooool00_ooo`030000oooo000000Coool00`000?ooooooo`37oooo00?o003ooooo
ool09?ooo`030000oooooooo00Goool001Ooool010000?ooooooooooo`<00003oooo00<0003ooooo
ool00oooo`030000oooooooo0<Soool2o`009?ooo`030000oooooooo00Goool002Ooool00`000?oo
ooooo`3:oooo0_l002;oool00`000?ooooooo`05oooo000Woooo00<0003oooooool0c?ooo`03o`00
oooooooo01koool200001oooo`009oooo`030000oooooooo0<goool2o`007oooo`030000oooooooo
00Goool002Ooool00`000?ooooooo`3?oooo0_l001goool00`000?ooooooo`05oooo000Woooo00<0
003oooooool0dOooo`;o000Koooo00<0003oooooool01Oooo`009oooo`030000oooooooo0=?oool2
o`006Oooo`030000oooooooo00Goool002Ooool00`000?ooooooo`3Eoooo00?o003oooooool05Ooo
o`800007oooo000Woooo00<0003oooooool0e_ooo`;o000Foooo00<0003oooooool01Oooo`009ooo
o`030000oooooooo0=Soool00ol00?ooooooo`0Coooo00<0003oooooool01Oooo`009oooo`030000
oooooooo0=Woool00ol00?ooooooo`0Boooo00<0003oooooool01Oooo`009oooo`030000oooooooo
0=[oool2o`004Oooo`800007oooo000Woooo00<0003oooooool0g?ooo`03o`00oooooooo00ooool0
0`000?ooooooo`05oooo000Woooo00<0003oooooool0gOooo`;o000?oooo00<0003oooooool01Ooo
o`009oooo`030000oooooooo0=ooool00ol00?ooooooo`0<oooo00<0003oooooool01Oooo`009ooo
o`030000oooooooo0>3oool2o`003?ooo`030000oooooooo00Goool002Ooool00`000?ooooooo`3R
oooo0_l000Woool200001oooo`009oooo`030000oooooooo0>Coool2o`002?ooo`030000oooooooo
00Goool002Ooool00`000?ooooooo`3Voooo0_l000Koool00`000?ooooooo`05oooo000Foooo0P00
00Coool00`000?ooooooo`02oooo00<0003oooooool00oooo`030000oooooooo0>Soool00ol00?oo
ooooo`03oooo00<0003oooooool01Oooo`006?ooo`060000oooooooo0000oooo00000_ooo`030000
oooo000000Coool00`000?ooooooo`3^oooo00<0003oooooool01Oooo`005oooo`070000oooooooo
oooo0000oooo000000;oool00`000?ooo`000004oooo0P0000?oool00`000?ooooooo`0Toooo00<0
003oooooool08oooo`030000oooooooo02?oool00`000?ooooooo`0Soooo00<0003oooooool09?oo
o`030000oooooooo02?oool010000?ooooooooooo`<00007oooo000Hoooo00H0003oooooool0003o
ool00002oooo00<0003oool000001?ooo`030000oooooooo00;oool00`000?ooooooo`05oooo00<0
003oooooool01Oooo`030000oooooooo00Coool00`000?ooooooo`05oooo00<0003oooooool01Ooo
o`030000oooooooo00Coool00`000?ooooooo`05oooo00<0003oooooool01Oooo`030000oooooooo
00Coool00`000?ooooooo`05oooo00<0003oooooool01Oooo`030000oooooooo00Coool00`000?oo
ooooo`05oooo00<0003oooooool01?ooo`030000oooooooo00Goool00`000?ooooooo`05oooo00<0
003oooooool01?ooo`030000oooooooo00Goool00`000?ooooooo`05oooo00<0003oooooool01?oo
o`030000oooooooo00Goool00`000?ooooooo`05oooo00<0003oooooool01?ooo`030000oooooooo
00Goool00`000?ooooooo`05oooo00<0003oooooool01?ooo`030000oooooooo00Goool00`000?oo
ooooo`05oooo00<0003oooooool01?ooo`030000oooooooo00Goool00`000?ooooooo`03oooo00<0
003oooooool01Oooo`005_ooo`030000oooo000000;oool00`000?ooo`000002oooo00<0003oool0
00001?oooo800007oooo000Goooo00<0003oooooool00_ooo`030000oooooooo00;oool00`000?oo
ooooo`3loooo003ooooo8Oooo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-5.84439, -59.5667, 
  0.130577, 2.08523}}],

Cell[TextData[{
  "It follows from this figure that the frequencies of the first and third \
modes are equal when ",
  Cell[BoxData[
      \(TraditionalForm\`b \@ \[Alpha] = b\^\(1 : 1\)\)]],
  ", the frequency of the third mode is three times that of the first mode \
when ",
  Cell[BoxData[
      \(TraditionalForm\`\(b \@ \[Alpha] = b\^\(1 : 3\), \)\)]],
  " and the frequency of the first mode is three times that of the third mode \
when ",
  Cell[BoxData[
      \(TraditionalForm\`\(b \@ \[Alpha] = \)\)]],
  "  ",
  Cell[BoxData[
      \(TraditionalForm\`b\^\(3 : 1\)\)]],
  ".  Next, we use the function ",
  StyleBox["MMSDirect11",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " to treat these three cases."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell["One-to-One Internal Resonance", "Subsubsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "We denote the frequency of the third mode by ",
  Cell[BoxData[
      \(\[Omega]\_2\)]],
  " and that of the first mode by ",
  Cell[BoxData[
      \(\[Omega]\_1\)]],
  " as follows:"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(frequencies3 = {\[Omega]\_1 -> \((\[Omega] /. \[Omega]Sol1)\), 
        \[Omega]\_2 -> \((\(\[Omega] /. \[Omega]Sol2\) /. n -> 3)\)}\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Omega]\_1 \[Rule] b\ \[Pi]\^2\ \@\[Alpha], 
      \[Omega]\_2 \[Rule] 6\ \@2\ \[Pi]\^2}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Moreover, we define ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_k\)]],
  " and the first and third eigenmodes of the buckled beam as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(modeshapes3 = 
      Join[modeshape, {\[CapitalPhi]\_1 -> Function[x, shape1 // Evaluate], 
          \[CapitalPhi]\_2 -> Function[x, shape3 // Evaluate]}]; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
We consider a combination of a one-to-one internal resonance between the \
first and third modes and a principal parametric resonance of the third mode; \
that is,\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ResonanceConds3 = {
        \[Omega]\_2 == \[Omega]\_1 + \[Epsilon]\^2\ \[Sigma]\_1, 
        \[CapitalOmega] == 2  \[Omega]\_2 + \[Epsilon]\^2\ \[Sigma]\_2}; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Using ",
  StyleBox["MMSDirect11",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain the solvability conditions"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(MMSDirect11[frequencies3, modeshapes3, ResonanceConds3] // Timing\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(26.8090000000001982`\ Second\), ",", 
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
              \(\(-2\)\ I\ \[Mu]\ \[Omega]\_1\ A\_1[T\_2]\), "-", 
                \(E\^\(2\ I\ T\_2\ \[Sigma]\_1 + I\ T\_2\ \[Sigma]\_2\)\ f\ 
                  \[Pi]\^2\ A\&_\_1[T\_2]\), "+", 
                \(12\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\^2\ A\&_\_1[T\_2]\), 
                "+", \(144\ E\^\(2\ I\ T\_2\ \[Sigma]\_1\)\ \[Pi]\^4\ 
                  \[Alpha]\ A\_2[T\_2]\^2\ A\&_\_1[T\_2]\), "+", 
                \(144\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\ A\_2[T\_2]\ 
                  A\&_\_2[T\_2]\), "-", 
                RowBox[{"2", " ", "I", " ", \(\[Omega]\_1\), " ", 
                  RowBox[{
                    SubsuperscriptBox["A", "1", "\[Prime]",
                      MultilineFunction->None], "[", \(T\_2\), "]"}]}]}], 
              "==", "0"}], ",", 
            RowBox[{
              RowBox[{
              \(\(-2\)\ I\ \[Mu]\ \[Omega]\_2\ A\_2[T\_2]\), "+", 
                \(144\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\ A\_2[T\_2]\ 
                  A\&_\_1[T\_2]\), "-", 
                \(6\ E\^\(I\ T\_2\ \[Sigma]\_2\)\ f\ \[Pi]\^2\ 
                  A\&_\_2[T\_2]\), "+", 
                \(144\ E\^\(\(-2\)\ I\ T\_2\ \[Sigma]\_1\)\ \[Pi]\^4\ 
                  \[Alpha]\ A\_1[T\_2]\^2\ A\&_\_2[T\_2]\), "-", 
                \(108\ \[Pi]\^4\ \[Alpha]\ A\_2[T\_2]\^2\ A\&_\_2[T\_2]\), 
                "-", 
                RowBox[{"2", " ", "I", " ", \(\[Omega]\_2\), " ", 
                  RowBox[{
                    SubsuperscriptBox["A", "2", "\[Prime]",
                      MultilineFunction->None], "[", \(T\_2\), "]"}]}]}], 
              "==", "0"}]}], "}"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
which indicates that the one-to-one internal resonance between the first and \
third modes is activated.\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["Three-to-One Internal Resonances", "Subsubsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "To treat the case in which the natural frequency of the third mode is \
approximately three times that of the first mode using the function ",
  StyleBox["MMSDirect11",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we denote these frequencies by ",
  Cell[BoxData[
      \(\[Omega]\_2\)]],
  " and ",
  Cell[BoxData[
      \(\[Omega]\_1\)]],
  ", respectively, according to"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(frequencies4 = {\[Omega]\_1 -> \((\[Omega] /. \[Omega]Sol1)\), 
        \[Omega]\_2 -> \((\(\[Omega] /. \[Omega]Sol2\) /. n -> 3)\)}\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Omega]\_1 \[Rule] b\ \[Pi]\^2\ \@\[Alpha], 
      \[Omega]\_2 \[Rule] 6\ \@2\ \[Pi]\^2}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Moreover, we define ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_k\)]],
  " and the first and third eigenmodes of the buckled beam as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(modeshapes4 = 
      Join[modeshape, {\[CapitalPhi]\_1 -> Function[x, shape1 // Evaluate], 
          \[CapitalPhi]\_2 -> Function[x, shape3 // Evaluate]}]; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
We consider a combination of a three-to-one internal resonance between the \
first and third modes and a principal parametric resonance of the third mode; \
that is,\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ResonanceConds4 = {
        \[Omega]\_2 == 3  \[Omega]\_1 + \[Epsilon]\^2\ \[Sigma]\_1, 
        \[CapitalOmega] == 2  \[Omega]\_2 + \[Epsilon]\^2\ \[Sigma]\_2}; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Using ",
  StyleBox["MMSDirect11",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain the solvability conditions"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(MMSDirect11[frequencies4, modeshapes4, ResonanceConds4] // Timing\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(21.6509999999998425`\ Second\), ",", 
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
              \(\(-2\)\ I\ \[Mu]\ \[Omega]\_1\ A\_1[T\_2]\), "+", 
                \(12\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\^2\ A\&_\_1[T\_2]\), 
                "+", \(1584\/35\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\ A\_2[T\_2]\ 
                  A\&_\_2[T\_2]\), "-", 
                RowBox[{"2", " ", "I", " ", \(\[Omega]\_1\), " ", 
                  RowBox[{
                    SubsuperscriptBox["A", "1", "\[Prime]",
                      MultilineFunction->None], "[", \(T\_2\), "]"}]}]}], 
              "==", "0"}], ",", 
            RowBox[{
              RowBox[{
              \(\(-2\)\ I\ \[Mu]\ \[Omega]\_2\ A\_2[T\_2]\), "+", 
                \(1584\/35\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\ A\_2[T\_2]\ 
                  A\&_\_1[T\_2]\), "-", 
                \(162\/35\ E\^\(I\ T\_2\ \[Sigma]\_2\)\ f\ \[Pi]\^2\ 
                  A\&_\_2[T\_2]\), "-", 
                \(2916\/35\ \[Pi]\^4\ \[Alpha]\ A\_2[T\_2]\^2\ 
                  A\&_\_2[T\_2]\), "-", 
                RowBox[{"2", " ", "I", " ", \(\[Omega]\_2\), " ", 
                  RowBox[{
                    SubsuperscriptBox["A", "2", "\[Prime]",
                      MultilineFunction->None], "[", \(T\_2\), "]"}]}]}], 
              "==", "0"}]}], "}"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
which indicates that this three-to-one internal resonance between the first \
and third modes is not activated.\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
When the frequency of first mode is approximately three times that of the \
third mode, we define the frequencies as\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(frequencies5 = {\[Omega]\_2 -> \((\[Omega] /. \[Omega]Sol1)\), 
        \[Omega]\_1 -> \((\(\[Omega] /. \[Omega]Sol2\) /. n -> 3)\)}\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Omega]\_2 \[Rule] b\ \[Pi]\^2\ \@\[Alpha], 
      \[Omega]\_1 \[Rule] 6\ \@2\ \[Pi]\^2}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Moreover, we define ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_k\)]],
  " and the third and first eigenmodes of the buckled beam as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(modeshapes5 = 
      Join[modeshape, {\[CapitalPhi]\_1 -> Function[x, shape3 // Evaluate], 
          \[CapitalPhi]\_2 -> Function[x, shape1 // Evaluate]}]; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
We consider a combination of a three-to-one internal resonance between the \
first and third modes and a principal parametric resonance of the third mode; \
that is,\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ResonanceConds5 = {
        \[Omega]\_2 == 3  \[Omega]\_1 + \[Epsilon]\^2\ \[Sigma]\_1, 
        \[CapitalOmega] == 2  \[Omega]\_2 + \[Epsilon]\^2\ \[Sigma]\_2}; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Using ",
  StyleBox["MMSDirect11",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain the solvability conditions"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(MMSDirect11[frequencies5, modeshapes5, ResonanceConds5] // Timing\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(21.7119999999995361`\ Second\), ",", 
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
              \(\(-2\)\ I\ \[Mu]\ \[Omega]\_1\ A\_1[T\_2]\), "+", 
                \(324\/5\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\^2\ A\&_\_1[T\_2]\), 
                "-", \(2736\/5\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\ A\_2[T\_2]\ 
                  A\&_\_2[T\_2]\), "-", 
                RowBox[{"2", " ", "I", " ", \(\[Omega]\_1\), " ", 
                  RowBox[{
                    SubsuperscriptBox["A", "1", "\[Prime]",
                      MultilineFunction->None], "[", \(T\_2\), "]"}]}]}], 
              "==", "0"}], ",", 
            RowBox[{
              RowBox[{
              \(\(-2\)\ I\ \[Mu]\ \[Omega]\_2\ A\_2[T\_2]\), "-", 
                \(2736\/5\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\ A\_2[T\_2]\ 
                  A\&_\_1[T\_2]\), "-", 
                \(E\^\(I\ T\_2\ \[Sigma]\_2\)\ f\ \[Pi]\^2\ A\&_\_2[T\_2]\), 
                "+", \(12\ \[Pi]\^4\ \[Alpha]\ A\_2[T\_2]\^2\ 
                  A\&_\_2[T\_2]\), "-", 
                RowBox[{"2", " ", "I", " ", \(\[Omega]\_2\), " ", 
                  RowBox[{
                    SubsuperscriptBox["A", "2", "\[Prime]",
                      MultilineFunction->None], "[", \(T\_2\), "]"}]}]}], 
              "==", "0"}]}], "}"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
which indicates that this three-to-one internal resonance between the first \
and third modes is not activated.\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
8.1.6 One-to-One and Three-to-One Internal Resonances Between the First and \
Fourth Modes\
\>", "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
Variations of the first and fourth natural frequencies with the buckling \
level are plotted as follows:\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.0190476 0.0147151 0.00119276 [
[.02381 -0.0125 -3 -9 ]
[.02381 -0.0125 3 0 ]
[.21429 -0.0125 -6 -9 ]
[.21429 -0.0125 6 0 ]
[.40476 -0.0125 -6 -9 ]
[.40476 -0.0125 6 0 ]
[.59524 -0.0125 -6 -9 ]
[.59524 -0.0125 6 0 ]
[.78571 -0.0125 -6 -9 ]
[.78571 -0.0125 6 0 ]
[.97619 -0.0125 -6 -9 ]
[.97619 -0.0125 6 0 ]
[ 0 0 -0.125 0 ]
[.5 -0.0125 -13.0313 -22 ]
[.5 -0.0125 13.0313 -9 ]
[-0.0125 .01472 -6 -4.5 ]
[-0.0125 .01472 0 4.5 ]
[-0.0125 .13399 -18 -4.5 ]
[-0.0125 .13399 0 4.5 ]
[-0.0125 .25327 -18 -4.5 ]
[-0.0125 .25327 0 4.5 ]
[-0.0125 .37254 -18 -4.5 ]
[-0.0125 .37254 0 4.5 ]
[-0.0125 .49182 -18 -4.5 ]
[-0.0125 .49182 0 4.5 ]
[-0.0125 .6111 -18 -4.5 ]
[-0.0125 .6111 0 4.5 ]
[ 0 0 -0.125 0 ]
[-0.01875 .30902 -28 -5.59375 ]
[-0.01875 .30902 -18 5.59375 ]
[ 0 .61803 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
.02381 0 m
.02381 .00625 L
s
[(0)] .02381 -0.0125 0 1 Mshowa
.21429 0 m
.21429 .00625 L
s
[(10)] .21429 -0.0125 0 1 Mshowa
.40476 0 m
.40476 .00625 L
s
[(20)] .40476 -0.0125 0 1 Mshowa
.59524 0 m
.59524 .00625 L
s
[(30)] .59524 -0.0125 0 1 Mshowa
.78571 0 m
.78571 .00625 L
s
[(40)] .78571 -0.0125 0 1 Mshowa
.97619 0 m
.97619 .00625 L
s
[(50)] .97619 -0.0125 0 1 Mshowa
.125 Mabswid
.0619 0 m
.0619 .00375 L
s
.1 0 m
.1 .00375 L
s
.1381 0 m
.1381 .00375 L
s
.17619 0 m
.17619 .00375 L
s
.25238 0 m
.25238 .00375 L
s
.29048 0 m
.29048 .00375 L
s
.32857 0 m
.32857 .00375 L
s
.36667 0 m
.36667 .00375 L
s
.44286 0 m
.44286 .00375 L
s
.48095 0 m
.48095 .00375 L
s
.51905 0 m
.51905 .00375 L
s
.55714 0 m
.55714 .00375 L
s
.63333 0 m
.63333 .00375 L
s
.67143 0 m
.67143 .00375 L
s
.70952 0 m
.70952 .00375 L
s
.74762 0 m
.74762 .00375 L
s
.82381 0 m
.82381 .00375 L
s
.8619 0 m
.8619 .00375 L
s
.9 0 m
.9 .00375 L
s
.9381 0 m
.9381 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
gsave
.5 -0.0125 -74.0313 -26 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 21 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 17.000000 moveto
 461.000000 17.000000 lineto
 461.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 17.000000 lineto
closepath
clip newpath
63.000000 13.750000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 13.750000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(b) show
0.000000 0.000000 0.000000 setrgbcolor
%%IncludeResource: font Math2Mono
%%IncludeFont: Math2Mono
/Math2Mono findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
68.500000 9.062500 moveto
(\\217) show
75.687500 9.062500 moveto
(!) show
78.375000 9.062500 moveto
(!) show
80.437500 9.062500 moveto
(!) show
77.000000 13.750000 moveto
%%IncludeResource: font Math1Mono
%%IncludeFont: Math1Mono
/Math1Mono findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(a) show
85.062500 13.750000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1Mono
%%+ font Math2Mono
%%+ font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1Mono
%%+ Math2Mono
%%+ Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1Mono
%%+ font Math2Mono
%%+ font Courier
grestore
0 .01472 m
.00625 .01472 L
s
[(0)] -0.0125 .01472 1 0 Mshowa
0 .13399 m
.00625 .13399 L
s
[(100)] -0.0125 .13399 1 0 Mshowa
0 .25327 m
.00625 .25327 L
s
[(200)] -0.0125 .25327 1 0 Mshowa
0 .37254 m
.00625 .37254 L
s
[(300)] -0.0125 .37254 1 0 Mshowa
0 .49182 m
.00625 .49182 L
s
[(400)] -0.0125 .49182 1 0 Mshowa
0 .6111 m
.00625 .6111 L
s
[(500)] -0.0125 .6111 1 0 Mshowa
.125 Mabswid
0 .03857 m
.00375 .03857 L
s
0 .06243 m
.00375 .06243 L
s
0 .08628 m
.00375 .08628 L
s
0 .11014 m
.00375 .11014 L
s
0 .15785 m
.00375 .15785 L
s
0 .1817 m
.00375 .1817 L
s
0 .20556 m
.00375 .20556 L
s
0 .22941 m
.00375 .22941 L
s
0 .27712 m
.00375 .27712 L
s
0 .30098 m
.00375 .30098 L
s
0 .32483 m
.00375 .32483 L
s
0 .34869 m
.00375 .34869 L
s
0 .3964 m
.00375 .3964 L
s
0 .42025 m
.00375 .42025 L
s
0 .44411 m
.00375 .44411 L
s
0 .46796 m
.00375 .46796 L
s
0 .51567 m
.00375 .51567 L
s
0 .53953 m
.00375 .53953 L
s
0 .56339 m
.00375 .56339 L
s
0 .58724 m
.00375 .58724 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
gsave
-0.01875 .30902 -89 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 15.187500 moveto
 461.000000 15.187500 lineto
 461.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 15.187500 lineto
closepath
clip newpath
63.000000 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 12.312500 moveto
%%IncludeResource: font Math1Mono
%%IncludeFont: Math1Mono
/Math1Mono findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(w) show
69.000000 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1Mono
%%+ font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1Mono
%%+ Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1Mono
%%+ font Courier
grestore
.02381 .61178 m
.02381 .61803 L
s
.21429 .61178 m
.21429 .61803 L
s
.40476 .61178 m
.40476 .61803 L
s
.59524 .61178 m
.59524 .61803 L
s
.78571 .61178 m
.78571 .61803 L
s
.97619 .61178 m
.97619 .61803 L
s
.125 Mabswid
.0619 .61428 m
.0619 .61803 L
s
.1 .61428 m
.1 .61803 L
s
.1381 .61428 m
.1381 .61803 L
s
.17619 .61428 m
.17619 .61803 L
s
.25238 .61428 m
.25238 .61803 L
s
.29048 .61428 m
.29048 .61803 L
s
.32857 .61428 m
.32857 .61803 L
s
.36667 .61428 m
.36667 .61803 L
s
.44286 .61428 m
.44286 .61803 L
s
.48095 .61428 m
.48095 .61803 L
s
.51905 .61428 m
.51905 .61803 L
s
.55714 .61428 m
.55714 .61803 L
s
.63333 .61428 m
.63333 .61803 L
s
.67143 .61428 m
.67143 .61803 L
s
.70952 .61428 m
.70952 .61803 L
s
.74762 .61428 m
.74762 .61803 L
s
.82381 .61428 m
.82381 .61803 L
s
.8619 .61428 m
.8619 .61803 L
s
.9 .61428 m
.9 .61803 L
s
.9381 .61428 m
.9381 .61803 L
s
.25 Mabswid
0 .61803 m
1 .61803 L
s
.99375 .01472 m
1 .01472 L
s
.99375 .13399 m
1 .13399 L
s
.99375 .25327 m
1 .25327 L
s
.99375 .37254 m
1 .37254 L
s
.99375 .49182 m
1 .49182 L
s
.99375 .6111 m
1 .6111 L
s
.125 Mabswid
.99625 .03857 m
1 .03857 L
s
.99625 .06243 m
1 .06243 L
s
.99625 .08628 m
1 .08628 L
s
.99625 .11014 m
1 .11014 L
s
.99625 .15785 m
1 .15785 L
s
.99625 .1817 m
1 .1817 L
s
.99625 .20556 m
1 .20556 L
s
.99625 .22941 m
1 .22941 L
s
.99625 .27712 m
1 .27712 L
s
.99625 .30098 m
1 .30098 L
s
.99625 .32483 m
1 .32483 L
s
.99625 .34869 m
1 .34869 L
s
.99625 .3964 m
1 .3964 L
s
.99625 .42025 m
1 .42025 L
s
.99625 .44411 m
1 .44411 L
s
.99625 .46796 m
1 .46796 L
s
.99625 .51567 m
1 .51567 L
s
.99625 .53953 m
1 .53953 L
s
.99625 .56339 m
1 .56339 L
s
.99625 .58724 m
1 .58724 L
s
.25 Mabswid
1 0 m
1 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
1 0 0 r
.5 Mabswid
.02381 .01472 m
.06244 .03859 L
.10458 .06463 L
.14415 .08909 L
.18221 .11261 L
.22272 .13765 L
.26171 .16175 L
.30316 .18736 L
.34309 .21204 L
.3815 .23578 L
.42237 .26104 L
.46172 .28536 L
.49955 .30874 L
.53984 .33364 L
.57861 .3576 L
.61984 .38308 L
.65954 .40762 L
.69774 .43123 L
.73838 .45635 L
.77751 .48053 L
.81909 .50623 L
.85916 .53099 L
.89771 .55482 L
.93871 .58016 L
.97619 .60332 L
s
0 0 1 r
.02381 .19709 m
.06244 .19709 L
.10458 .19709 L
.14415 .19709 L
.18221 .19709 L
.22272 .19709 L
.26171 .19709 L
.30316 .19709 L
.34309 .19709 L
.3815 .19709 L
.42237 .19709 L
.46172 .19709 L
.49955 .19709 L
.53984 .19709 L
.57861 .19709 L
.61984 .19709 L
.65954 .19709 L
.69774 .19709 L
.73838 .19709 L
.77751 .19709 L
.81909 .19709 L
.85916 .19709 L
.89771 .19709 L
.93871 .19709 L
.97619 .19709 L
s
0 g
gsave
.31889 .21856 -65.75 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 15.187500 moveto
 461.000000 15.187500 lineto
 461.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 15.187500 lineto
closepath
clip newpath
63.000000 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(C) show
68.500000 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.12217 .02068 -71.7188 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 15.187500 moveto
 461.000000 15.187500 lineto
 461.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 15.187500 lineto
closepath
clip newpath
63.000000 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(b) show
68.500000 8.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
72.312500 8.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(:) show
76.125000 8.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(3) show
80.437500 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.31889 .02068 -71.7188 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 15.187500 moveto
 461.000000 15.187500 lineto
 461.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 15.187500 lineto
closepath
clip newpath
63.000000 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(b) show
68.500000 8.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
72.312500 8.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(:) show
76.125000 8.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
80.437500 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.90906 .02068 -71.7188 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 15.187500 moveto
 461.000000 15.187500 lineto
 461.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 15.187500 lineto
closepath
clip newpath
63.000000 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(b) show
68.500000 8.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(3) show
72.312500 8.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(:) show
76.125000 8.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.125000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
80.437500 12.312500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  TextAlignment->Left,
  TextJustification->1,
  ImageSize->{288, 177.938},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/AP00`40o`000?l0003oooooob7oool0
0?oooolQoooo002Noooo0P00083oool009koool20000P?ooo`00U?ooo`@00005oooo00<0003oool0
00000oooo`8000000oooo`000?ooo`1hoooo002Eoooo00@0003oooooool00002oooo00D0003oool0
003oool000000_ooo`040000oooooooo0000N_ooo`00UOooo`040000oooooooo00000oooo`070000
oooooooooooo0000oooo000000;oool00`000?ooooooo`1hoooo002Eoooo0`0000Soool00`000?oo
o`000002oooo00<0003oooooool0N?ooo`00UOooo`030000oooooooo00Soool00`000?ooooooo`02
000000?oool0003oool0N?ooo`00U?ooo`80000;oooo00<0003oooooool0O?ooo`00XOooo`030000
oooooooo07coool00:3oool90000Moooo`00ooooob7oool00?oooolQoooo000/oooo00<0003ooooo
ool09_ooo`D00002oooo00<0003oooooool09Oooo`<00003oooo00<0003oooooool09Oooo`800004
oooo00<0003oooooool09_ooo`<00002oooo00<0003oooooool09_ooo`030000oooooooo00;oool0
0`000?ooooooo`09oooo000[oooo00<0003oool00000:Oooo`070000oooooooooooo0000oooo0000
02Koool00`000?ooooooo`02oooo00<0003oool00000:?ooo`060000oooooooo0000oooo0000:?oo
o`060000oooooooo0000oooo00009_ooo`030000oooo000000;oool00`000?ooo`00000:oooo000[
oooo00<0003oool00000:Oooo`070000oooooooooooo0000oooo000002Ooool01`000?oooooooooo
o`000?ooo`00000Woooo00L0003oooooooooool0003oool000009_ooo`@000001?ooo`000?ooo`00
02Soool01P000?ooooooo`000?ooo`0000[oool002_oool00`000?ooo`00000Yoooo00L0003ooooo
ooooool0003oool000009oooo`070000oooooooooooo0000oooo000002Soool01P000?ooooooo`00
0?ooo`0002Koool00`000?ooo`000002oooo00<0003oool000009_ooo`800003oooo00<0003oool0
00002_ooo`00:oooo`030000oooo000002Woool01`000?ooooooooooo`000?ooo`00000Voooo00<0
003oool000000_ooo`030000oooo000002Koool00`000?ooo`000002oooo00<0003oool000009ooo
o`800002oooo00<0003oool000009_ooo`030000oooooooo00;oool00`000?ooo`00000:oooo000/
oooo00<0003oooooool09oooo`800004oooo00<0003oooooool09_ooo`030000oooooooo00;oool0
0`000?ooooooo`0Voooo00<0003oooooool00_ooo`030000oooooooo02Ooool01@000?oooooooooo
o`00000Woooo0`0000?oool00`000?ooooooo`09oooo003ooooo8Oooo`00ooooob7oool00?oooolQ
oooo003ooooo8Oooo`00ooooob7oool002Ooooob00001oooo`008Oooo`030000oooooooo00?oool0
0`000?ooooooo`02oooo00<0003oooooool0:oooo`030000oooooooo02_oool00`000?ooooooo`0[
oooo00<0003oooooool0:oooo`030000oooooooo02_oool00`000?ooooooo`03oooo00<0003ooooo
ool01Oooo`008?ooo`030000oooo000000Coool00`000?ooooooo`3^oooo00<0003oooooool01Ooo
o`008?ooo`030000oooo000000Coool200000oooo`03o`00oooooooo00coool40000;?ooo`@0002:
oooo100001Woool300001oooo`008?ooo`030000oooo000000Coool00`000?ooooooo`03oooo0_l0
00goool010000?ooooooo`0002coool010000?ooooooo`0008[oool010000?ooooooo`0001[oool0
0`000?ooooooo`05oooo000Poooo00<0003oool000001?ooo`030000oooooooo00Goool2o`002ooo
o`040000oooooooo0000;?ooo`040000oooooooo0000R_ooo`040000oooooooo00006_ooo`030000
oooooooo00Goool0027oool00`000?ooooooo`03oooo00<0003oooooool01oooo`03o`00oooooooo
00Soool300000oooo`<00002oooo00<0003oool000000P00023oool300000_ooo`<00002oooo00<0
003oooooool00`0007koool300000oooo`<000001?ooo`000?ooooooo`<0000>oooo00<0003ooooo
ool01Oooo`009oooo`030000oooooooo00Soool2o`002?ooo`030000oooooooo00Coool00`000?oo
ooooo`05oooo00<0003oooooool07_ooo`030000oooooooo00?oool00`000?ooooooo`05oooo00<0
003oooooool0OOooo`030000oooooooo00Goool00`000?ooooooo`03oooo00<0003oooooool03Ooo
o`030000oooooooo00Goool002Ooool00`000?ooooooo`0:oooo0_l000Goool200001_ooo`050000
oooooooooooo000000;oool00`000?ooooooo`0Noooo0P0000Goool01@000?ooooooooooo`000003
oooo00<0003oooooool0O?ooo`800006oooo00@0003oooooool00003oooo00<0003oooooool03Ooo
o`030000oooooooo00Goool002Ooool00`000?ooooooo`0<oooo00?o003oooooool02_ooo`030000
oooooooo00Goool00`000?ooooooo`0Toooo00<0003oooooool01Oooo`030000oooooooo08Goool0
0`000?ooooooo`03oooo00<0003oooooool03?ooo`800007oooo000Woooo00<0003oooooool03Ooo
o`;o0009oooo0P0000Goool300009Oooo`800006oooo0P0008Goool300001?ooo`80000?oooo00<0
003oooooool01Oooo`009oooo`030000oooooooo00ooool2o`00gOooo`030000oooooooo00Goool0
02Ooool00`000?ooooooo`0Aoooo00?o003oooooool0f_ooo`030000oooooooo00Goool002Ooool0
0`000?ooooooo`0Boooo0_l00=[oool00`000?ooooooo`05oooo000Woooo00<0003oooooool05?oo
o`;o003Hoooo00<0003oooooool01Oooo`009oooo`030000oooooooo01Koool00ol00?ooooooo`3D
oooo0P0000Ooool002Ooool00`000?ooooooo`0Goooo0_l00=Goool00`000?ooooooo`05oooo000W
oooo00<0003oooooool06Oooo`03o`00oooooooo0=;oool00`000?ooooooo`05oooo000Woooo00<0
003oooooool06_ooo`;o003Boooo00<0003oooooool01Oooo`009oooo`030000oooooooo01coool0
0ol00?ooooooo`3?oooo00<0003oooooool01Oooo`009oooo`030000oooooooo01goool2o`00c_oo
o`800007oooo000Woooo00<0003oooooool07oooo`03o`00oooooooo0<coool00`000?ooooooo`05
oooo000Woooo00<0003oooooool08?ooo`;o003<oooo00<0003oooooool01Oooo`009oooo`030000
oooooooo02;oool00ol00?ooooooo`39oooo00<0003oooooool01Oooo`009oooo`030000oooooooo
02?oool2o`00bOooo`030000oooooooo00Goool002Ooool00`000?ooooooo`0Uoooo00?o003ooooo
ool0a_ooo`030000oooooooo00Goool002Ooool00`000?ooooooo`0Voooo0_l00<Goool200001ooo
o`009oooo`030000oooooooo02Soool00ol00?ooooooo`33oooo00<0003oooooool01Oooo`009ooo
o`030000oooooooo02Woool2o`00`oooo`030000oooooooo00Goool002Ooool00`000?ooooooo`0[
oooo0_l00<7oool00`000?ooooooo`05oooo000Eoooo1@0000;oool00`000?ooooooo`02oooo00<0
003oooooool00oooo`030000oooooooo02goool00ol00?ooooooo`2noooo00<0003oooooool01Ooo
o`005oooo`070000oooooooooooo0000oooo000000;oool00`000?ooo`000004oooo00<0003ooooo
ool0;_ooo`;o002noooo00<0003oooooool01Oooo`005oooo`070000oooooooooooo0000oooo0000
00;oool00`000?ooo`000004oooo0P00037oool2o`00^_ooo`<00007oooo000Goooo00L0003ooooo
ooooool0003oool000000_ooo`030000oooo000000Coool00`000?ooooooo`0boooo00?o003ooooo
ool0^Oooo`030000oooooooo00Goool001Ooool01`000?ooooooooooo`000?ooo`000002oooo00<0
003oool000001?ooo`030000oooooooo03?oool2o`00^Oooo`030000oooooooo00Goool001Koool2
00001?ooo`030000oooooooo00;oool00`000?ooooooo`03oooo00<0003oooooool0=Oooo`;o002g
oooo00<0003oooooool01Oooo`009oooo`030000oooooooo03Ooool00ol00?ooooooo`2doooo00<0
003oooooool01Oooo`009oooo`030000oooooooo03Soool2o`00]?ooo`030000oooooooo00Goool0
02Ooool00`000?ooooooo`0joooo0_l00;7oool200001oooo`009oooo`030000oooooooo03coool0
0ol00?ooooooo`2_oooo00<0003oooooool01Oooo`009oooo`030000oooooooo03goool2o`00[ooo
o`030000oooooooo00Goool002Ooool00`000?ooooooo`0ooooo0_l00:goool00`000?ooooooo`05
oooo000Woooo00<0003oooooool0@Oooo`03o`00oooooooo0:[oool00`000?ooooooo`05oooo000W
oooo00<0003oooooool0@_ooo`;o002Yoooo0P0000Ooool002Ooool00`000?ooooooo`14oooo0_l0
0:Soool00`000?ooooooo`05oooo000Woooo00<0003oooooool0A_ooo`03o`00oooooooo0:Goool0
0`000?ooooooo`05oooo000Woooo00<0003oooooool0Aoooo`;o002Uoooo00<0003oooooool01Ooo
o`009oooo`030000oooooooo00;ooooW003o1Oooo`030000oooooooo00Goool002Ooool00`000?oo
ooooo`1:oooo0_l00:;oool00`000?ooooooo`05oooo000Woooo00<0003oooooool0C?ooo`03o`00
oooooooo09koool200001oooo`009oooo`030000oooooooo04goool2o`00Woooo`030000oooooooo
00Goool002Ooool00`000?ooooooo`19oooo0P0000Coool00ol00?ooooooo`2Loooo00<0003ooooo
ool01Oooo`009oooo`030000oooooooo04Soool00`000?ooooooo`05oooo0_l009coool00`000?oo
ooooo`05oooo000Woooo00<0003oooooool0B?ooo`030000oooooooo00Ooool2o`00V_ooo`030000
oooooooo00Goool002Ooool00`000?ooooooo`18oooo00<0003oooooool02Oooo`;o002Hoooo00<0
003oooooool01Oooo`009oooo`030000oooooooo04Soool00`000?ooo`00000;oooo0_l009Goool2
00001oooo`009oooo`030000oooooooo04Soool300003Oooo`;o002Doooo00<0003oooooool01Ooo
o`009oooo`030000oooooooo05[oool00ol00?ooooooo`2Aoooo00<0003oooooool01Oooo`009ooo
o`030000oooooooo05_oool2o`00TOooo`030000oooooooo00Goool001Koool300000oooo`030000
oooooooo00;oool00`000?ooooooo`03oooo00<0003oooooool0GOooo`03o`00oooooooo08koool0
0`000?ooooooo`05oooo000Foooo00<0003oooooool00_ooo`030000oooo000000;oool00`000?oo
o`000004oooo00<0003oooooool0G_ooo`03o`00oooooooo08goool00`000?ooooooo`05oooo000G
oooo00L0003oooooooooool0003oool000000_ooo`030000oooo000000Coool20000H?ooo`;o002;
oooo0`0000Ooool001Ooool01`000?ooooooooooo`000?ooo`000002oooo00<0003oool000001?oo
o`030000oooooooo067oool00ol00?ooooooo`2:oooo00<0003oooooool01Oooo`005_ooo`030000
oooo000000;oool00`000?ooo`000002oooo00<0003oool000001?ooo`030000oooooooo06;oool0
0ol00?ooooooo`29oooo00<0003oooooool01Oooo`005oooo`030000oooooooo00;oool00`000?oo
ooooo`02oooo00<0003oooooool00oooo`030000oooooooo06?oool2o`00ROooo`030000oooooooo
00Goool002Ooool00`000?ooooooo`1Uoooo0_l008Ooool00`000?ooooooo`05oooo000Woooo00<0
003oooooool0Ioooo`;o0024oooo0P0000Ooool002Ooool00`000?ooooooo`1Yoooo0_l008?oool0
0`000?ooooooo`05oooo000Woooo00<0003oooooool0Joooo`;o0021oooo00<0003oooooool01Ooo
o`009oooo`030000oooooooo06goool2o`00Ooooo`030000oooooooo00Goool002Ooool00`000?oo
ooooo`1_oooo00?o003oooooool0O?ooo`030000oooooooo00Goool002Ooool00`000?ooooooo`1`
oooo0_l007coool00`000?ooooooo`05oooo0009oooo100001[oool00`000?ooooooo`1boooo00?o
003oooooool0N?ooo`800007oooo0008oooo00H0003oooooool0003oool0000Ioooo00<0003ooooo
ool0Loooo`;o001ioooo00<0003oooooool01Oooo`002?ooo`060000oooooooo0000oooo00006Ooo
o`030000oooooooo07Goool00ol00?ooooooo`1foooo00<0003oooooool01Oooo`002?ooo`030000
oooooooo00;oool00`000?ooooooo`0Goooo00<0003oooooool0M_ooo`;o001foooo00<0003ooooo
ool01Oooo`002?ooo`030000oooooooo00;oool00`000?ooooooo`0Goooo00<0003oooooool0N?oo
o`;o001doooo00<0003oooooool01Oooo`009oooo`030000oooooooo07[oool00ol00?ooooooo`1a
oooo00<0003oooooool01Oooo`009oooo`030000oooooooo07_oool2o`00L?ooo`800007oooo000W
oooo00<0003oooooool0OOooo`;o001_oooo00<0003oooooool01Oooo`009oooo`030000oooooooo
07ooool00ol00?ooooooo`1/oooo00<0003oooooool01Oooo`009oooo`030000oooooooo083oool2
o`00K?ooo`030000oooooooo00Goool002Ooool00`000?ooooooo`22oooo00?o003oooooool0JOoo
o`030000oooooooo00Goool002Ooool00`000?ooooooo`23oooo0_l006Woool00`000?ooooooo`05
oooo000Woooo00<0003oooooool0QOooo`03o`00oooooooo06Goool200001oooo`009oooo`030000
oooooooo08Koool2o`00I_ooo`030000oooooooo00Goool002Ooool00`000?ooooooo`28oooo00?o
003oooooool0Hoooo`030000oooooooo00Goool001Koool200001?ooo`030000oooooooo00;oool0
0`000?ooooooo`03oooo00<0003oooooool0ROooo`;o001Soooo00<0003oooooool01Oooo`006?oo
o`060000oooooooo0000oooo00000_ooo`030000oooo000000Coool00`000?ooooooo`2;oooo0_l0
067oool00`000?ooooooo`05oooo000Goooo00L0003oooooooooool0003oool000000_ooo`030000
oooo000000Coool20000S_ooo`03o`00oooooooo05coool300001oooo`006?ooo`060000oooooooo
0000oooo00000_ooo`030000oooo000000Coool00`000?ooooooo`2>oooo0_l005koool00`000?oo
ooooo`05oooo000Foooo00<0003oool000000_ooo`030000oooo000000;oool00`000?ooo`000004
oooo00<0003oooooool0T?ooo`;o001Loooo00<0003oooooool01Oooo`005oooo`030000oooooooo
00;oool00`000?ooooooo`02oooo00<0003oooooool00oooo`030000oooooooo09;oool00ol00?oo
ooooo`1Ioooo00<0003oooooool01Oooo`009oooo`030000oooooooo09?oool2o`00FOooo`030000
oooooooo00Goool002Ooool00`000?ooooooo`2Eoooo0_l005Ooool00`000?ooooooo`05oooo000W
oooo00<0003oooooool0Uoooo`03o`00oooooooo05?oool200001oooo`009oooo`030000oooooooo
09Soool2o`00E?ooo`030000oooooooo00Goool002Ooool00`000?ooooooo`2Joooo0_l005;oool0
0`000?ooooooo`05oooo000Woooo00<0003oooooool0W?ooo`03o`00oooooooo04ooool00`000?oo
ooooo`05oooo000Woooo00<0003oooooool0WOooo`;o001?oooo00<0003oooooool01Oooo`009ooo
o`030000oooooooo09ooool00ol00?ooooooo`1<oooo00<0003oooooool01Oooo`009oooo`030000
oooooooo0:3oool2o`00Boooo`800007oooo000Woooo00<0003oooooool0X_ooo`03o`00oooooooo
04Woool00`000?ooooooo`05oooo000Woooo00<0003oooooool0Xoooo`;o0019oooo00<0003ooooo
ool01Oooo`009oooo`030000oooooooo0:Goool00ol00?ooooooo`16oooo00<0003oooooool01Ooo
o`009oooo`030000oooooooo0:Koool2o`00A_ooo`030000oooooooo00Goool002Ooool00`000?oo
ooooo`2Xoooo0_l004Coool00`000?ooooooo`05oooo000Woooo00<0003oooooool0Z_ooo`03o`00
oooooooo043oool200001oooo`009oooo`030000oooooooo0:_oool2o`00@Oooo`030000oooooooo
00Goool002Ooool00`000?ooooooo`2]oooo0_l003ooool00`000?ooooooo`05oooo000Woooo00<0
003oooooool0[oooo`03o`00oooooooo03coool00`000?ooooooo`05oooo000Woooo00<0003ooooo
ool0/?ooo`;o000loooo00<0003oooooool01Oooo`009oooo`030000oooooooo0;;oool2o`00>Ooo
o`800007oooo000Woooo00<0003oooooool0]?ooo`;o000hoooo00<0003oooooool01Oooo`009ooo
o`030000oooooooo0;Koool2o`00=_ooo`030000oooooooo00Goool002Ooool00`000?ooooooo`2h
oooo00?o003oooooool0<oooo`030000oooooooo00Goool001Ooool300000_ooo`030000oooooooo
00;oool00`000?ooooooo`03oooo00<0003oooooool0^Oooo`;o000coooo00<0003oooooool01Ooo
o`006?ooo`060000oooooooo0000oooo00000_ooo`030000oooo000000Coool00`000?ooooooo`2k
oooo00?o003oooooool0<?ooo`030000oooooooo00Goool001Koool4000000Coool0003oool00002
oooo00<0003oool000001?ooo`80002moooo00?o003oooooool0;Oooo`<00007oooo000Foooo00<0
003oool000000_ooo`030000oooo000000;oool00`000?ooo`000004oooo00<0003oooooool0_Ooo
o`;o000_oooo00<0003oooooool01Oooo`005oooo`800002oooo00<0003oool000000_ooo`030000
oooo000000Coool00`000?ooooooo`2ooooo00?o003oooooool0;?ooo`030000oooooooo00Goool0
01Soool01@000?ooooooooooo`000004oooo00<0003oooooool00oooo`030000oooooooo0<3oool2
o`00;?ooo`030000oooooooo00Goool002Ooool00`000?ooooooo`32oooo00?o003oooooool0:Ooo
o`030000oooooooo00Goool002Ooool00`000?ooooooo`33oooo0_l002Woool00`000?ooooooo`05
oooo000Woooo00<0003oooooool0aOooo`;o000Voooo0P0000Ooool002Ooool00`000?ooooooo`37
oooo00?o003oooooool09?ooo`030000oooooooo00Goool002Ooool00`000?ooooooo`38oooo0_l0
02Coool00`000?ooooooo`05oooo000Woooo00<0003oooooool0b_ooo`;o000Roooo00<0003ooooo
ool01Oooo`009oooo`030000oooooooo0<coool00ol00?ooooooo`0Ooooo00<0003oooooool01Ooo
o`009oooo`030000oooooooo0<goool2o`007oooo`030000oooooooo00Goool002Ooool00`000?oo
ooooo`3?oooo0_l001coool200001oooo`009oooo`030000oooooooo0=7oool2o`006oooo`030000
oooooooo00Goool002Ooool00`000?ooooooo`3Coooo0_l001Woool00`000?ooooooo`05oooo000W
oooo00<0003oooooool0eOooo`03o`00oooooooo01Koool00`000?ooooooo`05oooo000Woooo00<0
003oooooool0e_ooo`;o000Foooo00<0003oooooool01Oooo`009oooo`030000oooooooo0=Soool0
0ol00?ooooooo`0Boooo0P0000Ooool002Ooool00`000?ooooooo`3Ioooo00?o003oooooool04_oo
o`030000oooooooo00Goool002Ooool00`000?ooooooo`3Joooo0_l001;oool00`000?ooooooo`05
oooo000Woooo00<0003oooooool0g?ooo`03o`00oooooooo00ooool00`000?ooooooo`05oooo000W
oooo00<0003oooooool0gOooo`;o000?oooo00<0003oooooool01Oooo`009oooo`030000oooooooo
0=ooool00ol00?ooooooo`0<oooo00<0003oooooool01Oooo`009oooo`030000oooooooo0>3oool2
o`002oooo`800007oooo000Woooo00<0003oooooool0h_ooo`;o000:oooo00<0003oooooool01Ooo
o`009oooo`030000oooooooo0>Coool2o`002?ooo`030000oooooooo00Goool002Ooool00`000?oo
ooooo`3Voooo0_l000Koool00`000?ooooooo`05oooo000Goooo00<0003oooooool00_ooo`030000
oooooooo00;oool00`000?ooooooo`03oooo00<0003oooooool0j?ooo`03o`00oooooooo00?oool0
0`000?ooooooo`05oooo000Foooo00<0003oool000000_ooo`030000oooo000000;oool00`000?oo
o`000004oooo00<0003oooooool0k_ooo`030000oooooooo00Goool001Soool01P000?ooooooo`00
0?ooo`0000;oool00`000?ooo`000004oooo0P0000?oool00`000?ooooooo`0[oooo00<0003ooooo
ool0:oooo`030000oooooooo02_oool00`000?ooooooo`0[oooo00<0003oooooool0:oooo`040000
oooooooooooo0`0000Ooool001Koool200000oooo`030000oooo000000;oool00`000?ooo`000004
oooo00<0003oooooool00_ooo`030000oooooooo00Koool00`000?ooooooo`07oooo00<0003ooooo
ool01_ooo`030000oooooooo00Koool00`000?ooooooo`06oooo00<0003oooooool01_ooo`030000
oooooooo00Ooool00`000?ooooooo`06oooo00<0003oooooool01_ooo`030000oooooooo00Koool0
0`000?ooooooo`06oooo00<0003oooooool01oooo`030000oooooooo00Koool00`000?ooooooo`06
oooo00<0003oooooool01_ooo`030000oooooooo00Koool00`000?ooooooo`06oooo00<0003ooooo
ool01oooo`030000oooooooo00Koool00`000?ooooooo`06oooo00<0003oooooool01_ooo`030000
oooooooo00Koool00`000?ooooooo`07oooo00<0003oooooool01_ooo`030000oooooooo00Koool0
0`000?ooooooo`03oooo00<0003oooooool01Oooo`005_ooo`030000oooooooo00;oool00`000?oo
o`000002oooo00<0003oool000001?oooo800007oooo000Foooo0`0000?oool00`000?ooooooo`02
oooo00<0003oooooool0o?ooo`00ooooob7oool00001\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-9.74065, -99.2777, 
  0.217628, 3.47537}}],

Cell[TextData[{
  "Again, there exist buckling levels for which the frequencies of the first \
and fourth modes are in the ratio of 1:1, 3:1, and 1:3. We use the function ",
  
  StyleBox["MMSDirect11",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " to treat these cases."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell["One-to-One Internal Resonance", "Subsubsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "When the frequencies of the first and fourth modes are approximately \
equal, we denote them by ",
  Cell[BoxData[
      \(\[Omega]\_1\)]],
  " and ",
  Cell[BoxData[
      \(\[Omega]\_2\)]],
  ", respectively, according to"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(frequencies6 = {\[Omega]\_1 -> \((\[Omega] /. \[Omega]Sol1)\), 
        \[Omega]\_2 -> \((\(\[Omega] /. \[Omega]Sol2\) /. n -> 4)\)}\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Omega]\_1 \[Rule] b\ \[Pi]\^2\ \@\[Alpha], 
      \[Omega]\_2 \[Rule] 4\ \@15\ \[Pi]\^2}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Moreover, we define ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_k\)]],
  " and the first and fourth eigenmodes of the buckled beam as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(modeshapes6 = 
      Join[modeshape, {\[CapitalPhi]\_1 -> Function[x, shape1 // Evaluate], 
          \[CapitalPhi]\_2 -> Function[x, shape4 // Evaluate]}]; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
We consider a combination of a one-to-one internal resonance between the \
first and fourth modes and a principal parametric resonance of the fourth \
mode; that is,\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ResonanceConds6 = {
        \[Omega]\_2 == \[Omega]\_1 + \[Epsilon]\^2\ \[Sigma]\_1, 
        \[CapitalOmega] == 2  \[Omega]\_2 + \[Epsilon]\^2\ \[Sigma]\_2}; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Using ",
  StyleBox["MMSDirect11",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain the solvability conditions"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(MMSDirect11[frequencies6, modeshapes6, ResonanceConds6] // Timing\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(29.5220000000008475`\ Second\), ",", 
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
              \(\(-2\)\ I\ \[Mu]\ \[Omega]\_1\ A\_1[T\_2]\), "-", 
                \(E\^\(2\ I\ T\_2\ \[Sigma]\_1 + I\ T\_2\ \[Sigma]\_2\)\ f\ 
                  \[Pi]\^2\ A\&_\_1[T\_2]\), "+", 
                \(12\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\^2\ A\&_\_1[T\_2]\), 
                "+", \(480\ E\^\(2\ I\ T\_2\ \[Sigma]\_1\)\ \[Pi]\^4\ 
                  \[Alpha]\ A\_2[T\_2]\^2\ A\&_\_1[T\_2]\), "+", 
                \(1216\/3\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\ A\_2[T\_2]\ 
                  A\&_\_2[T\_2]\), "-", 
                RowBox[{"2", " ", "I", " ", \(\[Omega]\_1\), " ", 
                  RowBox[{
                    SubsuperscriptBox["A", "1", "\[Prime]",
                      MultilineFunction->None], "[", \(T\_2\), "]"}]}]}], 
              "==", "0"}], ",", 
            RowBox[{
              RowBox[{
              \(\(-2\)\ I\ \[Mu]\ \[Omega]\_2\ A\_2[T\_2]\), "+", 
                \(1216\/3\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\ A\_2[T\_2]\ 
                  A\&_\_1[T\_2]\), "-", 
                \(32\/3\ E\^\(I\ T\_2\ \[Sigma]\_2\)\ f\ \[Pi]\^2\ 
                  A\&_\_2[T\_2]\), "+", 
                \(480\ E\^\(\(-2\)\ I\ T\_2\ \[Sigma]\_1\)\ \[Pi]\^4\ 
                  \[Alpha]\ A\_1[T\_2]\^2\ A\&_\_2[T\_2]\), "-", 
                \(1024\/3\ \[Pi]\^4\ \[Alpha]\ A\_2[T\_2]\^2\ 
                  A\&_\_2[T\_2]\), "-", 
                RowBox[{"2", " ", "I", " ", \(\[Omega]\_2\), " ", 
                  RowBox[{
                    SubsuperscriptBox["A", "2", "\[Prime]",
                      MultilineFunction->None], "[", \(T\_2\), "]"}]}]}], 
              "==", "0"}]}], "}"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
which indicates that this one-to-one internal resonance between the first and \
fourth modes is activated.\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["Three-to-One Internal Resonances", "Subsubsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "When the frequency of the fourth mode is approximately three times that of \
the first mode, we denote these frequencies by ",
  Cell[BoxData[
      \(\[Omega]\_2\)]],
  " and ",
  Cell[BoxData[
      \(\[Omega]\_1\)]],
  ", respectively, as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(frequencies7 = {\[Omega]\_1 -> \((\[Omega] /. \[Omega]Sol1)\), 
        \[Omega]\_2 -> \((\(\[Omega] /. \[Omega]Sol2\) /. n -> 4)\)}\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Omega]\_1 \[Rule] b\ \[Pi]\^2\ \@\[Alpha], 
      \[Omega]\_2 \[Rule] 4\ \@15\ \[Pi]\^2}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Moreover, we define ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_k\)]],
  " and the first and fourth eigenmodes of the buckled beam as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(modeshapes7 = 
      Join[modeshape, {\[CapitalPhi]\_1 -> Function[x, shape1 // Evaluate], 
          \[CapitalPhi]\_2 -> Function[x, shape4 // Evaluate]}]; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
We consider a combination of a three-to-one internal resonance between the \
first and fourth modes and a principal parametric resonance of the fourth \
mode; that is,\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ResonanceConds7 = {
        \[Omega]\_2 == 3  \[Omega]\_1 + \[Epsilon]\^2\ \[Sigma]\_1, 
        \[CapitalOmega] == 2  \[Omega]\_2 + \[Epsilon]\^2\ \[Sigma]\_2}; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Using ",
  StyleBox["MMSDirect11",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain the solvability conditions"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(MMSDirect11[frequencies7, modeshapes7, ResonanceConds7] // Timing\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(22.6530000000002473`\ Second\), ",", 
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
              \(\(-2\)\ I\ \[Mu]\ \[Omega]\_1\ A\_1[T\_2]\), "+", 
                \(12\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\^2\ A\&_\_1[T\_2]\), 
                "+", \(3264\/35\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\ A\_2[T\_2]\ 
                  A\&_\_2[T\_2]\), "-", 
                RowBox[{"2", " ", "I", " ", \(\[Omega]\_1\), " ", 
                  RowBox[{
                    SubsuperscriptBox["A", "1", "\[Prime]",
                      MultilineFunction->None], "[", \(T\_2\), "]"}]}]}], 
              "==", "0"}], ",", 
            RowBox[{
              RowBox[{
              \(\(-2\)\ I\ \[Mu]\ \[Omega]\_2\ A\_2[T\_2]\), "+", 
                \(3264\/35\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\ A\_2[T\_2]\ 
                  A\&_\_1[T\_2]\), "-", 
                \(288\/35\ E\^\(I\ T\_2\ \[Sigma]\_2\)\ f\ \[Pi]\^2\ 
                  A\&_\_2[T\_2]\), "-", 
                \(9216\/35\ \[Pi]\^4\ \[Alpha]\ A\_2[T\_2]\^2\ 
                  A\&_\_2[T\_2]\), "-", 
                RowBox[{"2", " ", "I", " ", \(\[Omega]\_2\), " ", 
                  RowBox[{
                    SubsuperscriptBox["A", "2", "\[Prime]",
                      MultilineFunction->None], "[", \(T\_2\), "]"}]}]}], 
              "==", "0"}]}], "}"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
which indicates that this three-to-one internal resonance between the first \
and fourth modes is not activated.\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "When the frequency of the first mode is approximately three times that of \
the fourth mode, we denote these frequencies by ",
  Cell[BoxData[
      \(\[Omega]\_2\)]],
  " and ",
  Cell[BoxData[
      \(\[Omega]\_1\)]],
  ", respectively, as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(frequencies8 = {\[Omega]\_2 -> \((\[Omega] /. \[Omega]Sol1)\), 
        \[Omega]\_1 -> \((\(\[Omega] /. \[Omega]Sol2\) /. n -> 4)\)}\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Omega]\_2 \[Rule] b\ \[Pi]\^2\ \@\[Alpha], 
      \[Omega]\_1 \[Rule] 4\ \@15\ \[Pi]\^2}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Moreover, we define ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_k\)]],
  " and the fourth and first eigenmodes of the buckled beam as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(modeshapes8 = 
      Join[modeshape, {\[CapitalPhi]\_1 -> Function[x, shape4 // Evaluate], 
          \[CapitalPhi]\_2 -> Function[x, shape1 // Evaluate]}]; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
We consider a combination of a three-to-one internal resonance between the \
first and fourth modes and a principal parametric resonance of the fourth \
mode; that is,\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ResonanceConds8 = {
        \[Omega]\_2 == 3  \[Omega]\_1 + \[Epsilon]\^2\ \[Sigma]\_1, 
        \[CapitalOmega] == 2  \[Omega]\_2 + \[Epsilon]\^2\ \[Sigma]\_2}; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Using ",
  StyleBox["MMSDirect11",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain the solvability conditions"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(MMSDirect11[frequencies8, modeshapes8, ResonanceConds8] // Timing\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(22.0720000000001181`\ Second\), ",", 
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
              \(\(-2\)\ I\ \[Mu]\ \[Omega]\_1\ A\_1[T\_2]\), "+", 
                \(1024\/5\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\^2\ 
                  A\&_\_1[T\_2]\), "-", 
                \(8896\/5\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\ A\_2[T\_2]\ 
                  A\&_\_2[T\_2]\), "-", 
                RowBox[{"2", " ", "I", " ", \(\[Omega]\_1\), " ", 
                  RowBox[{
                    SubsuperscriptBox["A", "1", "\[Prime]",
                      MultilineFunction->None], "[", \(T\_2\), "]"}]}]}], 
              "==", "0"}], ",", 
            RowBox[{
              RowBox[{
              \(\(-2\)\ I\ \[Mu]\ \[Omega]\_2\ A\_2[T\_2]\), "-", 
                \(8896\/5\ \[Pi]\^4\ \[Alpha]\ A\_1[T\_2]\ A\_2[T\_2]\ 
                  A\&_\_1[T\_2]\), "-", 
                \(E\^\(I\ T\_2\ \[Sigma]\_2\)\ f\ \[Pi]\^2\ A\&_\_2[T\_2]\), 
                "+", \(12\ \[Pi]\^4\ \[Alpha]\ A\_2[T\_2]\^2\ 
                  A\&_\_2[T\_2]\), "-", 
                RowBox[{"2", " ", "I", " ", \(\[Omega]\_2\), " ", 
                  RowBox[{
                    SubsuperscriptBox["A", "2", "\[Prime]",
                      MultilineFunction->None], "[", \(T\_2\), "]"}]}]}], 
              "==", "0"}]}], "}"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
which indicates again that this three-to-one internal resonance between the \
first and fourth modes is not activated.\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["8.2 Circular Cylindrical Shells", "Section",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
Following McIvor (1966) and Raouf and Nayfeh (1990), we write the equations \
of motion governing the dynamic response of an infinitely long cylindrical \
shell to a primary-resonance excitation of one of its two orthogonal flexural \
modes as\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(EOM = {
          w - \[Psi]\_\[Theta] + w\_\(t, t\) + 
              \[Alpha]\^2\ 
                \((w + 2\ w\_\(\[Theta], \[Theta]\) + 
                    w\_\(\[Theta], \[Theta], \[Theta], \[Theta]\))\) == 
            \(-\(w\_\[Theta]\%2\/2\)\) - w\ w\_\[Theta]\%2 - \[Psi]\_t\%2 + 
              w\ \[Psi]\_t\%2 - 2\ w\ \[Psi]\_\[Theta] + 
              1\/2\ w\_\[Theta]\%2\ \[Psi]\_\[Theta] + \[Psi]\_\[Theta]\%2 - 
              w\ \[Psi]\_\[Theta]\%2 + 
              f\ \[Epsilon]\^3\ Cos[n\ \[Theta]]\ Cos[\[CapitalOmega]\ t]\ 
                \((1 - w + \[Psi]\_\[Theta])\) - 
              w\ w\_\(\[Theta], \[Theta]\) - 
              w\^2\ w\_\(\[Theta], \[Theta]\) + 
              3\/2\ w\_\[Theta]\%2\ w\_\(\[Theta], \[Theta]\) + 
              \[Psi]\_\[Theta]\ w\_\(\[Theta], \[Theta]\) + 
              w\ \[Psi]\_\[Theta]\ w\_\(\[Theta], \[Theta]\) - 
              \[Psi]\_\[Theta]\%2\ w\_\(\[Theta], \[Theta]\) + 
              w\_\[Theta]\ \[Psi]\_\(\[Theta], \[Theta]\) + 
              w\ w\_\[Theta]\ \[Psi]\_\(\[Theta], \[Theta]\) - 
              2\ w\_\[Theta]\ \[Psi]\_\[Theta]\ 
                \[Psi]\_\(\[Theta], \[Theta]\) + 
              \[Alpha]\^2\ 
                \((\(-3\)\ w\^2 - \(11\ w\_\[Theta]\%2\)\/2 - 
                    11\ w\ w\_\(\[Theta], \[Theta]\) + 
                    4\ \[Psi]\_\[Theta]\ w\_\(\[Theta], \[Theta]\) - 
                    6\ w\_\(\[Theta], \[Theta]\)\%2 + 
                    4\ w\_\[Theta]\ \[Psi]\_\(\[Theta], \[Theta]\) - 
                    8\ w\_\[Theta]\ w\_\(\[Theta], \[Theta], \[Theta]\) + 
                    8\ \[Psi]\_\(\[Theta], \[Theta]\)\ 
                      w\_\(\[Theta], \[Theta], \[Theta]\) + 
                    w\ \[Psi]\_\(\[Theta], \[Theta], \[Theta]\) + 
                    5\ w\_\(\[Theta], \[Theta]\)\ 
                      \[Psi]\_\(\[Theta], \[Theta], \[Theta]\) - 
                    4\ w\ w\_\(\[Theta], \[Theta], \[Theta], \[Theta]\) + 
                    4\ \[Psi]\_\[Theta]\ 
                      w\_\(\[Theta], \[Theta], \[Theta], \[Theta]\) + 
                    w\_\[Theta]\ 
                      \[Psi]\_\(\[Theta], \[Theta], \[Theta], \[Theta]\))\), 
          w\_\[Theta] + \[Psi]\_\(t, t\) - \[Psi]\_\(\[Theta], \[Theta]\) == 
            2\ w\ w\_\[Theta] + w\_\[Theta]\%3\/2 + 2\ w\_t\ \[Psi]\_t - 
              2\ w\ w\_t\ \[Psi]\_t - 2\ w\_\[Theta]\ \[Psi]\_\[Theta] + 
              2\ w\ w\_\[Theta]\ \[Psi]\_\[Theta] + 
              w\_\[Theta]\ w\_\(\[Theta], \[Theta]\) + 
              w\ w\_\[Theta]\ w\_\(\[Theta], \[Theta]\) - 
              2\ w\_\[Theta]\ \[Psi]\_\[Theta]\ w\_\(\[Theta], \[Theta]\) + 
              2\ w\ \[Psi]\_\(t, t\) - w\^2\ \[Psi]\_\(t, t\) - 
              2\ w\ \[Psi]\_\(\[Theta], \[Theta]\) + 
              w\^2\ \[Psi]\_\(\[Theta], \[Theta]\) - 
              w\_\[Theta]\%2\ \[Psi]\_\(\[Theta], \[Theta]\) + 
              \[Alpha]\^2\ 
                \((w\_\[Theta]\ w\_\(\[Theta], \[Theta]\) - 
                    w\ w\_\(\[Theta], \[Theta], \[Theta]\) - 
                    w\_\(\[Theta], \[Theta]\)\ 
                      w\_\(\[Theta], \[Theta], \[Theta]\) + 
                    w\_\[Theta]\ 
                      w\_\(\[Theta], \[Theta], \[Theta], \[Theta]\))\)} /. {
          w -> w[\[Theta], t], 
          \((s : w | \[Psi])\)\_m__ :> \[PartialD]\_m s[\[Theta], t]}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "We seek a second-order uniform asymptotic expansion of the solution of ",
  StyleBox["EOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in the form"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(solRule = {
        w -> \((Evaluate[Sum[\[Epsilon]\^j\ w\_j[#1, #2, #3, #4], {j, 3}]]\ &)
            \), \[Psi] -> 
          \((Evaluate[Sum[\[Epsilon]\^j\ \[Psi]\_j[#1, #2, #3, #4], {j, 3}]]
                \ &)\)}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({w \[Rule] 
        \((\[Epsilon]\ w\_1[#1, #2, #3, #4] + 
              \[Epsilon]\^2\ w\_2[#1, #2, #3, #4] + 
              \[Epsilon]\^3\ w\_3[#1, #2, #3, #4]&)\), 
      \[Psi] \[Rule] 
        \((\[Epsilon]\ \[Psi]\_1[#1, #2, #3, #4] + 
              \[Epsilon]\^2\ \[Psi]\_2[#1, #2, #3, #4] + 
              \[Epsilon]\^3\ \[Psi]\_3[#1, #2, #3, #4]&)\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`#1\)]],
  " stands for ",
  Cell[BoxData[
      \(TraditionalForm\`\[Theta]\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`#2\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`#3\)]],
  ", and ",
  Cell[BoxData[
      \(TraditionalForm\`#4\)]],
  " stand for ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0, T\_1\)]],
  ", and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_2\)]],
  ", respectively. Transforming the time derivatives in ",
  StyleBox["EOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in terms of the scales ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`T\_1\)]],
  ", and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_2\)]],
  ", substituting ",
  StyleBox["solRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["EOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", expanding the result for small \[Epsilon], and discarding terms of order \
higher than ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\^3\)]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(eq82a = 
      \((\(EOM /. {w[x_, t] -> w[x, T\_0, T\_1, T\_2], 
                  \[Psi][x_, t] -> \[Psi][x, T\_0, T\_1, T\_2], 
                  \(\(Derivative[m_, n_]\)[w_]\)[x_, t] -> 
                    \(dt[n]\)[D[w[x, T\_0, T\_1, T\_2], {x, m}]], t -> T\_0}
                  \) /. solRule // ExpandAll)\) /. 
        \[Epsilon]\^\(n_ /; n > 3\) -> 0; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[
"Equating coefficients of like powers of \[Epsilon], we have"], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eqEps = 
      \(Rest[Thread[CoefficientList[Subtract\ @@\ #, \[Epsilon]] == 0]]\ &
            \)\ /@\ eq82a // Transpose; eqEps /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
                RowBox[{"2", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{\(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'", "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                InterpretationBox[
                  RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                    SubsuperscriptBox["\[Psi]", "1", 
                      InterpretationBox["\<\"'\"\>",
                        SequenceForm[ "'"],
                        Editable->False]]}],
                  SequenceForm[ "", 
                    Power[ 
                      Subscript[ \[Psi], 1], 
                      SequenceForm[ "'"]]],
                  Editable->False], "+", 
                InterpretationBox[
                  RowBox[{\(D\_0\%2\), "\[InvisibleSpace]", 
                    SubsuperscriptBox["w", "1", 
                      InterpretationBox["\[Null]",
                        SequenceForm[ ],
                        Editable->False]]}],
                  SequenceForm[ 
                    Power[ 
                      Subscript[ D, 0], 2], 
                    Power[ 
                      Subscript[ w, 1], 
                      SequenceForm[ ]]],
                  Editable->False], "+", \(w\_1\), "+", 
                \(\[Alpha]\^2\ w\_1\)}], "==", "0"}], ",", 
            RowBox[{
              RowBox[{
                InterpretationBox[
                  RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                    SubsuperscriptBox["w", "1", 
                      InterpretationBox["\<\"'\"\>",
                        SequenceForm[ "'"],
                        Editable->False]]}],
                  SequenceForm[ "", 
                    Power[ 
                      Subscript[ w, 1], 
                      SequenceForm[ "'"]]],
                  Editable->False], "-", 
                InterpretationBox[
                  RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                    SubsuperscriptBox["\[Psi]", "1", 
                      InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                        SequenceForm[ "'", "'"],
                        Editable->False]]}],
                  SequenceForm[ "", 
                    Power[ 
                      Subscript[ \[Psi], 1], 
                      SequenceForm[ "'", "'"]]],
                  Editable->False], "+", 
                InterpretationBox[
                  RowBox[{\(D\_0\%2\), "\[InvisibleSpace]", 
                    SubsuperscriptBox["\[Psi]", "1", 
                      InterpretationBox["\[Null]",
                        SequenceForm[ ],
                        Editable->False]]}],
                  SequenceForm[ 
                    Power[ 
                      Subscript[ D, 0], 2], 
                    Power[ 
                      Subscript[ \[Psi], 1], 
                      SequenceForm[ ]]],
                  Editable->False]}], "==", "0"}]}], "}"}], ",", 
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
                RowBox[{\(1\/2\), " ", 
                  SuperscriptBox[
                    RowBox[{"(", 
                      InterpretationBox[
                        RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                          SubsuperscriptBox["w", "1", 
                            InterpretationBox["\<\"'\"\>",
                              SequenceForm[ "'"],
                              Editable->False]]}],
                        SequenceForm[ "", 
                          Power[ 
                            Subscript[ w, 1], 
                            SequenceForm[ "'"]]],
                        Editable->False], ")"}], "2"]}], "+", 
                RowBox[{\(11\/2\), " ", \(\[Alpha]\^2\), " ", 
                  SuperscriptBox[
                    RowBox[{"(", 
                      InterpretationBox[
                        RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                          SubsuperscriptBox["w", "1", 
                            InterpretationBox["\<\"'\"\>",
                              SequenceForm[ "'"],
                              Editable->False]]}],
                        SequenceForm[ "", 
                          Power[ 
                            Subscript[ w, 1], 
                            SequenceForm[ "'"]]],
                        Editable->False], ")"}], "2"]}], "+", 
                RowBox[{"6", " ", \(\[Alpha]\^2\), " ", 
                  SuperscriptBox[
                    RowBox[{"(", 
                      InterpretationBox[
                        RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                          SubsuperscriptBox["w", "1", 
                            InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                              SequenceForm[ "'", "'"],
                              Editable->False]]}],
                        SequenceForm[ "", 
                          Power[ 
                            Subscript[ w, 1], 
                            SequenceForm[ "'", "'"]]],
                        Editable->False], ")"}], "2"]}], "+", 
                RowBox[{"8", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'", "'"]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{"2", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{\(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'", "'", "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"4", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"4", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'", "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}]}], "-", 
                SuperscriptBox[
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], "2"], "-", 
                RowBox[{
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"4", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"8", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"5", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'", "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{\(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'", "'", "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                InterpretationBox[
                  RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                    SubsuperscriptBox["\[Psi]", "2", 
                      InterpretationBox["\<\"'\"\>",
                        SequenceForm[ "'"],
                        Editable->False]]}],
                  SequenceForm[ "", 
                    Power[ 
                      Subscript[ \[Psi], 2], 
                      SequenceForm[ "'"]]],
                  Editable->False], "+", 
                SuperscriptBox[
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Subscript[ D, 0], 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}], "2"], "+", 
                InterpretationBox[
                  RowBox[{\(D\_0\%2\), "\[InvisibleSpace]", 
                    SubsuperscriptBox["w", "2", 
                      InterpretationBox["\[Null]",
                        SequenceForm[ ],
                        Editable->False]]}],
                  SequenceForm[ 
                    Power[ 
                      Subscript[ D, 0], 2], 
                    Power[ 
                      Subscript[ w, 2], 
                      SequenceForm[ ]]],
                  Editable->False], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\ D\_1\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Times[ 
                          Subscript[ D, 0], 
                          Subscript[ D, 1]], 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "+", 
                RowBox[{"11", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "+", 
                RowBox[{"4", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'", "'", "'"]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "-", 
                RowBox[{\(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'", "'", "'"]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "+", 
                \(3\ \[Alpha]\^2\ w\_1\%2\), "+", \(w\_2\), "+", 
                \(\[Alpha]\^2\ w\_2\)}], "==", "0"}], ",", 
            RowBox[{
              RowBox[{
                RowBox[{
                  RowBox[{"-", 
                    RowBox[{"(", 
                      InterpretationBox[
                        RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                          SubsuperscriptBox["w", "1", 
                            InterpretationBox["\<\"'\"\>",
                              SequenceForm[ "'"],
                              Editable->False]]}],
                        SequenceForm[ "", 
                          Power[ 
                            Subscript[ w, 1], 
                            SequenceForm[ "'"]]],
                        Editable->False], ")"}]}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{\(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{\(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{\(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'", "'", "'"]]],
                      Editable->False], ")"}]}], "+", 
                InterpretationBox[
                  RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                    SubsuperscriptBox["w", "2", 
                      InterpretationBox["\<\"'\"\>",
                        SequenceForm[ "'"],
                        Editable->False]]}],
                  SequenceForm[ "", 
                    Power[ 
                      Subscript[ w, 2], 
                      SequenceForm[ "'"]]],
                  Editable->False], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}]}], "-", 
                InterpretationBox[
                  RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                    SubsuperscriptBox["\[Psi]", "2", 
                      InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                        SequenceForm[ "'", "'"],
                        Editable->False]]}],
                  SequenceForm[ "", 
                    Power[ 
                      Subscript[ \[Psi], 2], 
                      SequenceForm[ "'", "'"]]],
                  Editable->False], "-", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Subscript[ D, 0], 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Subscript[ D, 0], 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}]}], "+", 
                InterpretationBox[
                  RowBox[{\(D\_0\%2\), "\[InvisibleSpace]", 
                    SubsuperscriptBox["\[Psi]", "2", 
                      InterpretationBox["\[Null]",
                        SequenceForm[ ],
                        Editable->False]]}],
                  SequenceForm[ 
                    Power[ 
                      Subscript[ D, 0], 2], 
                    Power[ 
                      Subscript[ \[Psi], 2], 
                      SequenceForm[ ]]],
                  Editable->False], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\ D\_1\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Times[ 
                          Subscript[ D, 0], 
                          Subscript[ D, 1]], 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "+", 
                RowBox[{\(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'", "'"]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "-", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\%2\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Power[ 
                          Subscript[ D, 0], 2], 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}], " ", \(w\_1\)}]}], "==", 
              "0"}]}], "}"}], ",", 
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
              \(\(-f\)\ Cos[n\ \[Theta]]\ Cos[T\_0\ \[CapitalOmega]]\), "-", 
                RowBox[{\(3\/2\), " ", 
                  SuperscriptBox[
                    RowBox[{"(", 
                      InterpretationBox[
                        RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                          SubsuperscriptBox["w", "1", 
                            InterpretationBox["\<\"'\"\>",
                              SequenceForm[ "'"],
                              Editable->False]]}],
                        SequenceForm[ "", 
                          Power[ 
                            Subscript[ w, 1], 
                            SequenceForm[ "'"]]],
                        Editable->False], ")"}], "2"], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{"11", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{"8", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{"12", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{"8", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'", "'", "'"]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{"2", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "3", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 3], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{\(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "3", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 3], 
                          SequenceForm[ "'", "'", "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{\(1\/2\), " ", 
                  SuperscriptBox[
                    RowBox[{"(", 
                      InterpretationBox[
                        RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                          SubsuperscriptBox["w", "1", 
                            InterpretationBox["\<\"'\"\>",
                              SequenceForm[ "'"],
                              Editable->False]]}],
                        SequenceForm[ "", 
                          Power[ 
                            Subscript[ w, 1], 
                            SequenceForm[ "'"]]],
                        Editable->False], ")"}], "2"], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"4", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"4", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'", "'", "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  SuperscriptBox[
                    RowBox[{"(", 
                      InterpretationBox[
                        RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                          SubsuperscriptBox["\[Psi]", "1", 
                            InterpretationBox["\<\"'\"\>",
                              SequenceForm[ "'"],
                              Editable->False]]}],
                        SequenceForm[ "", 
                          Power[ 
                            Subscript[ \[Psi], 1], 
                            SequenceForm[ "'"]]],
                        Editable->False], ")"}], "2"]}], "-", 
                RowBox[{
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"4", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"8", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'", "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"5", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'", "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{\(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'", "'", "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "2", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 2], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"4", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "2", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 2], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"4", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'", "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "2", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 2], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "2", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 2], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "2", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 2], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"4", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "2", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 2], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"8", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "2", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 2], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"5", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "2", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 2], 
                          SequenceForm[ "'", "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{\(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "2", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 2], 
                          SequenceForm[ "'", "'", "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                InterpretationBox[
                  RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                    SubsuperscriptBox["\[Psi]", "3", 
                      InterpretationBox["\<\"'\"\>",
                        SequenceForm[ "'"],
                        Editable->False]]}],
                  SequenceForm[ "", 
                    Power[ 
                      Subscript[ \[Psi], 3], 
                      SequenceForm[ "'"]]],
                  Editable->False], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Subscript[ D, 0], 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "2", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Subscript[ D, 0], 
                        Power[ 
                          Subscript[ \[Psi], 2], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}]}], "+", 
                InterpretationBox[
                  RowBox[{\(D\_0\%2\), "\[InvisibleSpace]", 
                    SubsuperscriptBox["w", "3", 
                      InterpretationBox["\[Null]",
                        SequenceForm[ ],
                        Editable->False]]}],
                  SequenceForm[ 
                    Power[ 
                      Subscript[ D, 0], 2], 
                    Power[ 
                      Subscript[ w, 3], 
                      SequenceForm[ ]]],
                  Editable->False], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Subscript[ D, 0], 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_1\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Subscript[ D, 1], 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\ D\_1\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Times[ 
                          Subscript[ D, 0], 
                          Subscript[ D, 1]], 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}]}], "+", 
                InterpretationBox[
                  RowBox[{\(D\_1\%2\), "\[InvisibleSpace]", 
                    SubsuperscriptBox["w", "1", 
                      InterpretationBox["\[Null]",
                        SequenceForm[ ],
                        Editable->False]]}],
                  SequenceForm[ 
                    Power[ 
                      Subscript[ D, 1], 2], 
                    Power[ 
                      Subscript[ w, 1], 
                      SequenceForm[ ]]],
                  Editable->False], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\ D\_2\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Times[ 
                          Subscript[ D, 0], 
                          Subscript[ D, 2]], 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{"(", 
                      InterpretationBox[
                        RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                          SubsuperscriptBox["w", "1", 
                            InterpretationBox["\<\"'\"\>",
                              SequenceForm[ "'"],
                              Editable->False]]}],
                        SequenceForm[ "", 
                          Power[ 
                            Subscript[ w, 1], 
                            SequenceForm[ "'"]]],
                        Editable->False], ")"}], "2"], " ", \(w\_1\)}], "+", 
                RowBox[{
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "+", 
                RowBox[{"11", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "+", 
                RowBox[{"4", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'", "'", "'", "'"]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "-", 
                RowBox[{
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "+", 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{"(", 
                      InterpretationBox[
                        RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                          SubsuperscriptBox["\[Psi]", "1", 
                            InterpretationBox["\<\"'\"\>",
                              SequenceForm[ "'"],
                              Editable->False]]}],
                        SequenceForm[ "", 
                          Power[ 
                            Subscript[ \[Psi], 1], 
                            SequenceForm[ "'"]]],
                        Editable->False], ")"}], "2"], " ", \(w\_1\)}], "-", 
                RowBox[{
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "2", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 2], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "-", 
                RowBox[{\(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "2", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 2], 
                          SequenceForm[ "'", "'", "'"]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "-", 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{"(", 
                      InterpretationBox[
                        RowBox[{\(D\_0\), "\[InvisibleSpace]", 
                          SubsuperscriptBox["\[Psi]", "1", 
                            InterpretationBox["\[Null]",
                              SequenceForm[ ],
                              Editable->False]]}],
                        SequenceForm[ 
                          Subscript[ D, 0], 
                          Power[ 
                            Subscript[ \[Psi], 1], 
                            SequenceForm[ ]]],
                        Editable->False], ")"}], "2"], " ", \(w\_1\)}], "+", 
                RowBox[{
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", \(w\_1\%2\)}], "+", 
                RowBox[{
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", \(w\_2\)}], "+", 
                RowBox[{"11", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", \(w\_2\)}], "+", 
                RowBox[{"4", " ", \(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'", "'", "'"]]],
                      Editable->False], ")"}], " ", \(w\_2\)}], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", \(w\_2\)}], "-", 
                RowBox[{\(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'", "'", "'"]]],
                      Editable->False], ")"}], " ", \(w\_2\)}], "+", 
                \(6\ \[Alpha]\^2\ w\_1\ w\_2\), "+", \(w\_3\), "+", 
                \(\[Alpha]\^2\ w\_3\)}], "==", "0"}], ",", 
            RowBox[{
              RowBox[{
                RowBox[{\(-\(1\/2\)\), " ", 
                  SuperscriptBox[
                    RowBox[{"(", 
                      InterpretationBox[
                        RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                          SubsuperscriptBox["w", "1", 
                            InterpretationBox["\<\"'\"\>",
                              SequenceForm[ "'"],
                              Editable->False]]}],
                        SequenceForm[ "", 
                          Power[ 
                            Subscript[ w, 1], 
                            SequenceForm[ "'"]]],
                        Editable->False], ")"}], "3"]}], "-", 
                RowBox[{
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{\(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{\(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'", "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{\(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{\(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{\(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'", "'", "'"]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{\(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\
\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'", "'", "'", "'"]]],
                      Editable->False], ")"}]}], "+", 
                InterpretationBox[
                  RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                    SubsuperscriptBox["w", "3", 
                      InterpretationBox["\<\"'\"\>",
                        SequenceForm[ "'"],
                        Editable->False]]}],
                  SequenceForm[ "", 
                    Power[ 
                      Subscript[ w, 3], 
                      SequenceForm[ "'"]]],
                  Editable->False], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{
                  SuperscriptBox[
                    RowBox[{"(", 
                      InterpretationBox[
                        RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                          SubsuperscriptBox["w", "1", 
                            InterpretationBox["\<\"'\"\>",
                              SequenceForm[ "'"],
                              Editable->False]]}],
                        SequenceForm[ "", 
                          Power[ 
                            Subscript[ w, 1], 
                            SequenceForm[ "'"]]],
                        Editable->False], ")"}], "2"], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "2", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 2], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}]}], "-", 
                InterpretationBox[
                  RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                    SubsuperscriptBox["\[Psi]", "3", 
                      InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                        SequenceForm[ "'", "'"],
                        Editable->False]]}],
                  SequenceForm[ "", 
                    Power[ 
                      Subscript[ \[Psi], 3], 
                      SequenceForm[ "'", "'"]]],
                  Editable->False], "-", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Subscript[ D, 0], 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Subscript[ D, 0], 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Subscript[ D, 0], 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "2", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Subscript[ D, 0], 
                        Power[ 
                          Subscript[ \[Psi], 2], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}]}], "+", 
                InterpretationBox[
                  RowBox[{\(D\_0\%2\), "\[InvisibleSpace]", 
                    SubsuperscriptBox["\[Psi]", "3", 
                      InterpretationBox["\[Null]",
                        SequenceForm[ ],
                        Editable->False]]}],
                  SequenceForm[ 
                    Power[ 
                      Subscript[ D, 0], 2], 
                    Power[ 
                      Subscript[ \[Psi], 3], 
                      SequenceForm[ ]]],
                  Editable->False], "-", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Subscript[ D, 0], 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_1\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Subscript[ D, 1], 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Subscript[ D, 0], 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_1\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Subscript[ D, 1], 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\ D\_1\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "2", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Times[ 
                          Subscript[ D, 0], 
                          Subscript[ D, 1]], 
                        Power[ 
                          Subscript[ \[Psi], 2], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}]}], "+", 
                InterpretationBox[
                  RowBox[{\(D\_1\%2\), "\[InvisibleSpace]", 
                    SubsuperscriptBox["\[Psi]", "1", 
                      InterpretationBox["\[Null]",
                        SequenceForm[ ],
                        Editable->False]]}],
                  SequenceForm[ 
                    Power[ 
                      Subscript[ D, 1], 2], 
                    Power[ 
                      Subscript[ \[Psi], 1], 
                      SequenceForm[ ]]],
                  Editable->False], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\ D\_2\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Times[ 
                          Subscript[ D, 0], 
                          Subscript[ D, 2]], 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "-", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "+", 
                RowBox[{\(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "2", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 2], 
                          SequenceForm[ "'", "'", "'"]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "-", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "2", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 2], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Subscript[ D, 0], 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Subscript[ D, 0], 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "-", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\%2\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "2", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Power[ 
                          Subscript[ D, 0], 2], 
                        Power[ 
                          Subscript[ \[Psi], 2], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "-", 
                RowBox[{"4", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\ D\_1\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Times[ 
                          Subscript[ D, 0], 
                          Subscript[ D, 1]], 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}], " ", \(w\_1\)}], "-", 
                RowBox[{
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", \(w\_1\%2\)}], "+", 
                RowBox[{
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\%2\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Power[ 
                          Subscript[ D, 0], 2], 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}], " ", \(w\_1\%2\)}], "-", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          InterpretationBox["\<\"'\"\>",
                            SequenceForm[ "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'"]]],
                      Editable->False], ")"}], " ", \(w\_2\)}], "+", 
                RowBox[{\(\[Alpha]\^2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["w", "1", 
                          
                          InterpretationBox[
                            \("'"\[InvisibleSpace]"'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ w, 1], 
                          SequenceForm[ "'", "'", "'"]]],
                      Editable->False], ")"}], " ", \(w\_2\)}], "+", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{"\<\"\"\>", "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox[\("'"\[InvisibleSpace]"'"\),
                            SequenceForm[ "'", "'"],
                            Editable->False]]}],
                      SequenceForm[ "", 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ "'", "'"]]],
                      Editable->False], ")"}], " ", \(w\_2\)}], "-", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    InterpretationBox[
                      RowBox[{\(D\_0\%2\), "\[InvisibleSpace]", 
                        SubsuperscriptBox["\[Psi]", "1", 
                          InterpretationBox["\[Null]",
                            SequenceForm[ ],
                            Editable->False]]}],
                      SequenceForm[ 
                        Power[ 
                          Subscript[ D, 0], 2], 
                        Power[ 
                          Subscript[ \[Psi], 1], 
                          SequenceForm[ ]]],
                      Editable->False], ")"}], " ", \(w\_2\)}]}], "==", 
              "0"}]}], "}"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["8.2.1 First-Order Solution", "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "To determine the solution of the first-order problem, ",
  StyleBox["eqEps[[1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we use the method of separation of variables and let"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(svRule = {
        w\_1 -> \((A\ E\^\(I\ m\ #1\)\ E\^\(I\ \[Omega]\_m\ #2\)\ &)\), 
        \[Psi]\_1 -> \((B\ E\^\(I\ m\ #1\)\ E\^\(I\ \[Omega]\_m\ #2\)\ &)\)}; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting the ",
  StyleBox["svRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["eqEps[[1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain the characteristic equations"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eqEps1lhs = 
      \(Expand[E\^\(\(-I\)\ m\ \[Theta] - I\ \[Omega]\_m\ T\_0\)\ #[
                  \([1]\)] /. svRule]\ &\)\ /@\ eqEps[\([1]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({A - I\ B\ m + A\ \[Alpha]\^2 - 2\ A\ m\^2\ \[Alpha]\^2 + 
        A\ m\^4\ \[Alpha]\^2 - A\ \[Omega]\_m\%2, 
      I\ A\ m + B\ m\^2 - B\ \[Omega]\_m\%2}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Setting the determinant of the coefficient matrix in ",
  StyleBox["eqEps1lhs",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " equal to zero yields the following equation governing the natural \
frequencies ",
  Cell[BoxData[
      \(TraditionalForm\`\[Omega]\_m\)]],
  " of the shell:"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((\(Outer[D, eqEps1lhs, {A, B}] // Det\) // Collect[#, \[Omega]\_m]\ &)
        \) == 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(m\^2\ \[Alpha]\^2 - 2\ m\^4\ \[Alpha]\^2 + m\^6\ \[Alpha]\^2 + 
        \((\(-1\) - m\^2 - \[Alpha]\^2 + 2\ m\^2\ \[Alpha]\^2 - 
              m\^4\ \[Alpha]\^2)\)\ \[Omega]\_m\%2 + \[Omega]\_m\%4 == 0\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "It follows from ",
  StyleBox["eqEps1lhs",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " that the ratio ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalGamma]\_m = I  B\/A\)]],
  " of the amplitudes of ",
  Cell[BoxData[
      \(TraditionalForm\`\[Psi]\_1\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`w\_1\)]],
  " is given by"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\[CapitalGamma]Rule = 
      \[CapitalGamma]\_m -> 
        Factor[I  B\/A /. \(Solve[eqEps1lhs[\([1]\)] == 0, B]\)[\([1]\)]]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\[CapitalGamma]\_m \[Rule] 
      \(1 + \[Alpha]\^2 - 2\ m\^2\ \[Alpha]\^2 + m\^4\ \[Alpha]\^2 - 
          \[Omega]\_m\%2\)\/m\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Hence, the solution of ",
  StyleBox["eqEps[[1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " can be expressed in terms of the linear free-vibration modes"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(modeShapes = 
      \(ComplexExpand[{Re[#], Im[#]}]\ &\)\ /@
        \ {Exp[I\ n\ \[Theta]], 
          \(-I\)\ \[CapitalGamma]\_n\ Exp[I\ n\ \[Theta]]}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({{Cos[n\ \[Theta]], Sin[n\ \[Theta]]}, {
        Sin[n\ \[Theta]]\ \[CapitalGamma]\_n, 
        \(-Cos[n\ \[Theta]]\)\ \[CapitalGamma]\_n}}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["as", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol1Form = 
      modeShapes . 
          Table[A\_\(i, n\)[T\_1, T\_2]\ Exp[I\ \[Omega]\_n\ T\_0] + 
              A\&_\_\(i, n\)[T\_1, T\_2]\ Exp[\(-I\)\ \[Omega]\_n\ T\_0], {i, 
              2}] // Expand\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({E\^\(I\ T\_0\ \[Omega]\_n\)\ Cos[n\ \[Theta]]\ 
          A\_\(1, n\)[T\_1, T\_2] + 
        E\^\(I\ T\_0\ \[Omega]\_n\)\ Sin[n\ \[Theta]]\ 
          A\_\(2, n\)[T\_1, T\_2] + 
        E\^\(\(-I\)\ T\_0\ \[Omega]\_n\)\ Cos[n\ \[Theta]]\ 
          A\&_\_\(1, n\)[T\_1, T\_2] + 
        E\^\(\(-I\)\ T\_0\ \[Omega]\_n\)\ Sin[n\ \[Theta]]\ 
          A\&_\_\(2, n\)[T\_1, T\_2], 
      E\^\(I\ T\_0\ \[Omega]\_n\)\ Sin[n\ \[Theta]]\ \[CapitalGamma]\_n\ 
          A\_\(1, n\)[T\_1, T\_2] - 
        E\^\(I\ T\_0\ \[Omega]\_n\)\ Cos[n\ \[Theta]]\ \[CapitalGamma]\_n\ 
          A\_\(2, n\)[T\_1, T\_2] + 
        E\^\(\(-I\)\ T\_0\ \[Omega]\_n\)\ Sin[n\ \[Theta]]\ 
          \[CapitalGamma]\_n\ A\&_\_\(1, n\)[T\_1, T\_2] - 
        E\^\(\(-I\)\ T\_0\ \[Omega]\_n\)\ Cos[n\ \[Theta]]\ 
          \[CapitalGamma]\_n\ A\&_\_\(2, n\)[T\_1, T\_2]}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "It follows from the ",
  StyleBox["sol1Form",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " that all of the flexural modes are degenerate because two linearly \
independent eigenfunctions, namely ",
  Cell[BoxData[
      FormBox[
        StyleBox[
          RowBox[{"Cos", "[", 
            RowBox[{
              StyleBox["n",
                FontColor->RGBColor[0, 0, 1]], 
              StyleBox[" ",
                FontColor->RGBColor[0, 0, 1]], "\[Theta]"}], "]"}],
          FontColor->RGBColor[0, 0, 1]], TraditionalForm]]],
  " and ",
  Cell[BoxData[
      FormBox[
        StyleBox[
          RowBox[{"Sin", "[", 
            RowBox[{
              StyleBox["n",
                FontColor->RGBColor[0, 0, 1]], 
              StyleBox[" ",
                FontColor->RGBColor[0, 0, 1]], "\[Theta]"}], "]"}],
          FontColor->RGBColor[0, 0, 1]], TraditionalForm]]],
  ", correspond to the same frequency ",
  Cell[BoxData[
      \(TraditionalForm\`\[Omega]\_n\)]],
  " for each ",
  StyleBox["n",
    FontSlant->"Italic"],
  "."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "We consider the case of a primary-resonance excitation of the ",
  Cell[BoxData[
      FormBox[
        StyleBox[
          RowBox[{"Cos", "[", 
            RowBox[{
              StyleBox["n",
                FontColor->RGBColor[0, 0, 1]], 
              StyleBox[" ",
                FontColor->RGBColor[0, 0, 1]], "\[Theta]"}], "]"}],
          FontColor->RGBColor[0, 0, 1]], TraditionalForm]]],
  " mode corresponding to the frequency ",
  Cell[BoxData[
      \(TraditionalForm\`\[Omega]\_n\)]],
  "; it is called the ",
  StyleBox["driven mode",
    FontColor->RGBColor[0, 0, 1]],
  ". Because of the degeneracy, this mode is involved in a one-to-one \
internal resonance with the ",
  Cell[BoxData[
      FormBox[
        StyleBox[\(Sin[n\ \[Theta]]\),
          FontColor->RGBColor[0, 0, 1]], TraditionalForm]]],
  " mode, which is called the ",
  StyleBox["companion mode",
    FontColor->RGBColor[0, 0, 1]],
  ". We assume that neither of these two modes is involved in an internal \
resonance with any other mode. Therefore, in the presence of viscous damping, \
out of the infinite number of modes present in ",
  Cell[BoxData[
      \(TraditionalForm\`w\_1\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Psi]\_1\)]],
  ", only the driven mode ",
  Cell[BoxData[
      \(TraditionalForm\`Cos[n\ \[Theta]]\)]],
  " and its companion mode ",
  Cell[BoxData[
      \(TraditionalForm\`Sin[n\ \[Theta]]\)]],
  " will contribute to the steady-state response (Nayfeh and Mook, 1979). \
Consequently, we take the solution of the first-order problem to consist only \
of these two modes and express it as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(sol1 = {
        w\_1 -> Function[{\[Theta], T\_0, T\_1, T\_2}, 
            sol1Form[\([1]\)] // Evaluate], 
        \[Psi]\_1 -> 
          Function[{\[Theta], T\_0, T\_1, T\_2}, 
            sol1Form[\([2]\)] // Evaluate]}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[
"To describe the nearness of the primary resonance, we introduce the detuning \
parameter \[Sigma] defined by"], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ResonanceCond = {
        \[CapitalOmega] == \[Omega]\_n + \[Epsilon]\^2\ \[Sigma]}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where the detuning is ordered at ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\^2\)]],
  " because secular terms appear first at order ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\^3\)]],
  "."
}], "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["8.2.2 Second-Order Solution", "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting the ",
  StyleBox["sol1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  StyleBox[" ",
    FontColor->RGBColor[1, 0, 1]],
  "into the second-order problem, ",
  StyleBox["eqEps[[2]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(order2Eq = 
      Table[\((eqEps[\([1, i, 1]\)] /. {w\_1 -> w\_2, \[Psi]\_1 -> \[Psi]\_2})
            \) == \((
              eqEps[\([1, i, 1]\)] /. {w\_1 -> w\_2, \[Psi]\_1 -> \[Psi]\_2})
              \) - \((Subtract\ @@\ eqEps[\([2, i]\)] /. sol1 // Expand)\), {
          i, 2}]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Collecting the terms that may lead to secular terms, the terms \
proportional to ",
  Cell[BoxData[
      \(E\^\(I\ \[Omega]\_n\ T\_0\)\)]],
  ", we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(ST1 = 
      \(Coefficient[#[\([2]\)], E\^\(I\ \[Omega]\_n\ T\_0\)]\ &\)\ /@\ 
        order2Eq\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
          \(-2\), " ", "I", " ", \(Cos[n\ \[Theta]]\), " ", \(\[Omega]\_n\), 
            " ", 
            RowBox[{
              SubsuperscriptBox["A", \(1, n\), 
                TagBox[\((1, 0)\),
                  Derivative],
                MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}]}], "-", 
          RowBox[{
          "2", " ", "I", " ", \(Sin[n\ \[Theta]]\), " ", \(\[Omega]\_n\), 
            " ", 
            RowBox[{
              SubsuperscriptBox["A", \(2, n\), 
                TagBox[\((1, 0)\),
                  Derivative],
                MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}]}]}], 
        ",", 
        RowBox[{
          RowBox[{
          \(-2\), " ", "I", " ", \(Sin[n\ \[Theta]]\), " ", 
            \(\[CapitalGamma]\_n\), " ", \(\[Omega]\_n\), " ", 
            RowBox[{
              SubsuperscriptBox["A", \(1, n\), 
                TagBox[\((1, 0)\),
                  Derivative],
                MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}]}], "+", 
          RowBox[{
          "2", " ", "I", " ", \(Cos[n\ \[Theta]]\), " ", 
            \(\[CapitalGamma]\_n\), " ", \(\[Omega]\_n\), " ", 
            RowBox[{
              SubsuperscriptBox["A", \(2, n\), 
                TagBox[\((1, 0)\),
                  Derivative],
                MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}]}]}]}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Because the homogeneous parts of ",
  StyleBox["order2Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  StyleBox[" ",
    FontColor->RGBColor[1, 0, 1]],
  "have a nontrivial solution, the corresponding nonhomogeneous equations \
have a solution only if solvability conditions are satisfied. These \
conditions demand that ",
  StyleBox["ST1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " be orthogonal to every solution of the adjoint homogeneous equations. In \
this case, these equations are self-adjoint and hence solutions of the \
adjoint equations are given by the transpose of the ",
  StyleBox["modeShapes",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ". Demanding that ",
  StyleBox["ST1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  StyleBox[" ",
    FontColor->RGBColor[1, 0, 1]],
  "be orthogonal to the transpose of the ",
  StyleBox["modeShapes",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain the solvability conditions"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"SCond1", "=", 
      RowBox[{
        RowBox[{"Solve", "[", 
          RowBox[{
          \(\((\(\(\(int[#, {\[Theta], 0, 2  \[Pi]}]&\)\ /@\ 
                          \((Transpose[modeShapes] . ST1)\) /. intRule1\) //. 
                      intRule2\) /. {int[Cos[s_]\ Sin[s_], _] -> 0, 
                      int[f_[n\ \[Theta]]^2, arg_] -> \[Pi]})\) == 0 // Thread
              \), ",", 
            RowBox[{"{", 
              RowBox[{
                RowBox[{
                  SubsuperscriptBox["A", \(1, n\), 
                    TagBox[\((1, 0)\),
                      Derivative],
                    MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}], 
                ",", 
                RowBox[{
                  SubsuperscriptBox["A", \(2, n\), 
                    TagBox[\((1, 0)\),
                      Derivative],
                    MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}]}], 
              "}"}]}], "]"}], "[", \([1]\), "]"}]}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            SubsuperscriptBox["A", \(1, n\), 
              TagBox[\((1, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}], 
          "\[Rule]", "0"}], ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["A", \(2, n\), 
              TagBox[\((1, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}], 
          "\[Rule]", "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Having determined the solvability conditions of the second-order \
equations, ",
  StyleBox["order2Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we use a combination of the superposition principle and the method of \
undetermined coefficients to determine a particular solution of them. To this \
end, we identify the forms of all of the possible terms on the right-hand \
sides of ",
  StyleBox["order2Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " as follows:"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(collectForm = 
      Outer[Times, {Cos[n\ \[Theta]], Sin[n\ \[Theta]]}, 
          Table[{A\_\(i, n\)[T\_1, T\_2]\ E\^\(I\ \[Omega]\_n\ T\_0\), 
                A\&_\_\(i, n\)[T\_1, T\_2]\ 
                  E\^\(\(-I\)\ \[Omega]\_n\ T\_0\)}, {i, 2}] // Flatten] // 
        Flatten\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({E\^\(I\ T\_0\ \[Omega]\_n\)\ Cos[n\ \[Theta]]\ 
        A\_\(1, n\)[T\_1, T\_2], 
      E\^\(\(-I\)\ T\_0\ \[Omega]\_n\)\ Cos[n\ \[Theta]]\ 
        A\&_\_\(1, n\)[T\_1, T\_2], 
      E\^\(I\ T\_0\ \[Omega]\_n\)\ Cos[n\ \[Theta]]\ A\_\(2, n\)[T\_1, T\_2], 
      E\^\(\(-I\)\ T\_0\ \[Omega]\_n\)\ Cos[n\ \[Theta]]\ 
        A\&_\_\(2, n\)[T\_1, T\_2], 
      E\^\(I\ T\_0\ \[Omega]\_n\)\ Sin[n\ \[Theta]]\ A\_\(1, n\)[T\_1, T\_2], 
      E\^\(\(-I\)\ T\_0\ \[Omega]\_n\)\ Sin[n\ \[Theta]]\ 
        A\&_\_\(1, n\)[T\_1, T\_2], 
      E\^\(I\ T\_0\ \[Omega]\_n\)\ Sin[n\ \[Theta]]\ A\_\(2, n\)[T\_1, T\_2], 
      E\^\(\(-I\)\ T\_0\ \[Omega]\_n\)\ Sin[n\ \[Theta]]\ 
        A\&_\_\(2, n\)[T\_1, T\_2]}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(possibleTerms = 
      Flatten[\(If[Head[#] === Plus, List\ @@\ #, #]\ &\)\ /@\ 
              \((\(\(\(Outer[Times, collectForm, collectForm] // Flatten\) // 
                      Union\) // TrigReduce\) // Expand)\)] /. 
          c_?NumberQ\ form_ -> form // Union; \n
    possibleTerms /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({E\^\(2\ I\ T\_0\ \[Omega]\_n\)\ A\_\(1, n\)\%2, 
      E\^\(2\ I\ T\_0\ \[Omega]\_n\)\ Cos[2\ n\ \[Theta]]\ A\_\(1, n\)\%2, 
      E\^\(2\ I\ T\_0\ \[Omega]\_n\)\ Sin[2\ n\ \[Theta]]\ A\_\(1, n\)\%2, 
      E\^\(2\ I\ T\_0\ \[Omega]\_n\)\ A\_\(1, n\)\ A\_\(2, n\), 
      E\^\(2\ I\ T\_0\ \[Omega]\_n\)\ Cos[2\ n\ \[Theta]]\ A\_\(1, n\)\ 
        A\_\(2, n\), 
      E\^\(2\ I\ T\_0\ \[Omega]\_n\)\ Sin[2\ n\ \[Theta]]\ A\_\(1, n\)\ 
        A\_\(2, n\), E\^\(2\ I\ T\_0\ \[Omega]\_n\)\ A\_\(2, n\)\%2, 
      E\^\(2\ I\ T\_0\ \[Omega]\_n\)\ Cos[2\ n\ \[Theta]]\ A\_\(2, n\)\%2, 
      E\^\(2\ I\ T\_0\ \[Omega]\_n\)\ Sin[2\ n\ \[Theta]]\ A\_\(2, n\)\%2, 
      A\_\(1, n\)\ A\&_\_\(1, n\), 
      Cos[2\ n\ \[Theta]]\ A\_\(1, n\)\ A\&_\_\(1, n\), 
      Sin[2\ n\ \[Theta]]\ A\_\(1, n\)\ A\&_\_\(1, n\), 
      A\_\(2, n\)\ A\&_\_\(1, n\), 
      Cos[2\ n\ \[Theta]]\ A\_\(2, n\)\ A\&_\_\(1, n\), 
      Sin[2\ n\ \[Theta]]\ A\_\(2, n\)\ A\&_\_\(1, n\), 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_n\)\ A\&_\_\(1, n\)\%2, 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_n\)\ Cos[2\ n\ \[Theta]]\ 
        A\&_\_\(1, n\)\%2, 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_n\)\ Sin[2\ n\ \[Theta]]\ 
        A\&_\_\(1, n\)\%2, A\_\(1, n\)\ A\&_\_\(2, n\), 
      Cos[2\ n\ \[Theta]]\ A\_\(1, n\)\ A\&_\_\(2, n\), 
      Sin[2\ n\ \[Theta]]\ A\_\(1, n\)\ A\&_\_\(2, n\), 
      A\_\(2, n\)\ A\&_\_\(2, n\), 
      Cos[2\ n\ \[Theta]]\ A\_\(2, n\)\ A\&_\_\(2, n\), 
      Sin[2\ n\ \[Theta]]\ A\_\(2, n\)\ A\&_\_\(2, n\), 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_n\)\ A\&_\_\(1, n\)\ A\&_\_\(2, n\), 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_n\)\ Cos[2\ n\ \[Theta]]\ 
        A\&_\_\(1, n\)\ A\&_\_\(2, n\), 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_n\)\ Sin[2\ n\ \[Theta]]\ 
        A\&_\_\(1, n\)\ A\&_\_\(2, n\), 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_n\)\ A\&_\_\(2, n\)\%2, 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_n\)\ Cos[2\ n\ \[Theta]]\ 
        A\&_\_\(2, n\)\%2, 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_n\)\ Sin[2\ n\ \[Theta]]\ 
        A\&_\_\(2, n\)\%2}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(possibleTerms // Length\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(30\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "We note that the complex conjugate of each term on the right-hand sides of \
",
  StyleBox["order2Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " is also a term on their right-hand sides. Therefore, we use this fact to \
reduce the algebra and calculate only the particular solutions corresponding \
to almost half of the terms because the particular solutions corresponding to \
the remaining terms are the complex conjugates of those already calculated. \
Then, we express the right-hand side of ",
  StyleBox["order2Eq[[i]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " as the sum of two terms, ",
  StyleBox["tempList[i]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and its complex conjugate, for ",
  Cell[BoxData[
      \(TraditionalForm\`\(i = \)\)]],
  " 1 and 2, according to"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(Do[eq2rhsList = List\ @@\ order2Eq[\([i, 2]\)]; 
      tempList[i] = eq2rhsList; \n\t\ \ 
      \(If[term = # /. conjugateRule; MemberQ[tempList[i], term], 
            tempList[i] = Complement[tempList[i], {#}]; 
            If[# === term, AppendTo[tempList[i], 1\/2\ #]]]\ &\)\ /@\ 
        Reverse[eq2rhsList], {i, 2}]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Next, we determine the coefficients of the ",
  StyleBox["possibleTerms",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  StyleBox[" ",
    FontColor->RGBColor[1, 0, 1]],
  "in each of the ",
  StyleBox["tempList[i]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " according to"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(Do[rhs = \(Plus\ @@\ tempList[i] /. SCond1 // TrigReduce\) // Expand; 
      coeffs[i] = 
        Coefficient[rhs, possibleTerms] /. 
          \((Cos | Sin)\)[2  n\ \[Theta]] -> 0, {i, 2}]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "and denote them by ",
  Cell[BoxData[
      \(TraditionalForm\`c\_\(i, \ j\)\)]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(symbolList1 = 
      MapIndexed[If[#1 =!= 0, c\_\(Sequence\ @@\ #2\), 0]\ &, 
        Array[coeffs, 2], {2}]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({{c\_\(1, 1\), c\_\(1, 2\), 0, 0, 0, c\_\(1, 6\), c\_\(1, 7\), 
        c\_\(1, 8\), 0, c\_\(1, 10\), c\_\(1, 11\), 0, 0, 0, c\_\(1, 15\), 0, 
        0, 0, 0, 0, 0, c\_\(1, 22\), c\_\(1, 23\), 0, 0, 0, 0, 0, 0, 0}, {0, 
        0, c\_\(2, 3\), 0, c\_\(2, 5\), 0, 0, 0, c\_\(2, 9\), 0, 0, 
        c\_\(2, 12\), 0, c\_\(2, 14\), 0, 0, 0, 0, 0, 0, 0, 0, 0, 
        c\_\(2, 24\), 0, 0, 0, 0, 0, 0}}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[BoxData[
    \(\(symbolList2[s_] = Plus\ @@\ symbolList1 /. c -> s; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`\(c\_\(i, \ j\) = \)\)]],
  " ",
  StyleBox["coeffs[i][[j]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ". Moreover, we associate with the ",
  Cell[BoxData[
      \(TraditionalForm\`j\)]],
  "th term in the ",
  StyleBox["symbolList1[[i]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " the undetermined coefficient ",
  Cell[BoxData[
      \(TraditionalForm\`d\_\(i, \ j\)\)]],
  " in ",
  Cell[BoxData[
      \(TraditionalForm\`w\_2\)]],
  " and the undetermined coefficient ",
  Cell[BoxData[
      \(TraditionalForm\`e\_\(i, \ j\)\)]],
  " in ",
  Cell[BoxData[
      \(TraditionalForm\`\[Psi]\_2\)]],
  " by replacing the ",
  Cell[BoxData[
      \(TraditionalForm\`c\_\(i, \ j\)\)]],
  " in ",
  StyleBox["symbolList1[[i]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " with ",
  Cell[BoxData[
      \(TraditionalForm\`d\_\(i, \ j\)\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`e\_\(i, \ j\)\)]],
  " according to"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(sol2temp = {
        w\_2 -> Function[{\[Theta], T\_0, T\_1, T\_2}, 
            symbolList2[d] . possibleTerms // Evaluate], 
        \[Psi]\_2 -> 
          Function[{\[Theta], T\_0, T\_1, T\_2}, 
            symbolList2[e] . possibleTerms // Evaluate]}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting the ",
  StyleBox["sol2temp",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into the left-hand sides of the second-order equations, ",
  StyleBox["order2Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", setting the coefficient of the ",
  Cell[BoxData[
      \(TraditionalForm\`j\)]],
  "th term in ",
  StyleBox["possibleTerms",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in the ",
  Cell[BoxData[
      \(TraditionalForm\`i\)]],
  "th equation",
  StyleBox[" ",
    FontSlant->"Italic"],
  "equal to ",
  Cell[BoxData[
      \(TraditionalForm\`c\_\(i, \ j\)\)]],
  ", and solving the resulting algebraic equations, we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(deRule = 
      \(Solve[\(Flatten[
                  Table[Coefficient[order2Eq[\([i, 1]\)] /. sol2temp, 
                        possibleTerms] /. f_[2  n\ \[Theta]] -> 0, {i, 
                      2}]] == Flatten[symbolList1] // Thread\) // Union, 
          \(\({symbolList2[d], symbolList2[e]} // Flatten\) // Union\) // 
            Rest]\)[\([1]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(Solve::"svars" \( : \ \) 
      "Equations may not give solutions for all \"solve\" variables."\)], 
  "Message",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({e\_\(1, 1\) \[Rule] 0, e\_\(1, 7\) \[Rule] 0, 
      d\_\(1, 10\) \[Rule] c\_\(1, 10\)\/\(1 + \[Alpha]\^2\), 
      d\_\(1, 22\) \[Rule] c\_\(1, 22\)\/\(1 + \[Alpha]\^2\), 
      e\_\(1, 11\) \[Rule] 0, e\_\(1, 15\) \[Rule] 0, e\_\(1, 23\) \[Rule] 0, 
      e\_\(2, 12\) \[Rule] 
        \(-\(\(\(-2\)\ n\ c\_\(1, 11\) - c\_\(2, 12\)\)\/\(4\ n\^2\ 
                  \((\(-1\) + 4\ n\^2)\)\^2\ \[Alpha]\^2\)\)\) + 
          c\_\(2, 12\)\/\(4\ n\^2\), 
      e\_\(2, 14\) \[Rule] 
        c\_\(2, 14\)\/\(4\ n\^2\) + 
          \(\(-2\)\ n\ c\_\(1, 15\) + c\_\(2, 14\)\)\/\(4\ n\^2\ 
              \((\(-1\) + 4\ n\^2)\)\^2\ \[Alpha]\^2\), 
      e\_\(2, 24\) \[Rule] 
        \(-\(\(\(-2\)\ n\ c\_\(1, 23\) - c\_\(2, 24\)\)\/\(4\ n\^2\ 
                  \((\(-1\) + 4\ n\^2)\)\^2\ \[Alpha]\^2\)\)\) + 
          c\_\(2, 24\)\/\(4\ n\^2\), 
      d\_\(1, 1\) \[Rule] 
        c\_\(1, 1\)\/\(1 + \[Alpha]\^2 - 4\ \[Omega]\_n\%2\), 
      d\_\(1, 7\) \[Rule] 
        c\_\(1, 7\)\/\(1 + \[Alpha]\^2 - 4\ \[Omega]\_n\%2\), 
      e\_\(1, 2\) \[Rule] 0, e\_\(1, 6\) \[Rule] 0, e\_\(1, 8\) \[Rule] 0, 
      e\_\(2, 3\) \[Rule] 
        c\_\(2, 3\)\/\(4\ n\^2 - 4\ \[Omega]\_n\%2\) - 
          \(2\ n\ \((
                \(-\((4\ n\^2 - 4\ \[Omega]\_n\%2)\)\)\ c\_\(1, 2\) - 
                  2\ n\ c\_\(2, 3\))\)\)\/\(\((4\ n\^2 - 4\ \[Omega]\_n\%2)
                \)\ \((\(-4\)\ n\^2 + 
                  \((4\ n\^2 - 4\ \[Omega]\_n\%2)\)\ 
                    \((1 + \[Alpha]\^2 - 8\ n\^2\ \[Alpha]\^2 + 
                        16\ n\^4\ \[Alpha]\^2 - 4\ \[Omega]\_n\%2)\))\)\), 
      e\_\(2, 5\) \[Rule] 
        c\_\(1, 6\)\/\(2\ n\) + 
          \(\((1 + \[Alpha]\^2 - 8\ n\^2\ \[Alpha]\^2 + 
                  16\ n\^4\ \[Alpha]\^2 - 4\ \[Omega]\_n\%2)\)\ 
              \((\((4\ n\^2 - 4\ \[Omega]\_n\%2)\)\ c\_\(1, 6\) - 
                  2\ n\ c\_\(2, 5\))\)\)\/\(2\ n\ 
              \((4\ n\^2 - 
                  \((4\ n\^2 - 4\ \[Omega]\_n\%2)\)\ 
                    \((1 + \[Alpha]\^2 - 8\ n\^2\ \[Alpha]\^2 + 
                        16\ n\^4\ \[Alpha]\^2 - 4\ \[Omega]\_n\%2)\))\)\), 
      e\_\(2, 9\) \[Rule] 
        \(-\(c\_\(1, 8\)\/\(2\ n\)\)\) - 
          \(\((1 + \[Alpha]\^2 - 8\ n\^2\ \[Alpha]\^2 + 
                  16\ n\^4\ \[Alpha]\^2 - 4\ \[Omega]\_n\%2)\)\ 
              \((\((4\ n\^2 - 4\ \[Omega]\_n\%2)\)\ c\_\(1, 8\) + 
                  2\ n\ c\_\(2, 9\))\)\)\/\(2\ n\ 
              \((4\ n\^2 - 
                  \((4\ n\^2 - 4\ \[Omega]\_n\%2)\)\ 
                    \((1 + \[Alpha]\^2 - 8\ n\^2\ \[Alpha]\^2 + 
                        16\ n\^4\ \[Alpha]\^2 - 4\ \[Omega]\_n\%2)\))\)\), 
      d\_\(1, 11\) \[Rule] 
        \(-\(\(\(-2\)\ n\ c\_\(1, 11\) - c\_\(2, 12\)\)\/\(2\ n\ 
                \((\(-1\) + 4\ n\^2)\)\^2\ \[Alpha]\^2\)\)\), 
      d\_\(1, 15\) \[Rule] 
        \(-\(\(\(-2\)\ n\ c\_\(1, 15\) + c\_\(2, 14\)\)\/\(2\ n\ 
                \((\(-1\) + 4\ n\^2)\)\^2\ \[Alpha]\^2\)\)\), 
      d\_\(1, 23\) \[Rule] 
        \(-\(\(\(-2\)\ n\ c\_\(1, 23\) - c\_\(2, 24\)\)\/\(2\ n\ 
                \((\(-1\) + 4\ n\^2)\)\^2\ \[Alpha]\^2\)\)\), 
      d\_\(2, 12\) \[Rule] 0, d\_\(2, 14\) \[Rule] 0, d\_\(2, 24\) \[Rule] 0, 
      d\_\(1, 2\) \[Rule] 
        \(-\(\(\(-\((4\ n\^2 - 4\ \[Omega]\_n\%2)\)\)\ c\_\(1, 2\) - 
                2\ n\ c\_\(2, 3\)\)\/\(\(-4\)\ n\^2 + 
                \((4\ n\^2 - 4\ \[Omega]\_n\%2)\)\ 
                  \((1 + \[Alpha]\^2 - 8\ n\^2\ \[Alpha]\^2 + 
                      16\ n\^4\ \[Alpha]\^2 - 4\ \[Omega]\_n\%2)\)\)\)\), 
      d\_\(1, 6\) \[Rule] 
        \(-\(\(\((4\ n\^2 - 4\ \[Omega]\_n\%2)\)\ c\_\(1, 6\) - 
                2\ n\ c\_\(2, 5\)\)\/\(4\ n\^2 - 
                \((4\ n\^2 - 4\ \[Omega]\_n\%2)\)\ 
                  \((1 + \[Alpha]\^2 - 8\ n\^2\ \[Alpha]\^2 + 
                      16\ n\^4\ \[Alpha]\^2 - 4\ \[Omega]\_n\%2)\)\)\)\), 
      d\_\(1, 8\) \[Rule] 
        \(-\(\(\((4\ n\^2 - 4\ \[Omega]\_n\%2)\)\ c\_\(1, 8\) + 
                2\ n\ c\_\(2, 9\)\)\/\(4\ n\^2 - 
                \((4\ n\^2 - 4\ \[Omega]\_n\%2)\)\ 
                  \((1 + \[Alpha]\^2 - 8\ n\^2\ \[Alpha]\^2 + 
                      16\ n\^4\ \[Alpha]\^2 - 4\ \[Omega]\_n\%2)\)\)\)\), 
      d\_\(2, 3\) \[Rule] 0, d\_\(2, 5\) \[Rule] 0, d\_\(2, 9\) \[Rule] 0}
      \)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
To simplify the expressions in the second-order solution, we introduce the \
notation\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(paperNotationRule = 
      Thread[\((\(symbolList2[c] // Union\) // Rest)\) -> {c\_2, c\_1, 
              2  c\_1, c\_2, \(-c\_1\), \(-c\_4\), \(-\(1\/2\)\) c\_3, 
              \(-c\_3\), \(-c\_4\), \(1\/2\) c\_3, d\_1, \(-2\) d\_1, 
              \(-d\_1\), \(-\(1\/2\)\) d\_2, d\_2, \(1\/2\) d\_2}]~
        Join~{\((
              1 + \[Alpha]\^2 - 8\ n\^2\ \[Alpha]\^2 + 
                16\ n\^4\ \[Alpha]\^2 - 4\ \[Omega]\_n\%2)\) -> 
            \(\[CapitalDelta]\_n + 4  n\^2\)\/\(4\ n\^2 - 
                4\ \[Omega]\_n\%2\), 
          Power[\(-1\) + 4\ n\^2, \(-2\)]\ Power[\[Alpha], \(-2\)] -> 
            \(4\ n\^2\)\/\[CapitalTheta]\_n}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({c\_\(1, 1\) \[Rule] c\_2, c\_\(1, 2\) \[Rule] c\_1, 
      c\_\(1, 6\) \[Rule] 2\ c\_1, c\_\(1, 7\) \[Rule] c\_2, 
      c\_\(1, 8\) \[Rule] \(-c\_1\), c\_\(1, 10\) \[Rule] \(-c\_4\), 
      c\_\(1, 11\) \[Rule] \(-\(c\_3\/2\)\), c\_\(1, 15\) \[Rule] \(-c\_3\), 
      c\_\(1, 22\) \[Rule] \(-c\_4\), c\_\(1, 23\) \[Rule] c\_3\/2, 
      c\_\(2, 3\) \[Rule] d\_1, c\_\(2, 5\) \[Rule] \(-2\)\ d\_1, 
      c\_\(2, 9\) \[Rule] \(-d\_1\), c\_\(2, 12\) \[Rule] \(-\(d\_2\/2\)\), 
      c\_\(2, 14\) \[Rule] d\_2, c\_\(2, 24\) \[Rule] d\_2\/2, 
      1 + \[Alpha]\^2 - 8\ n\^2\ \[Alpha]\^2 + 16\ n\^4\ \[Alpha]\^2 - 
          4\ \[Omega]\_n\%2 \[Rule] 
        \(4\ n\^2 + \[CapitalDelta]\_n\)\/\(4\ n\^2 - 4\ \[Omega]\_n\%2\), 
      1\/\(\((\(-1\) + 4\ n\^2)\)\^2\ \[Alpha]\^2\) \[Rule] 
        \(4\ n\^2\)\/\[CapitalTheta]\_n}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["Then, we express the second-order solution as", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(w2Sol = 
      \(symbolList2[d] . possibleTerms /. deRule\) //. paperNotationRule; \n
    w2Sol /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(E\^\(2\ I\ T\_0\ \[Omega]\_n\)\ c\_2\ A\_\(1, n\)\%2\)\/\(1 + 
          \[Alpha]\^2 - 4\ \[Omega]\_n\%2\) - 
      \(E\^\(2\ I\ T\_0\ \[Omega]\_n\)\ Cos[2\ n\ \[Theta]]\ 
          \((\(-2\)\ n\ d\_1 - c\_1\ \((4\ n\^2 - 4\ \[Omega]\_n\%2)\))\)\ 
          A\_\(1, n\)\%2\)\/\[CapitalDelta]\_n + 
      \(E\^\(2\ I\ T\_0\ \[Omega]\_n\)\ Sin[2\ n\ \[Theta]]\ 
          \((4\ n\ d\_1 + 2\ c\_1\ \((4\ n\^2 - 4\ \[Omega]\_n\%2)\))\)\ 
          A\_\(1, n\)\ A\_\(2, n\)\)\/\[CapitalDelta]\_n + 
      \(E\^\(2\ I\ T\_0\ \[Omega]\_n\)\ c\_2\ A\_\(2, n\)\%2\)\/\(1 + 
          \[Alpha]\^2 - 4\ \[Omega]\_n\%2\) + 
      \(E\^\(2\ I\ T\_0\ \[Omega]\_n\)\ Cos[2\ n\ \[Theta]]\ 
          \((\(-2\)\ n\ d\_1 - c\_1\ \((4\ n\^2 - 4\ \[Omega]\_n\%2)\))\)\ 
          A\_\(2, n\)\%2\)\/\[CapitalDelta]\_n - 
      \(c\_4\ A\_\(1, n\)\ A\&_\_\(1, n\)\)\/\(1 + \[Alpha]\^2\) - 
      \(2\ n\ Cos[2\ n\ \[Theta]]\ \((n\ c\_3 + d\_2\/2)\)\ A\_\(1, n\)\ 
          A\&_\_\(1, n\)\)\/\[CapitalTheta]\_n - 
      \(2\ n\ Sin[2\ n\ \[Theta]]\ \((2\ n\ c\_3 + d\_2)\)\ A\_\(2, n\)\ 
          A\&_\_\(1, n\)\)\/\[CapitalTheta]\_n - 
      \(c\_4\ A\_\(2, n\)\ A\&_\_\(2, n\)\)\/\(1 + \[Alpha]\^2\) - 
      \(2\ n\ Cos[2\ n\ \[Theta]]\ \((\(-n\)\ c\_3 - d\_2\/2)\)\ A\_\(2, n\)\ 
          A\&_\_\(2, n\)\)\/\[CapitalTheta]\_n\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(\[Psi]2Sol = 
      \(\(symbolList2[e] . possibleTerms /. deRule\) //. paperNotationRule
          \) /. e\_i__ -> 0; \n\[Psi]2Sol /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(E\^\(2\ I\ T\_0\ \[Omega]\_n\)\ Sin[2\ n\ \[Theta]]\ 
        \((d\_1\/\(4\ n\^2 - 4\ \[Omega]\_n\%2\) - 
            \(2\ n\ \((
                  \(-2\)\ n\ d\_1 - c\_1\ \((4\ n\^2 - 4\ \[Omega]\_n\%2)\))
                  \)\)\/\(\[CapitalDelta]\_n\ 
                \((4\ n\^2 - 4\ \[Omega]\_n\%2)\)\))\)\ A\_\(1, n\)\%2 + 
      E\^\(2\ I\ T\_0\ \[Omega]\_n\)\ Cos[2\ n\ \[Theta]]\ 
        \((c\_1\/n - 
            \(\((4\ n\^2 + \[CapitalDelta]\_n)\)\ 
                \((4\ n\ d\_1 + 2\ c\_1\ \((4\ n\^2 - 4\ \[Omega]\_n\%2)\))
                  \)\)\/\(2\ n\ \[CapitalDelta]\_n\ 
                \((4\ n\^2 - 4\ \[Omega]\_n\%2)\)\))\)\ A\_\(1, n\)\ 
        A\_\(2, n\) + 
      E\^\(2\ I\ T\_0\ \[Omega]\_n\)\ Sin[2\ n\ \[Theta]]\ 
        \((c\_1\/\(2\ n\) + 
            \(\((4\ n\^2 + \[CapitalDelta]\_n)\)\ 
                \((\(-2\)\ n\ d\_1 - c\_1\ \((4\ n\^2 - 4\ \[Omega]\_n\%2)\))
                  \)\)\/\(2\ n\ \[CapitalDelta]\_n\ 
                \((4\ n\^2 - 4\ \[Omega]\_n\%2)\)\))\)\ A\_\(2, n\)\%2 + 
      Sin[2\ n\ \[Theta]]\ 
        \((\(-\(d\_2\/\(8\ n\^2\)\)\) - 
            \(n\ c\_3 + d\_2\/2\)\/\[CapitalTheta]\_n)\)\ A\_\(1, n\)\ 
        A\&_\_\(1, n\) + 
      Cos[2\ n\ \[Theta]]\ 
        \((d\_2\/\(4\ n\^2\) + \(2\ n\ c\_3 + d\_2\)\/\[CapitalTheta]\_n)\)\ 
        A\_\(2, n\)\ A\&_\_\(1, n\) + 
      Sin[2\ n\ \[Theta]]\ 
        \((d\_2\/\(8\ n\^2\) - \(\(-n\)\ c\_3 - d\_2\/2\)\/\[CapitalTheta]\_n)
          \)\ A\_\(2, n\)\ A\&_\_\(2, n\)\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["We express this solution in pure function form as", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(sol2 = {
        w\_2 -> Function[{\[Theta], T\_0, T\_1, T\_2}, 
            w2Sol + \((w2Sol /. conjugateRule)\) // Evaluate], 
        \[Psi]\_2 -> 
          Function[{\[Theta], T\_0, T\_1, T\_2}, 
            \[Psi]2Sol + \((\[Psi]2Sol /. conjugateRule)\) // Evaluate]}; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["8.2.3 Solvability Conditions", "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting the ",
  StyleBox["sol1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["sol2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["eqEps[[3]] ",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  "yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(order3Eq = 
        Table[\((
              eqEps[\([1, i, 1]\)] /. {w\_1 -> w\_3, \[Psi]\_1 -> \[Psi]\_3})
              \) == \((
                eqEps[\([1, i, 1]\)] /. {w\_1 -> w\_3, 
                    \[Psi]\_1 -> \[Psi]\_3})\) - 
              \((\(\(Subtract\ @@\ eqEps[\([3, i]\)] /. sol1\) /. sol2 // 
                    TrigToExp\) // Expand)\), {i, 2}]; \) // Timing\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({90.5000000000000071`\ Second, Null}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To convert the terms that lead to small-divisor terms in the third-order \
equations, ",
  StyleBox["order3Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", into terms that lead to secular terms, we introduce the rule"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(expRule1 = 
      Exp[arg_] :> 
        Exp[Expand[arg /. \((ResonanceCond /. Equal -> Rule)\)] /. 
            \[Epsilon]\^2\ T\_0 -> T\_2]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Then, the sources of secular terms in ",
  StyleBox["order3Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", the terms proportional to ",
  Cell[BoxData[
      \(TraditionalForm
      \`Exp[\(\[PlusMinus]\ I\)\ n\ \[Theta] + I\ \[Omega]\_n\ T\_0]\)]],
  ", are"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(ST2 = 
        \(Expand[
              Coefficient[
                  \(#[\([2]\)] /. 
                      Exp[c1_.  + 
                            Complex[0, s_ /; Abs[s] != 1]\ n\ \[Theta]] -> 0
                        \) /. expRule1, Exp[I\ \[Omega]\_n\ T\_0]] /. 
                Exp[c1_.  + \ Complex[0, m_]\ n\ \[Theta]] -> 
                  \((Cos[n\ \[Theta]] + I\ m\ Sin[n\ \[Theta]])\)\ Exp[c1]]\ &
            \)\ /@\ order3Eq; \) // Timing\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({64.3429999999998525`\ Second, Null}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "These terms consist of linear and cubic terms in the complex-valued \
amplitudes ",
  Cell[BoxData[
      \(TraditionalForm\`A\_i\)]],
  ". The cubic terms are proportional to"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(cubicTerms = 
      \(\(\(Flatten[
                  \(If[Head[#] === Plus, List\ @@\ #, #]\ &\)\ /@\ 
                    \((\(\(\(Nest[Outer[Times, collectForm, #]\ &, 
                                collectForm, 2] // Flatten\) // Union\) // 
                          TrigReduce\) // Expand)\)] /. {
                  f_[s_\ n\ \[Theta]] -> 0, 
                  Exp[c1_.  + Complex[0, s_ /; s != 1]\ \[Omega]\_n\ T\_0] -> 
                    0}\) /. Exp[_] -> 1\) /. c_?NumberQ\ form_ -> form // 
          Union\) // Rest; \ncubicTerms /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({Cos[n\ \[Theta]]\ A\_\(1, n\)\%2\ A\&_\_\(1, n\), 
      Sin[n\ \[Theta]]\ A\_\(1, n\)\%2\ A\&_\_\(1, n\), 
      Cos[n\ \[Theta]]\ A\_\(1, n\)\ A\_\(2, n\)\ A\&_\_\(1, n\), 
      Sin[n\ \[Theta]]\ A\_\(1, n\)\ A\_\(2, n\)\ A\&_\_\(1, n\), 
      Cos[n\ \[Theta]]\ A\_\(2, n\)\%2\ A\&_\_\(1, n\), 
      Sin[n\ \[Theta]]\ A\_\(2, n\)\%2\ A\&_\_\(1, n\), 
      Cos[n\ \[Theta]]\ A\_\(1, n\)\%2\ A\&_\_\(2, n\), 
      Sin[n\ \[Theta]]\ A\_\(1, n\)\%2\ A\&_\_\(2, n\), 
      Cos[n\ \[Theta]]\ A\_\(1, n\)\ A\_\(2, n\)\ A\&_\_\(2, n\), 
      Sin[n\ \[Theta]]\ A\_\(1, n\)\ A\_\(2, n\)\ A\&_\_\(2, n\), 
      Cos[n\ \[Theta]]\ A\_\(2, n\)\%2\ A\&_\_\(2, n\), 
      Sin[n\ \[Theta]]\ A\_\(2, n\)\%2\ A\&_\_\(2, n\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "And, their coefficients in ",
  StyleBox["ST2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " are"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(coef2 = Outer[Coefficient, ST2, cubicTerms]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["which we denote by", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(symbolList3 = 
      MapIndexed[If[#1 =!= 0, g\_\(Sequence\ @@\ #2\), 0]\ &, coef2, {2}]; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(notationRule = 
      \((\(\(symbolList3 // Flatten\) // Union\) // Rest)\) -> {E\_2, E\_1, 
            E\_3, E\_3, E\_1, E\_2, \(-G\_2\), G\_1, \(-G\_3\), G\_3, 
            \(-G\_1\), G\_2} // Thread\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({g\_\(1, 1\) \[Rule] E\_2, g\_\(1, 4\) \[Rule] E\_1, 
      g\_\(1, 5\) \[Rule] E\_3, g\_\(1, 8\) \[Rule] E\_3, 
      g\_\(1, 9\) \[Rule] E\_1, g\_\(1, 12\) \[Rule] E\_2, 
      g\_\(2, 2\) \[Rule] \(-G\_2\), g\_\(2, 3\) \[Rule] G\_1, 
      g\_\(2, 6\) \[Rule] \(-G\_3\), g\_\(2, 7\) \[Rule] G\_3, 
      g\_\(2, 10\) \[Rule] \(-G\_1\), g\_\(2, 11\) \[Rule] G\_2}\)], "Output",\

  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "In terms of this notation, ",
  StyleBox["ST2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " can be rewritten as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(ST2New = 
      \(\((\(# /. Thread[cubicTerms -> 0]\ &\)\ /@\ ST2)\) + 
            symbolList3 . cubicTerms /. notationRule\) /. D[SCond1, T\_1]; \n
    ST2New /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
        \(1\/2\ E\^\(I\ T\_2\ \[Sigma]\)\ f\ Cos[n\ \[Theta]]\), "-", 
          RowBox[{"2", " ", "I", " ", \(Cos[n\ \[Theta]]\), " ", 
            RowBox[{"(", 
              InterpretationBox[\(D\_2\[InvisibleSpace]A\_\(1, n\)\),
                SequenceForm[ 
                  Subscript[ D, 2], 
                  Subscript[ A, 1, n]],
                Editable->False], ")"}], " ", \(\[Omega]\_n\)}], "-", 
          RowBox[{"2", " ", "I", " ", 
            RowBox[{"(", 
              InterpretationBox[\(D\_2\[InvisibleSpace]A\_\(2, n\)\),
                SequenceForm[ 
                  Subscript[ D, 2], 
                  Subscript[ A, 2, n]],
                Editable->False], ")"}], " ", \(Sin[n\ \[Theta]]\), " ", 
            \(\[Omega]\_n\)}], "+", 
          \(Cos[n\ \[Theta]]\ E\_2\ A\_\(1, n\)\%2\ A\&_\_\(1, n\)\), "+", 
          \(Sin[n\ \[Theta]]\ E\_1\ A\_\(1, n\)\ A\_\(2, n\)\ 
            A\&_\_\(1, n\)\), "+", 
          \(Cos[n\ \[Theta]]\ E\_3\ A\_\(2, n\)\%2\ A\&_\_\(1, n\)\), "+", 
          \(Sin[n\ \[Theta]]\ E\_3\ A\_\(1, n\)\%2\ A\&_\_\(2, n\)\), "+", 
          \(Cos[n\ \[Theta]]\ E\_1\ A\_\(1, n\)\ A\_\(2, n\)\ 
            A\&_\_\(2, n\)\), "+", 
          \(Sin[n\ \[Theta]]\ E\_2\ A\_\(2, n\)\%2\ A\&_\_\(2, n\)\)}], ",", 
        RowBox[{
          RowBox[{"2", " ", "I", " ", \(Cos[n\ \[Theta]]\), " ", 
            RowBox[{"(", 
              InterpretationBox[\(D\_2\[InvisibleSpace]A\_\(2, n\)\),
                SequenceForm[ 
                  Subscript[ D, 2], 
                  Subscript[ A, 2, n]],
                Editable->False], ")"}], " ", \(\[CapitalGamma]\_n\), " ", 
            \(\[Omega]\_n\)}], "-", 
          RowBox[{"2", " ", "I", " ", 
            RowBox[{"(", 
              InterpretationBox[\(D\_2\[InvisibleSpace]A\_\(1, n\)\),
                SequenceForm[ 
                  Subscript[ D, 2], 
                  Subscript[ A, 1, n]],
                Editable->False], ")"}], " ", \(Sin[n\ \[Theta]]\), " ", 
            \(\[CapitalGamma]\_n\), " ", \(\[Omega]\_n\)}], "-", 
          \(Sin[n\ \[Theta]]\ G\_2\ A\_\(1, n\)\%2\ A\&_\_\(1, n\)\), "+", 
          \(Cos[n\ \[Theta]]\ G\_1\ A\_\(1, n\)\ A\_\(2, n\)\ 
            A\&_\_\(1, n\)\), "-", 
          \(Sin[n\ \[Theta]]\ G\_3\ A\_\(2, n\)\%2\ A\&_\_\(1, n\)\), "+", 
          \(Cos[n\ \[Theta]]\ G\_3\ A\_\(1, n\)\%2\ A\&_\_\(2, n\)\), "-", 
          \(Sin[n\ \[Theta]]\ G\_1\ A\_\(1, n\)\ A\_\(2, n\)\ 
            A\&_\_\(2, n\)\), "+", 
          \(Cos[n\ \[Theta]]\ G\_2\ A\_\(2, n\)\%2\ A\&_\_\(2, n\)\)}]}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The solvability conditions of ",
  StyleBox["order3Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " demand that the vector ",
  StyleBox["ST2New",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " be orthogonal to the transpose of the ",
  StyleBox["modeShapes",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  "; that is,"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(SCond2 = 
      Expand[\(\(\(1\/\[Pi]\ int[#, {\[Theta], 0, 2  \[Pi]}]\ &\)\ /@\ 
                    \((Transpose[modeShapes] . ST2New)\) /. intRule1\) //. 
                intRule2\) /. {int[Cos[s_]\ Sin[s_], _] -> 0, 
                int[f_[n\ \[Theta]]^2, _] -> \[Pi]}] == 0 // Thread; \n
    SCond2 /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{\(1\/2\ E\^\(I\ T\_2\ \[Sigma]\)\ f\), "-", 
            RowBox[{"2", " ", "I", " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_2\[InvisibleSpace]A\_\(1, n\)\),
                  SequenceForm[ 
                    Subscript[ D, 2], 
                    Subscript[ A, 1, n]],
                  Editable->False], ")"}], " ", \(\[Omega]\_n\)}], "-", 
            RowBox[{"2", " ", "I", " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_2\[InvisibleSpace]A\_\(1, n\)\),
                  SequenceForm[ 
                    Subscript[ D, 2], 
                    Subscript[ A, 1, n]],
                  Editable->False], ")"}], " ", \(\[CapitalGamma]\_n\%2\), 
              " ", \(\[Omega]\_n\)}], "+", 
            \(E\_2\ A\_\(1, n\)\%2\ A\&_\_\(1, n\)\), "-", 
            \(G\_2\ \[CapitalGamma]\_n\ A\_\(1, n\)\%2\ A\&_\_\(1, n\)\), 
            "+", \(E\_3\ A\_\(2, n\)\%2\ A\&_\_\(1, n\)\), "-", 
            \(G\_3\ \[CapitalGamma]\_n\ A\_\(2, n\)\%2\ A\&_\_\(1, n\)\), 
            "+", \(E\_1\ A\_\(1, n\)\ A\_\(2, n\)\ A\&_\_\(2, n\)\), "-", 
            \(G\_1\ \[CapitalGamma]\_n\ A\_\(1, n\)\ A\_\(2, n\)\ 
              A\&_\_\(2, n\)\)}], "==", "0"}], ",", 
        RowBox[{
          RowBox[{
            RowBox[{\(-2\), " ", "I", " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_2\[InvisibleSpace]A\_\(2, n\)\),
                  SequenceForm[ 
                    Subscript[ D, 2], 
                    Subscript[ A, 2, n]],
                  Editable->False], ")"}], " ", \(\[Omega]\_n\)}], "-", 
            RowBox[{"2", " ", "I", " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_2\[InvisibleSpace]A\_\(2, n\)\),
                  SequenceForm[ 
                    Subscript[ D, 2], 
                    Subscript[ A, 2, n]],
                  Editable->False], ")"}], " ", \(\[CapitalGamma]\_n\%2\), 
              " ", \(\[Omega]\_n\)}], "+", 
            \(E\_1\ A\_\(1, n\)\ A\_\(2, n\)\ A\&_\_\(1, n\)\), "-", 
            \(G\_1\ \[CapitalGamma]\_n\ A\_\(1, n\)\ A\_\(2, n\)\ 
              A\&_\_\(1, n\)\), "+", 
            \(E\_3\ A\_\(1, n\)\%2\ A\&_\_\(2, n\)\), "-", 
            \(G\_3\ \[CapitalGamma]\_n\ A\_\(1, n\)\%2\ A\&_\_\(2, n\)\), 
            "+", \(E\_2\ A\_\(2, n\)\%2\ A\&_\_\(2, n\)\), "-", 
            \(G\_2\ \[CapitalGamma]\_n\ A\_\(2, n\)\%2\ A\&_\_\(2, n\)\)}], 
          "==", "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["8.3 Near-Square Plates", "Section",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "We consider the nonlinear transverse vibrations of a near-square simply \
supported plate that is initially flat. We introduce a Cartesian coordinate \
system ",
  Cell[BoxData[
      \(TraditionalForm\`x\ y\ z\)]],
  ", with the ",
  Cell[BoxData[
      \(TraditionalForm\`x\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`y\)]],
  " axes lying in the undisturbed plane of the middle surface of the plate, \
and let ",
  Cell[BoxData[
      \(TraditionalForm\`u\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`v\)]],
  ", and ",
  Cell[BoxData[
      \(TraditionalForm\`w\)]],
  " denote the displacements of the middle surface in the ",
  Cell[BoxData[
      \(TraditionalForm\`x\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`y\)]],
  ", and ",
  Cell[BoxData[
      \(TraditionalForm\`z\)]],
  " directions, respectively. Then, the equations of motion are (Chu and \
Herrmann, 1956)"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(Needs["\<Calculus`VectorAnalysis`\>"]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(BH[w] = Biharmonic[w[x, y, t], Cartesian[x, y, z]]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SuperscriptBox["w", 
          TagBox[\((0, 4, 0)\),
            Derivative],
          MultilineFunction->None], "[", \(x, y, t\), "]"}], "+", 
      RowBox[{"2", " ", 
        RowBox[{
          SuperscriptBox["w", 
            TagBox[\((2, 2, 0)\),
              Derivative],
            MultilineFunction->None], "[", \(x, y, t\), "]"}]}], "+", 
      RowBox[{
        SuperscriptBox["w", 
          TagBox[\((4, 0, 0)\),
            Derivative],
          MultilineFunction->None], "[", \(x, y, t\), "]"}]}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[BoxData[
    \(\(EOM = {\n\t\t
          u\_\(x, x\) + w\_x\ w\_\(x, x\) + 
              \[Nu]\ \((v\_\(x, y\) + w\_y\ w\_\(x, y\))\) + 
              \(1\/2\) \((1 - \[Nu])\)\ 
                \((u\_\(y, y\) + v\_\(x, y\) + w\_x\ w\_\(y, y\) + 
                    w\_y\ w\_\(x, y\))\) == c\_p\%\(-2\)\ u\_\(t, t\), \n\t\t
          v\_\(y, y\) + w\_y\ w\_\(y, y\) + 
              \[Nu]\ \((u\_\(x, y\) + w\_x\ w\_\(x, y\))\) + 
              \(1\/2\) \((1 - \[Nu])\) 
                \((v\_\(x, x\) + u\_\(x, y\) + w\_y\ w\_\(x, x\) + 
                    w\_x\ w\_\(x, y\))\) == c\_p\%\(-2\)\ v\_\(t, t\), \n\t\t
          \(1\/12\) h\^2\ BH[w] - N\_1\%i\ w\_\(x, x\) - 
              2  N\_12\%i\ w\_\(x, y\) - N\_2\%i\ w\_\(y, y\) - 
              u\_x\ w\_\(x, x\) - \(1\/2\) w\_x\%2\ w\_\(x, x\) - 
              v\_y\ w\_\(y, y\) - \(1\/2\) w\_y\%2\ w\_\(y, y\) - 
              \[Nu] \((
                  v\_y\ w\_\(x, x\) + \(1\/2\) w\_y\%2\ w\_\(x, x\) + 
                    u\_x\ w\_\(y, y\) + \(1\/2\) w\_x\%2\ w\_\(y, y\))\) - 
              \((1 - \[Nu])\) \((u\_y + v\_x + w\_x\ w\_y)\) w\_\(x, y\) == 
            \(-c\_p\%\(-2\)\) 
                \((w\_x\ u\_\(t, t\) + w\_y\ v\_\(t, t\) + w\_\(t, t\))\) + 
              \(F \((1 - \[Nu]\^2)\)\)\/\(E\_s\ h\)\n\t} /. 
        \((s : u | v | w)\)\_m__ :> \[PartialD]\_m s[x, y, t]; \)\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell["and the boundary conditions are", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(BC = {{u[x, y, t] == 0, w[x, y, t] == 0, 
              \[PartialD]\_\(x, x\)w[x, y, t] == 0} /. {{x -> 0}, {x -> a}}, {
              v[x, y, t] == 0, w[x, y, t] == 0, 
              \[PartialD]\_\(y, y\)w[x, y, t] == 0} /. {{y -> 0}, {
                y -> b}}} // Flatten\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(u[0, y, t] == 0\), ",", \(w[0, y, t] == 0\), ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["w", 
              TagBox[\((2, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(0, y, t\), "]"}], "==", "0"}], 
        ",", \(u[a, y, t] == 0\), ",", \(w[a, y, t] == 0\), ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["w", 
              TagBox[\((2, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(a, y, t\), "]"}], "==", "0"}], 
        ",", \(v[x, 0, t] == 0\), ",", \(w[x, 0, t] == 0\), ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["w", 
              TagBox[\((0, 2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(x, 0, t\), "]"}], "==", "0"}], 
        ",", \(v[x, b, t] == 0\), ",", \(w[x, b, t] == 0\), ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["w", 
              TagBox[\((0, 2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(x, b, t\), "]"}], "==", 
          "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`c\_p\%2 = E\_s/\((\[Rho](1 - \[Nu]\^2))\)\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`E\_s\)]],
  " is Young's modulus, ",
  Cell[BoxData[
      \(TraditionalForm\`\[Nu]\)]],
  " is Poisson's ratio, ",
  Cell[BoxData[
      \(TraditionalForm\`\[Rho]\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`h\)]],
  " are the density and thickness of the plate, respectively, ",
  Cell[BoxData[
      \(TraditionalForm\`F\)]],
  " is the transverse load, the ",
  Cell[BoxData[
      \(TraditionalForm\`N\_k\%i\)]],
  " are the uniformly distributed in-plane applied edge loads, and ",
  Cell[BoxData[
      \(TraditionalForm\`a\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`b\)]],
  " are the edge lengths, with ",
  Cell[BoxData[
      \(TraditionalForm\`b \[TildeTilde] a\)]],
  ". ",
  StyleBox["EOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " are usually called the ",
  StyleBox["dynamic analog of the von Karman equations",
    FontColor->RGBColor[0, 0, 1]],
  " because they reduce to von Karman's equations (1910) in the absence of \
the time derivatives."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "For thin plates, ",
  Cell[BoxData[
      \(TraditionalForm\`h/a\)]],
  " is very small, which we denote by ",
  Cell[BoxData[
      \(TraditionalForm\`\@\[Epsilon]\)]],
  ". Next, we introduce the aspect ratio ",
  Cell[BoxData[
      \(TraditionalForm\`c = a/b\)]],
  ", define the rule"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(scaleRule1 = {h\^2 -> \[Epsilon]\ a\^2, a -> c\ b}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["and nondimensionalize the variables as", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(\(charT = \(\@\(12/\[Epsilon]\)\) a/c\_p; 
    \)\ \ \ \ \ \ \ \ \ \ \  (*\ characteristic\ time\ *) \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(nondimRule = {
        \((dv : u | v | w)\) -> \((a\ dv[#1/a, #2/b, #3/charT]\ &)\), 
        F -> F\ E\_s\ h/\((12  a \((1 - \[Nu]\^2)\))\), x -> a\ x, y -> b\ y, 
        t -> charT\ t}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting ",
  StyleBox["nondimRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["scaleRule1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["EOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["BC",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and considering the case of no in-plane edge loads, we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(EOM1 = 
      \(\((\(\(a \((Subtract\ @@\ #)\) /. N\_k_\%i -> 0\) /. nondimRule\) //. 
                  scaleRule1 // Expand)\) == 0\ &\)\ /@\ EOM; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(BC1 = \(BC /. nondimRule\) /. c_\ h_[a__, t] :> h[a, t]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(u[0, y, t] == 0\), ",", \(w[0, y, t] == 0\), ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["w", 
              TagBox[\((2, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(0, y, t\), "]"}], "==", "0"}], 
        ",", \(u[1, y, t] == 0\), ",", \(w[1, y, t] == 0\), ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["w", 
              TagBox[\((2, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(1, y, t\), "]"}], "==", "0"}], 
        ",", \(v[x, 0, t] == 0\), ",", \(w[x, 0, t] == 0\), ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["w", 
              TagBox[\((0, 2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(x, 0, t\), "]"}], "==", "0"}], 
        ",", \(v[x, 1, t] == 0\), ",", \(w[x, 1, t] == 0\), ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["w", 
              TagBox[\((0, 2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(x, 1, t\), "]"}], "==", 
          "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To write ",
  StyleBox["EOM1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in a more concise way, we introduce the display rule"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(display1[expr_] := 
      expr /. {\(\(Derivative[a_, b_, c_]\)[w_]\)[x, y, t] :> 
            Subscript[w, 
              Sequence\ @@\ 
                Join[Table[x, {a}], Table[y, {b}], Table[t, {c}]]]}\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["and obtain", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[{
    \(EOM1[\([1]\)] // display1\), 
    \(EOM1[\([2]\)] // display1\), 
    \(EOM1[\([3]\)] // display1\)}], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(-\(1\/12\)\)\ \[Epsilon]\ u\_\(t, t\) + u\_\(x, x\) + 
        1\/2\ c\^2\ u\_\(y, y\) - 1\/2\ c\^2\ \[Nu]\ u\_\(y, y\) + 
        1\/2\ c\ v\_\(x, y\) + 1\/2\ c\ \[Nu]\ v\_\(x, y\) + 
        w\_x\ w\_\(x, x\) + 1\/2\ c\^2\ w\_y\ w\_\(x, y\) + 
        1\/2\ c\^2\ \[Nu]\ w\_y\ w\_\(x, y\) + 
        1\/2\ c\^2\ w\_x\ w\_\(y, y\) - 
        1\/2\ c\^2\ \[Nu]\ w\_x\ w\_\(y, y\) == 0\)], "Output",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(1\/2\ c\ u\_\(x, y\) + 1\/2\ c\ \[Nu]\ u\_\(x, y\) - 
        1\/12\ \[Epsilon]\ v\_\(t, t\) + v\_\(x, x\)\/2 - 
        1\/2\ \[Nu]\ v\_\(x, x\) + c\^2\ v\_\(y, y\) + 
        1\/2\ c\ w\_y\ w\_\(x, x\) - 1\/2\ c\ \[Nu]\ w\_y\ w\_\(x, x\) + 
        1\/2\ c\ w\_x\ w\_\(x, y\) + 1\/2\ c\ \[Nu]\ w\_x\ w\_\(x, y\) + 
        c\^3\ w\_y\ w\_\(y, y\) == 0\)], "Output",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(-\(F\/12\)\) + 1\/12\ \[Epsilon]\ w\_x\ u\_\(t, t\) + 
        1\/12\ c\ \[Epsilon]\ w\_y\ v\_\(t, t\) + 
        1\/12\ \[Epsilon]\ w\_\(t, t\) - u\_x\ w\_\(x, x\) - 
        c\ \[Nu]\ v\_y\ w\_\(x, x\) - 1\/2\ w\_x\%2\ w\_\(x, x\) - 
        1\/2\ c\^2\ \[Nu]\ w\_y\%2\ w\_\(x, x\) - c\^2\ u\_y\ w\_\(x, y\) + 
        c\^2\ \[Nu]\ u\_y\ w\_\(x, y\) - c\ v\_x\ w\_\(x, y\) + 
        c\ \[Nu]\ v\_x\ w\_\(x, y\) - c\^2\ w\_x\ w\_y\ w\_\(x, y\) + 
        c\^2\ \[Nu]\ w\_x\ w\_y\ w\_\(x, y\) - 
        c\^2\ \[Nu]\ u\_x\ w\_\(y, y\) - c\^3\ v\_y\ w\_\(y, y\) - 
        1\/2\ c\^2\ \[Nu]\ w\_x\%2\ w\_\(y, y\) - 
        1\/2\ c\^4\ w\_y\%2\ w\_\(y, y\) + 
        1\/12\ \[Epsilon]\ w\_\(x, x, x, x\) + 
        1\/6\ c\^2\ \[Epsilon]\ w\_\(x, x, y, y\) + 
        1\/12\ c\^4\ \[Epsilon]\ w\_\(y, y, y, y\) == 0\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "We use the method of multiple scales to directly attack ",
  StyleBox["EOM1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["BC1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ". To transform the time derivatives in ",
  StyleBox["EOM1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in terms of the scales ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_1\)]],
  ", we let"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(multiScales = {
        \((dv : u | v | w)\)[x_, y_, t] -> dv[x, y, T\_0, T\_1], 
        \(\(Derivative[m_, n_, o_]\)[u_]\)[x_, y_, t] -> 
          \(dt[o]\)[D[u[x, y, T\_0, T\_1], {x, m}, {y, n}]]}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
Then, we seek a second-order approximate solution in the form\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(solRule = {
        \((dv : u | v)\) -> 
          \((Evaluate[
                Sum[\[Epsilon]\^\(j + 1\)\ dv\_j[#1, #2, #3, #4], {j, 2}]]\ &)
            \), w -> 
          \((Evaluate[Sum[\[Epsilon]\^j\ w\_j[#1, #2, #3, #4], {j, 3}]]\ &)\)}
        \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({dv : u | v \[Rule] 
        \((\[Epsilon]\^2\ dv\_1[#1, #2, #3, #4] + 
              \[Epsilon]\^3\ dv\_2[#1, #2, #3, #4]&)\), 
      w \[Rule] 
        \((\[Epsilon]\ w\_1[#1, #2, #3, #4] + 
              \[Epsilon]\^2\ w\_2[#1, #2, #3, #4] + 
              \[Epsilon]\^3\ w\_3[#1, #2, #3, #4]&)\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where #1, #2, #3, and #4 stand for ",
  Cell[BoxData[
      \(TraditionalForm\`x, y, T\_0, \)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_1\)]],
  ", respectively."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "We introduce the detuning ",
  Cell[BoxData[
      \(TraditionalForm\`\[Sigma]\_1\)]],
  " to describe the nearness of the edge lengths ",
  Cell[BoxData[
      \(TraditionalForm\`a\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`b\)]],
  ", consider the case of primary resonance, and hence let"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(scaleRule2 = {c -> 1 + \[Epsilon]\ \[Sigma]\_1, 
        F -> \[Epsilon]\^3\ F[x, y]\ Cos[\[CapitalOmega]\ T\_0]}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting ",
  StyleBox["multiScales",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", ",
  StyleBox["solRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", and ",
  StyleBox["scaleRule2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["EOM1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["BC1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", expanding the result for small \[Epsilon], and discarding terms of order \
higher than ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\^3\)]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(eq83a = 
      \((\(\(\(EOM1 /. multiScales\) /. solRule\) /. scaleRule2 // TrigToExp
              \) // ExpandAll)\) /. \[Epsilon]\^\(n_ /; n > 3\) -> 0; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(bc83a = \(BC1 /. multiScales\) /. solRule; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Equating coefficients of like powers of \[Epsilon] in ",
  StyleBox["eq83a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["bc83a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(eqEps = 
      \(\(\(Thread[CoefficientList[Subtract\ @@\ #, \[Epsilon]] == 0]\ &\)\ /@
              \ eq83a // Transpose\) // Rest\) // Rest; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(bcEpsuv = 
      \(\(\(Thread[CoefficientList[Subtract\ @@\ #, \[Epsilon]] == 0]\ &\)\ /@
              \ Select[bc83a, FreeQ[#, w]\ &] // Transpose\) // Rest\) // Rest
        \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({{u\_1[0, y, T\_0, T\_1] == 0, u\_1[1, y, T\_0, T\_1] == 0, 
        v\_1[x, 0, T\_0, T\_1] == 0, v\_1[x, 1, T\_0, T\_1] == 0}, {
        u\_2[0, y, T\_0, T\_1] == 0, u\_2[1, y, T\_0, T\_1] == 0, 
        v\_2[x, 0, T\_0, T\_1] == 0, v\_2[x, 1, T\_0, T\_1] == 0}}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(bcEpsw = 
      \(\(Thread[CoefficientList[Subtract\ @@\ #, \[Epsilon]] == 0]\ &\)\ /@\ 
            Select[bc83a, \(! FreeQ[#, w]\)\ &] // Transpose\) // Rest\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{"{", 
          RowBox[{\(w\_1[0, y, T\_0, T\_1] == 0\), ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["w", "1", 
                  TagBox[\((2, 0, 0, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(0, y, T\_0, T\_1\), "]"}], 
              "==", "0"}], ",", \(w\_1[1, y, T\_0, T\_1] == 0\), ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["w", "1", 
                  TagBox[\((2, 0, 0, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(1, y, T\_0, T\_1\), "]"}], 
              "==", "0"}], ",", \(w\_1[x, 0, T\_0, T\_1] == 0\), ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["w", "1", 
                  TagBox[\((0, 2, 0, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(x, 0, T\_0, T\_1\), "]"}], 
              "==", "0"}], ",", \(w\_1[x, 1, T\_0, T\_1] == 0\), ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["w", "1", 
                  TagBox[\((0, 2, 0, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(x, 1, T\_0, T\_1\), "]"}], 
              "==", "0"}]}], "}"}], ",", 
        RowBox[{"{", 
          RowBox[{\(w\_2[0, y, T\_0, T\_1] == 0\), ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["w", "2", 
                  TagBox[\((2, 0, 0, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(0, y, T\_0, T\_1\), "]"}], 
              "==", "0"}], ",", \(w\_2[1, y, T\_0, T\_1] == 0\), ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["w", "2", 
                  TagBox[\((2, 0, 0, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(1, y, T\_0, T\_1\), "]"}], 
              "==", "0"}], ",", \(w\_2[x, 0, T\_0, T\_1] == 0\), ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["w", "2", 
                  TagBox[\((0, 2, 0, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(x, 0, T\_0, T\_1\), "]"}], 
              "==", "0"}], ",", \(w\_2[x, 1, T\_0, T\_1] == 0\), ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["w", "2", 
                  TagBox[\((0, 2, 0, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(x, 1, T\_0, T\_1\), "]"}], 
              "==", "0"}]}], "}"}], ",", 
        RowBox[{"{", 
          RowBox[{\(w\_3[0, y, T\_0, T\_1] == 0\), ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["w", "3", 
                  TagBox[\((2, 0, 0, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(0, y, T\_0, T\_1\), "]"}], 
              "==", "0"}], ",", \(w\_3[1, y, T\_0, T\_1] == 0\), ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["w", "3", 
                  TagBox[\((2, 0, 0, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(1, y, T\_0, T\_1\), "]"}], 
              "==", "0"}], ",", \(w\_3[x, 0, T\_0, T\_1] == 0\), ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["w", "3", 
                  TagBox[\((0, 2, 0, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(x, 0, T\_0, T\_1\), "]"}], 
              "==", "0"}], ",", \(w\_3[x, 1, T\_0, T\_1] == 0\), ",", 
            RowBox[{
              RowBox[{
                SubsuperscriptBox["w", "3", 
                  TagBox[\((0, 2, 0, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(x, 1, T\_0, T\_1\), "]"}], 
              "==", "0"}]}], "}"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["Next, we introduce the display rule", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(displayRule1 = {
        \(\(Derivative[a_, b_, c__]\)[w_\_i_]\)[x_, y_, T\_0, __] :> 
          SequenceForm[
            If[arg1 = 
                Times\ @@\ MapIndexed[D\_\(#2[\([1]\)] - 1\)\%#1\ &, {c}]; 
              arg1 =!= 1, arg1, "\<\>"], 
            Subscript[w, 
              Sequence\ @@\ Join[{i}, Table[x, {a}], Table[y, {b}]]]], 
        \(\(Derivative[a__]\)[A\_i__]\)[__] :> 
          SequenceForm[Times\ @@\ MapIndexed[D\_\(#2[\([1]\)]\)\%#1\ &, {a}], 
            A\_i], \(\(Derivative[a__]\)[A\&_\_i__]\)[__] :> 
          SequenceForm[Times\ @@\ MapIndexed[D\_\(#2[\([1]\)]\)\%#1\ &, {a}], 
            A\&_\_i], w_\_i_[__, T\_0, __] -> w\_i, A\_i__[__] -> A\_i, 
        A\&_\_i__[__] -> A\&_\_i}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "and write ",
  StyleBox["eqEps",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in a more concise way as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((eqEpsm = {eqEps[\([1, 3]\)], eqEps[\([1, {1, 2}]\)], 
            eqEps[\([2, 3]\)]})\) /. displayRule1\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            RowBox[{\(1\/12\), " ", 
              RowBox[{"(", 
                
                InterpretationBox[
                  \(""\[InvisibleSpace]w\_\(1, x, x, x, x\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, x, x, x, x]],
                  Editable->False], ")"}]}], "+", 
            RowBox[{\(1\/6\), " ", 
              RowBox[{"(", 
                
                InterpretationBox[
                  \(""\[InvisibleSpace]w\_\(1, x, x, y, y\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, x, x, y, y]],
                  Editable->False], ")"}]}], "+", 
            RowBox[{\(1\/12\), " ", 
              RowBox[{"(", 
                
                InterpretationBox[
                  \(""\[InvisibleSpace]w\_\(1, y, y, y, y\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, y, y, y, y]],
                  Editable->False], ")"}]}], "+", 
            RowBox[{\(1\/12\), " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_0\%2\[InvisibleSpace]w\_1\),
                  SequenceForm[ 
                    Power[ 
                      Subscript[ D, 0], 2], 
                    Subscript[ w, 1]],
                  Editable->False], ")"}]}]}], "==", "0"}], ",", 
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
                InterpretationBox[\(""\[InvisibleSpace]u\_\(1, x, x\)\),
                  SequenceForm[ "", 
                    Subscript[ u, 1, x, x]],
                  Editable->False], "+", 
                RowBox[{\(1\/2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]u\_\(1, y, y\)\),
                      SequenceForm[ "", 
                        Subscript[ u, 1, y, y]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{\(1\/2\), " ", "\[Nu]", " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]u\_\(1, y, y\)\),
                      SequenceForm[ "", 
                        Subscript[ u, 1, y, y]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{\(1\/2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]v\_\(1, x, y\)\),
                      SequenceForm[ "", 
                        Subscript[ v, 1, x, y]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{\(1\/2\), " ", "\[Nu]", " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]v\_\(1, x, y\)\),
                      SequenceForm[ "", 
                        Subscript[ v, 1, x, y]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x\)\),
                      SequenceForm[ "", 
                        Subscript[ w, 1, x]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x, x\)\),
                      SequenceForm[ "", 
                        Subscript[ w, 1, x, x]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{\(1\/2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]w\_\(1, y\)\),
                      SequenceForm[ "", 
                        Subscript[ w, 1, y]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x, y\)\),
                      SequenceForm[ "", 
                        Subscript[ w, 1, x, y]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{\(1\/2\), " ", "\[Nu]", " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]w\_\(1, y\)\),
                      SequenceForm[ "", 
                        Subscript[ w, 1, y]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x, y\)\),
                      SequenceForm[ "", 
                        Subscript[ w, 1, x, y]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{\(1\/2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x\)\),
                      SequenceForm[ "", 
                        Subscript[ w, 1, x]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]w\_\(1, y, y\)\),
                      SequenceForm[ "", 
                        Subscript[ w, 1, y, y]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{\(1\/2\), " ", "\[Nu]", " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x\)\),
                      SequenceForm[ "", 
                        Subscript[ w, 1, x]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]w\_\(1, y, y\)\),
                      SequenceForm[ "", 
                        Subscript[ w, 1, y, y]],
                      Editable->False], ")"}]}]}], "==", "0"}], ",", 
            RowBox[{
              RowBox[{
                RowBox[{\(1\/2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]u\_\(1, x, y\)\),
                      SequenceForm[ "", 
                        Subscript[ u, 1, x, y]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{\(1\/2\), " ", "\[Nu]", " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]u\_\(1, x, y\)\),
                      SequenceForm[ "", 
                        Subscript[ u, 1, x, y]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{\(1\/2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]v\_\(1, x, x\)\),
                      SequenceForm[ "", 
                        Subscript[ v, 1, x, x]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{\(1\/2\), " ", "\[Nu]", " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]v\_\(1, x, x\)\),
                      SequenceForm[ "", 
                        Subscript[ v, 1, x, x]],
                      Editable->False], ")"}]}], "+", 
                InterpretationBox[\(""\[InvisibleSpace]v\_\(1, y, y\)\),
                  SequenceForm[ "", 
                    Subscript[ v, 1, y, y]],
                  Editable->False], "+", 
                RowBox[{\(1\/2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]w\_\(1, y\)\),
                      SequenceForm[ "", 
                        Subscript[ w, 1, y]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x, x\)\),
                      SequenceForm[ "", 
                        Subscript[ w, 1, x, x]],
                      Editable->False], ")"}]}], "-", 
                RowBox[{\(1\/2\), " ", "\[Nu]", " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]w\_\(1, y\)\),
                      SequenceForm[ "", 
                        Subscript[ w, 1, y]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x, x\)\),
                      SequenceForm[ "", 
                        Subscript[ w, 1, x, x]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{\(1\/2\), " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x\)\),
                      SequenceForm[ "", 
                        Subscript[ w, 1, x]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x, y\)\),
                      SequenceForm[ "", 
                        Subscript[ w, 1, x, y]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{\(1\/2\), " ", "\[Nu]", " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x\)\),
                      SequenceForm[ "", 
                        Subscript[ w, 1, x]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x, y\)\),
                      SequenceForm[ "", 
                        Subscript[ w, 1, x, y]],
                      Editable->False], ")"}]}], "+", 
                RowBox[{
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]w\_\(1, y\)\),
                      SequenceForm[ "", 
                        Subscript[ w, 1, y]],
                      Editable->False], ")"}], " ", 
                  RowBox[{"(", 
                    InterpretationBox[\(""\[InvisibleSpace]w\_\(1, y, y\)\),
                      SequenceForm[ "", 
                        Subscript[ w, 1, y, y]],
                      Editable->False], ")"}]}]}], "==", "0"}]}], "}"}], 
        ",", 
        RowBox[{
          RowBox[{
          \(\(-\(1\/24\)\)\ E\^\(\(-I\)\ T\_0\ \[CapitalOmega]\)\ F[x, y]\), 
            "-", \(1\/24\ E\^\(I\ T\_0\ \[CapitalOmega]\)\ F[x, y]\), "-", 
            RowBox[{
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]u\_\(1, x\)\),
                  SequenceForm[ "", 
                    Subscript[ u, 1, x]],
                  Editable->False], ")"}], " ", 
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x, x\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, x, x]],
                  Editable->False], ")"}]}], "-", 
            RowBox[{"\[Nu]", " ", 
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]v\_\(1, y\)\),
                  SequenceForm[ "", 
                    Subscript[ v, 1, y]],
                  Editable->False], ")"}], " ", 
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x, x\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, x, x]],
                  Editable->False], ")"}]}], "-", 
            RowBox[{\(1\/2\), " ", 
              SuperscriptBox[
                RowBox[{"(", 
                  InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x\)\),
                    SequenceForm[ "", 
                      Subscript[ w, 1, x]],
                    Editable->False], ")"}], "2"], " ", 
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x, x\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, x, x]],
                  Editable->False], ")"}]}], "-", 
            RowBox[{\(1\/2\), " ", "\[Nu]", " ", 
              SuperscriptBox[
                RowBox[{"(", 
                  InterpretationBox[\(""\[InvisibleSpace]w\_\(1, y\)\),
                    SequenceForm[ "", 
                      Subscript[ w, 1, y]],
                    Editable->False], ")"}], "2"], " ", 
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x, x\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, x, x]],
                  Editable->False], ")"}]}], "-", 
            RowBox[{
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]u\_\(1, y\)\),
                  SequenceForm[ "", 
                    Subscript[ u, 1, y]],
                  Editable->False], ")"}], " ", 
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x, y\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, x, y]],
                  Editable->False], ")"}]}], "+", 
            RowBox[{"\[Nu]", " ", 
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]u\_\(1, y\)\),
                  SequenceForm[ "", 
                    Subscript[ u, 1, y]],
                  Editable->False], ")"}], " ", 
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x, y\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, x, y]],
                  Editable->False], ")"}]}], "-", 
            RowBox[{
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]v\_\(1, x\)\),
                  SequenceForm[ "", 
                    Subscript[ v, 1, x]],
                  Editable->False], ")"}], " ", 
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x, y\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, x, y]],
                  Editable->False], ")"}]}], "+", 
            RowBox[{"\[Nu]", " ", 
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]v\_\(1, x\)\),
                  SequenceForm[ "", 
                    Subscript[ v, 1, x]],
                  Editable->False], ")"}], " ", 
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x, y\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, x, y]],
                  Editable->False], ")"}]}], "-", 
            RowBox[{
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, x]],
                  Editable->False], ")"}], " ", 
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]w\_\(1, y\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, y]],
                  Editable->False], ")"}], " ", 
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x, y\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, x, y]],
                  Editable->False], ")"}]}], "+", 
            RowBox[{"\[Nu]", " ", 
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, x]],
                  Editable->False], ")"}], " ", 
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]w\_\(1, y\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, y]],
                  Editable->False], ")"}], " ", 
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x, y\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, x, y]],
                  Editable->False], ")"}]}], "-", 
            RowBox[{"\[Nu]", " ", 
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]u\_\(1, x\)\),
                  SequenceForm[ "", 
                    Subscript[ u, 1, x]],
                  Editable->False], ")"}], " ", 
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]w\_\(1, y, y\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, y, y]],
                  Editable->False], ")"}]}], "-", 
            RowBox[{
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]v\_\(1, y\)\),
                  SequenceForm[ "", 
                    Subscript[ v, 1, y]],
                  Editable->False], ")"}], " ", 
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]w\_\(1, y, y\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, y, y]],
                  Editable->False], ")"}]}], "-", 
            RowBox[{\(1\/2\), " ", "\[Nu]", " ", 
              SuperscriptBox[
                RowBox[{"(", 
                  InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x\)\),
                    SequenceForm[ "", 
                      Subscript[ w, 1, x]],
                    Editable->False], ")"}], "2"], " ", 
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]w\_\(1, y, y\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, y, y]],
                  Editable->False], ")"}]}], "-", 
            RowBox[{\(1\/2\), " ", 
              SuperscriptBox[
                RowBox[{"(", 
                  InterpretationBox[\(""\[InvisibleSpace]w\_\(1, y\)\),
                    SequenceForm[ "", 
                      Subscript[ w, 1, y]],
                    Editable->False], ")"}], "2"], " ", 
              RowBox[{"(", 
                InterpretationBox[\(""\[InvisibleSpace]w\_\(1, y, y\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, y, y]],
                  Editable->False], ")"}]}], "+", 
            RowBox[{\(1\/12\), " ", 
              RowBox[{"(", 
                
                InterpretationBox[
                  \(""\[InvisibleSpace]w\_\(2, x, x, x, x\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 2, x, x, x, x]],
                  Editable->False], ")"}]}], "+", 
            RowBox[{\(1\/6\), " ", 
              RowBox[{"(", 
                
                InterpretationBox[
                  \(""\[InvisibleSpace]w\_\(2, x, x, y, y\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 2, x, x, y, y]],
                  Editable->False], ")"}]}], "+", 
            RowBox[{\(1\/12\), " ", 
              RowBox[{"(", 
                
                InterpretationBox[
                  \(""\[InvisibleSpace]w\_\(2, y, y, y, y\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 2, y, y, y, y]],
                  Editable->False], ")"}]}], "+", 
            RowBox[{\(1\/12\), " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_0\%2\[InvisibleSpace]w\_2\),
                  SequenceForm[ 
                    Power[ 
                      Subscript[ D, 0], 2], 
                    Subscript[ w, 2]],
                  Editable->False], ")"}]}], "+", 
            RowBox[{\(1\/6\), " ", 
              RowBox[{"(", 
                InterpretationBox[\(\(D\_0\ D\_1\)\[InvisibleSpace]w\_1\),
                  SequenceForm[ 
                    Times[ 
                      Subscript[ D, 0], 
                      Subscript[ D, 1]], 
                    Subscript[ w, 1]],
                  Editable->False], ")"}]}], "+", 
            RowBox[{\(1\/3\), " ", 
              RowBox[{"(", 
                
                InterpretationBox[
                  \(""\[InvisibleSpace]w\_\(1, x, x, y, y\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, x, x, y, y]],
                  Editable->False], ")"}], " ", \(\[Sigma]\_1\)}], "+", 
            RowBox[{\(1\/3\), " ", 
              RowBox[{"(", 
                
                InterpretationBox[
                  \(""\[InvisibleSpace]w\_\(1, y, y, y, y\)\),
                  SequenceForm[ "", 
                    Subscript[ w, 1, y, y, y, y]],
                  Editable->False], ")"}], " ", \(\[Sigma]\_1\)}]}], "==", 
          "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["8.3.1 First-Order Solution", "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Linear vibrations are governed by ",
  StyleBox["eqEpsm[[1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["bcEpsw[[1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  "; that is,"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((order1Eq = eqEpsm[\([1]\)])\) /. displayRule1\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{\(1\/12\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x, x, x, x\)\),
              SequenceForm[ "", 
                Subscript[ w, 1, x, x, x, x]],
              Editable->False], ")"}]}], "+", 
        RowBox[{\(1\/6\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(""\[InvisibleSpace]w\_\(1, x, x, y, y\)\),
              SequenceForm[ "", 
                Subscript[ w, 1, x, x, y, y]],
              Editable->False], ")"}]}], "+", 
        RowBox[{\(1\/12\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(""\[InvisibleSpace]w\_\(1, y, y, y, y\)\),
              SequenceForm[ "", 
                Subscript[ w, 1, y, y, y, y]],
              Editable->False], ")"}]}], "+", 
        RowBox[{\(1\/12\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_0\%2\[InvisibleSpace]w\_1\),
              SequenceForm[ 
                Power[ 
                  Subscript[ D, 0], 2], 
                Subscript[ w, 1]],
              Editable->False], ")"}]}]}], "==", "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(order1BC = bcEpsw[\([1]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(w\_1[0, y, T\_0, T\_1] == 0\), ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["w", "1", 
              TagBox[\((2, 0, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(0, y, T\_0, T\_1\), "]"}], 
          "==", "0"}], ",", \(w\_1[1, y, T\_0, T\_1] == 0\), ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["w", "1", 
              TagBox[\((2, 0, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(1, y, T\_0, T\_1\), "]"}], 
          "==", "0"}], ",", \(w\_1[x, 0, T\_0, T\_1] == 0\), ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["w", "1", 
              TagBox[\((0, 2, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(x, 0, T\_0, T\_1\), "]"}], 
          "==", "0"}], ",", \(w\_1[x, 1, T\_0, T\_1] == 0\), ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["w", "1", 
              TagBox[\((0, 2, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(x, 1, T\_0, T\_1\), "]"}], 
          "==", "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "It follows from ",
  StyleBox["order1Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["order1BC",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " that the linear modeshapes and associated frequencies are given by"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(modeshapes = {
        \[Phi]\_\(m_, n_\) :> \((Sin[m\ \[Pi]\ #1]\ Sin[n\ \[Pi]\ #2]\ &)\)}; 
    \nfrequencies = {\[Omega]\_\(m_, n_\) :> \((m\^2 + n\^2)\) \[Pi]\^2}; 
    \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "To investigate the case of one-to-one internal resonance between the ",
  StyleBox["mn",
    FontSlant->"Italic"],
  "th and ",
  StyleBox["nm",
    FontSlant->"Italic"],
  "th modes (",
  Cell[BoxData[
      \(TraditionalForm\`m \[NotEqual] n\)]],
  "), we let"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(w1sol = {
        w\_1 -> Function[{x, y, T\_0, T\_1}, 
            A\_1[T\_1]\ E\^\(I\ \[Omega]\_\(m, n\)\ T\_0\)\ 
                \[Phi]\_\(m, n\)[x, y] + 
              A\&_\_1[T\_1]\ E\^\(\(-I\)\ \[Omega]\_\(m, n\)\ T\_0\)\ 
                \[Phi]\_\(m, n\)[x, y] + 
              A\_2[T\_1]\ E\^\(I\ \[Omega]\_\(m, n\)\ T\_0\)\ 
                \[Phi]\_\(n, m\)[x, y] + 
              A\&_\_2[T\_1]\ E\^\(\(-I\)\ \[Omega]\_\(m, n\)\ T\_0\)\ 
                \[Phi]\_\(n, m\)[x, y]]}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["8.3.2 Second-Order Solution", "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting the ",
  StyleBox["w1sol",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["eqEpsm[[2]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(order2Equ = 
        \((eqEpsm[\([2, 1, 1]\)] /. w\_1 -> \((0\ &)\))\) == 
          \((\(\(\((eqEpsm[\([2, 1, 1]\)] /. w\_1 -> \((0\ &)\))\) - 
                      eqEpsm[\([2, 1, 1]\)] /. w1sol\) /. modeshapes // 
                TrigReduce\) // Expand)\); \) // Timing\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({4.38999999999998635`\ Second, Null}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(order2Eqv = 
        \((eqEpsm[\([2, 2, 1]\)] /. w\_1 -> \((0\ &)\))\) == 
          \((\(\(\((eqEpsm[\([2, 2, 1]\)] /. w\_1 -> \((0\ &)\))\) - 
                      eqEpsm[\([2, 2, 1]\)] /. w1sol\) /. modeshapes // 
                TrigReduce\) // Expand)\); \) // Timing\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({4.28000000000000113`\ Second, Null}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["The associated boundary conditions are", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order2BC = bcEpsuv[\([1]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({u\_1[0, y, T\_0, T\_1] == 0, u\_1[1, y, T\_0, T\_1] == 0, 
      v\_1[x, 0, T\_0, T\_1] == 0, v\_1[x, 1, T\_0, T\_1] == 0}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To obtain particular solutions for ",
  Cell[BoxData[
      \(TraditionalForm\`u\_1\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`v\_1\)]],
  ", we use the method of undetermined coefficients. To accomplish this, we \
first look for all possible terms that appear on the right-hand sides of ",
  StyleBox["order2Equ",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["order2Eqv",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " as follows:"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(possibleTerms1 = 
      Cases[{order2Equ[\([2]\)], order2Eqv[\([2]\)]}, _\ Sin[a_]\ E\^b_ :> 
            Sin[a]\ E\^b, \[Infinity]] // Union; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(possibleTerms2 = possibleTerms1 /. Exp[_] -> 1 // Union; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(possibleTerms = Join[possibleTerms1, possibleTerms2]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ Sin[2\ m\ \[Pi]\ x], 
      E\^\(2\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ Sin[2\ m\ \[Pi]\ x], 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ Sin[2\ n\ \[Pi]\ x], 
      E\^\(2\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ Sin[2\ n\ \[Pi]\ x], 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ Sin[2\ m\ \[Pi]\ y], 
      E\^\(2\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ Sin[2\ m\ \[Pi]\ y], 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ Sin[2\ n\ \[Pi]\ y], 
      E\^\(2\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ Sin[2\ n\ \[Pi]\ y], 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[2\ n\ \[Pi]\ x - 2\ m\ \[Pi]\ y], 
      E\^\(2\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[2\ n\ \[Pi]\ x - 2\ m\ \[Pi]\ y], 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[2\ n\ \[Pi]\ x + 2\ m\ \[Pi]\ y], 
      E\^\(2\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[2\ n\ \[Pi]\ x + 2\ m\ \[Pi]\ y], 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[2\ m\ \[Pi]\ x - 2\ n\ \[Pi]\ y], 
      E\^\(2\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[2\ m\ \[Pi]\ x - 2\ n\ \[Pi]\ y], 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y], 
      E\^\(2\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y], 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y], 
      E\^\(2\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y], 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y], 
      E\^\(2\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y], 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y], 
      E\^\(2\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y], 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y], 
      E\^\(2\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y], 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y], 
      E\^\(2\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y], 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y], 
      E\^\(2\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y], 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y], 
      E\^\(2\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y], 
      E\^\(\(-2\)\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[2\ m\ \[Pi]\ x + 2\ n\ \[Pi]\ y], 
      E\^\(2\ I\ T\_0\ \[Omega]\_\(m, n\)\)\ 
        Sin[2\ m\ \[Pi]\ x + 2\ n\ \[Pi]\ y], Sin[2\ m\ \[Pi]\ x], 
      Sin[2\ n\ \[Pi]\ x], Sin[2\ m\ \[Pi]\ y], Sin[2\ n\ \[Pi]\ y], 
      Sin[2\ n\ \[Pi]\ x - 2\ m\ \[Pi]\ y], 
      Sin[2\ n\ \[Pi]\ x + 2\ m\ \[Pi]\ y], 
      Sin[2\ m\ \[Pi]\ x - 2\ n\ \[Pi]\ y], 
      Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y], 
      Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y], 
      Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y], 
      Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y], 
      Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y], 
      Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y], 
      Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y], 
      Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y], 
      Sin[2\ m\ \[Pi]\ x + 2\ n\ \[Pi]\ y]}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "which satisfy ",
  StyleBox["order2BC",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ". We then define the following symbol lists to represent the undetermined \
coefficients."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(usymbolList = Table[c\_i, {i, Length[possibleTerms]}]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({c\_1, c\_2, c\_3, c\_4, c\_5, c\_6, c\_7, c\_8, c\_9, c\_10, c\_11, 
      c\_12, c\_13, c\_14, c\_15, c\_16, c\_17, c\_18, c\_19, c\_20, c\_21, 
      c\_22, c\_23, c\_24, c\_25, c\_26, c\_27, c\_28, c\_29, c\_30, c\_31, 
      c\_32, c\_33, c\_34, c\_35, c\_36, c\_37, c\_38, c\_39, c\_40, c\_41, 
      c\_42, c\_43, c\_44, c\_45, c\_46, c\_47, c\_48}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(vsymbolList = usymbolList /. c -> d\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({d\_1, d\_2, d\_3, d\_4, d\_5, d\_6, d\_7, d\_8, d\_9, d\_10, d\_11, 
      d\_12, d\_13, d\_14, d\_15, d\_16, d\_17, d\_18, d\_19, d\_20, d\_21, 
      d\_22, d\_23, d\_24, d\_25, d\_26, d\_27, d\_28, d\_29, d\_30, d\_31, 
      d\_32, d\_33, d\_34, d\_35, d\_36, d\_37, d\_38, d\_39, d\_40, d\_41, 
      d\_42, d\_43, d\_44, d\_45, d\_46, d\_47, d\_48}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["Then, we express the solution as", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(u1sol = 
      u\_1 -> Function[{x, y, T\_0, T\_1}, 
          usymbolList . possibleTerms // Evaluate]; \n
    v1sol = v\_1 -> 
        Function[{x, y, T\_0, T\_1}, 
          vsymbolList . possibleTerms // Evaluate]; \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting ",
  StyleBox["u1sol",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["v1sol",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["order2Equ",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["order2Eqv",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", collecting coefficients of ",
  StyleBox["possibleTerms",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", and solving for the ",
  StyleBox["usymbolList",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["vsymbolList",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(algEq1 = 
        Coefficient[\(Subtract\ @@\ order2Equ /. u1sol\) /. v1sol, 
                possibleTerms] == 0 /. Exp[_\ T\_0] -> 0 // Thread; \) // 
      Timing\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({67.4499999999999921`\ Second, Null}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(algEq2 = 
        Coefficient[\(Subtract\ @@\ order2Eqv /. u1sol\) /. v1sol, 
                possibleTerms] == 0 /. Exp[_\ T\_0] -> 0 // Thread; \) // 
      Timing\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({65.9100000000000285`\ Second, Null}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(symbolRule1 = 
      \(Solve[Join[algEq1, algEq2], Join[usymbolList, vsymbolList]]\)[
          \([1]\)] // Simplify\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({c\_1 \[Rule] 
        \(-\(\(\[Pi]\ \((m\^2 - n\^2\ \[Nu])\)\ A\&_\_1[T\_1]\^2\)\/\(16\ m
                \)\)\), 
      c\_2 \[Rule] 
        \(-\(\(\[Pi]\ \((m\^2 - n\^2\ \[Nu])\)\ A\_1[T\_1]\^2\)\/\(16\ m
                \)\)\), 
      c\_3 \[Rule] 
        \(-\(\(\[Pi]\ \((n\^2 - m\^2\ \[Nu])\)\ A\&_\_2[T\_1]\^2\)\/\(16\ n
                \)\)\), 
      c\_4 \[Rule] 
        \(-\(\(\[Pi]\ \((n\^2 - m\^2\ \[Nu])\)\ A\_2[T\_1]\^2\)\/\(16\ n
                \)\)\), 
      c\_33 \[Rule] 
        \(-\(\(\[Pi]\ \((m\^2 - n\^2\ \[Nu])\)\ A\_1[T\_1]\ 
                A\&_\_1[T\_1]\)\/\(8\ m\)\)\), 
      c\_34 \[Rule] 
        \(-\(\(\[Pi]\ \((n\^2 - m\^2\ \[Nu])\)\ A\_2[T\_1]\ 
                A\&_\_2[T\_1]\)\/\(8\ n\)\)\), 
      d\_5 \[Rule] 
        \(-\(\(\[Pi]\ \((m\^2 - n\^2\ \[Nu])\)\ A\&_\_2[T\_1]\^2\)\/\(16\ m
                \)\)\), 
      d\_6 \[Rule] 
        \(-\(\(\[Pi]\ \((m\^2 - n\^2\ \[Nu])\)\ A\_2[T\_1]\^2\)\/\(16\ m
                \)\)\), 
      d\_7 \[Rule] 
        \(-\(\(\[Pi]\ \((n\^2 - m\^2\ \[Nu])\)\ A\&_\_1[T\_1]\^2\)\/\(16\ n
                \)\)\), 
      d\_8 \[Rule] 
        \(-\(\(\[Pi]\ \((n\^2 - m\^2\ \[Nu])\)\ A\_1[T\_1]\^2\)\/\(16\ n
                \)\)\), 
      d\_35 \[Rule] 
        \(-\(\(\[Pi]\ \((m\^2 - n\^2\ \[Nu])\)\ A\_2[T\_1]\ 
                A\&_\_2[T\_1]\)\/\(8\ m\)\)\), 
      d\_36 \[Rule] 
        \(-\(\(\[Pi]\ \((n\^2 - m\^2\ \[Nu])\)\ A\_1[T\_1]\ 
                A\&_\_1[T\_1]\)\/\(8\ n\)\)\), c\_5 \[Rule] 0, 
      c\_6 \[Rule] 0, c\_7 \[Rule] 0, c\_8 \[Rule] 0, c\_35 \[Rule] 0, 
      c\_36 \[Rule] 0, d\_1 \[Rule] 0, d\_2 \[Rule] 0, d\_3 \[Rule] 0, 
      d\_4 \[Rule] 0, d\_33 \[Rule] 0, d\_34 \[Rule] 0, 
      c\_9 \[Rule] 1\/32\ n\ \[Pi]\ A\&_\_2[T\_1]\^2, 
      c\_10 \[Rule] 1\/32\ n\ \[Pi]\ A\_2[T\_1]\^2, 
      c\_11 \[Rule] 1\/32\ n\ \[Pi]\ A\&_\_2[T\_1]\^2, 
      c\_12 \[Rule] 1\/32\ n\ \[Pi]\ A\_2[T\_1]\^2, 
      c\_13 \[Rule] 1\/32\ m\ \[Pi]\ A\&_\_1[T\_1]\^2, 
      c\_14 \[Rule] 1\/32\ m\ \[Pi]\ A\_1[T\_1]\^2, 
      c\_31 \[Rule] 1\/32\ m\ \[Pi]\ A\&_\_1[T\_1]\^2, 
      c\_32 \[Rule] 1\/32\ m\ \[Pi]\ A\_1[T\_1]\^2, 
      c\_37 \[Rule] 1\/16\ n\ \[Pi]\ A\_2[T\_1]\ A\&_\_2[T\_1], 
      c\_38 \[Rule] 1\/16\ n\ \[Pi]\ A\_2[T\_1]\ A\&_\_2[T\_1], 
      c\_39 \[Rule] 1\/16\ m\ \[Pi]\ A\_1[T\_1]\ A\&_\_1[T\_1], 
      c\_48 \[Rule] 1\/16\ m\ \[Pi]\ A\_1[T\_1]\ A\&_\_1[T\_1], 
      d\_9 \[Rule] \(-\(1\/32\)\)\ m\ \[Pi]\ A\&_\_2[T\_1]\^2, 
      d\_10 \[Rule] \(-\(1\/32\)\)\ m\ \[Pi]\ A\_2[T\_1]\^2, 
      d\_11 \[Rule] 1\/32\ m\ \[Pi]\ A\&_\_2[T\_1]\^2, 
      d\_12 \[Rule] 1\/32\ m\ \[Pi]\ A\_2[T\_1]\^2, 
      d\_13 \[Rule] \(-\(1\/32\)\)\ n\ \[Pi]\ A\&_\_1[T\_1]\^2, 
      d\_14 \[Rule] \(-\(1\/32\)\)\ n\ \[Pi]\ A\_1[T\_1]\^2, 
      d\_31 \[Rule] 1\/32\ n\ \[Pi]\ A\&_\_1[T\_1]\^2, 
      d\_32 \[Rule] 1\/32\ n\ \[Pi]\ A\_1[T\_1]\^2, 
      d\_37 \[Rule] \(-\(1\/16\)\)\ m\ \[Pi]\ A\_2[T\_1]\ A\&_\_2[T\_1], 
      d\_38 \[Rule] 1\/16\ m\ \[Pi]\ A\_2[T\_1]\ A\&_\_2[T\_1], 
      d\_39 \[Rule] \(-\(1\/16\)\)\ n\ \[Pi]\ A\_1[T\_1]\ A\&_\_1[T\_1], 
      d\_48 \[Rule] 1\/16\ n\ \[Pi]\ A\_1[T\_1]\ A\&_\_1[T\_1], 
      c\_15 \[Rule] 
        1\/32\ \((m - n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ A\&_\_1[T\_1]\ 
          A\&_\_2[T\_1], 
      c\_16 \[Rule] 
        1\/32\ \((m - n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ A\_1[T\_1]\ 
          A\_2[T\_1], 
      c\_21 \[Rule] 
        1\/32\ \((m + n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ A\&_\_1[T\_1]\ 
          A\&_\_2[T\_1], 
      c\_22 \[Rule] 
        1\/32\ \((m + n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ A\_1[T\_1]\ 
          A\_2[T\_1], 
      c\_25 \[Rule] 
        1\/32\ \((m + n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ A\&_\_1[T\_1]\ 
          A\&_\_2[T\_1], 
      c\_26 \[Rule] 
        1\/32\ \((m + n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ A\_1[T\_1]\ 
          A\_2[T\_1], 
      c\_27 \[Rule] 
        1\/32\ \((m - n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ A\&_\_1[T\_1]\ 
          A\&_\_2[T\_1], 
      c\_28 \[Rule] 
        1\/32\ \((m - n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ A\_1[T\_1]\ 
          A\_2[T\_1], 
      c\_40 \[Rule] 
        1\/32\ \((m - n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ 
          \((A\_2[T\_1]\ A\&_\_1[T\_1] + A\_1[T\_1]\ A\&_\_2[T\_1])\), 
      c\_43 \[Rule] 
        1\/32\ \((m + n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ 
          \((A\_2[T\_1]\ A\&_\_1[T\_1] + A\_1[T\_1]\ A\&_\_2[T\_1])\), 
      c\_45 \[Rule] 
        1\/32\ \((m + n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ 
          \((A\_2[T\_1]\ A\&_\_1[T\_1] + A\_1[T\_1]\ A\&_\_2[T\_1])\), 
      c\_46 \[Rule] 
        1\/32\ \((m - n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ 
          \((A\_2[T\_1]\ A\&_\_1[T\_1] + A\_1[T\_1]\ A\&_\_2[T\_1])\), 
      d\_15 \[Rule] 
        \(-\(1\/32\)\)\ \((m + n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ 
          A\&_\_1[T\_1]\ A\&_\_2[T\_1], 
      d\_16 \[Rule] 
        \(-\(1\/32\)\)\ \((m + n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ A\_1[T\_1]\ 
          A\_2[T\_1], 
      d\_21 \[Rule] 
        1\/32\ \((m - n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ A\&_\_1[T\_1]\ 
          A\&_\_2[T\_1], 
      d\_22 \[Rule] 
        1\/32\ \((m - n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ A\_1[T\_1]\ 
          A\_2[T\_1], 
      d\_25 \[Rule] 
        \(-\(1\/32\)\)\ \((m - n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ 
          A\&_\_1[T\_1]\ A\&_\_2[T\_1], 
      d\_26 \[Rule] 
        \(-\(1\/32\)\)\ \((m - n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ A\_1[T\_1]\ 
          A\_2[T\_1], 
      d\_27 \[Rule] 
        1\/32\ \((m + n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ A\&_\_1[T\_1]\ 
          A\&_\_2[T\_1], 
      d\_28 \[Rule] 
        1\/32\ \((m + n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ A\_1[T\_1]\ 
          A\_2[T\_1], 
      d\_40 \[Rule] 
        \(-\(1\/32\)\)\ \((m + n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ 
          \((A\_2[T\_1]\ A\&_\_1[T\_1] + A\_1[T\_1]\ A\&_\_2[T\_1])\), 
      d\_43 \[Rule] 
        1\/32\ \((m - n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ 
          \((A\_2[T\_1]\ A\&_\_1[T\_1] + A\_1[T\_1]\ A\&_\_2[T\_1])\), 
      d\_45 \[Rule] 
        \(-\(1\/32\)\)\ \((m - n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ 
          \((A\_2[T\_1]\ A\&_\_1[T\_1] + A\_1[T\_1]\ A\&_\_2[T\_1])\), 
      d\_46 \[Rule] 
        1\/32\ \((m + n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)\ 
          \((A\_2[T\_1]\ A\&_\_1[T\_1] + A\_1[T\_1]\ A\&_\_2[T\_1])\), 
      c\_17 \[Rule] 
        \(-\(\(\[Pi]\ 
                \((m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) - 
                    2\ m\ n\ \((1 + \[Nu])\))\)\ A\&_\_1[T\_1]\ 
                A\&_\_2[T\_1]\)\/\(32\ \((m + n)\)\)\)\), 
      c\_18 \[Rule] 
        \(-\(\(\[Pi]\ 
                \((m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) - 
                    2\ m\ n\ \((1 + \[Nu])\))\)\ A\_1[T\_1]\ 
                A\_2[T\_1]\)\/\(32\ \((m + n)\)\)\)\), 
      c\_19 \[Rule] 
        \(-\(\(\[Pi]\ 
                \((m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) + 
                    2\ m\ n\ \((1 + \[Nu])\))\)\ A\&_\_1[T\_1]\ 
                A\&_\_2[T\_1]\)\/\(32\ \((m - n)\)\)\)\), 
      c\_20 \[Rule] 
        \(-\(\(\[Pi]\ 
                \((m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) + 
                    2\ m\ n\ \((1 + \[Nu])\))\)\ A\_1[T\_1]\ 
                A\_2[T\_1]\)\/\(32\ \((m - n)\)\)\)\), 
      c\_23 \[Rule] 
        \(-\(\(\[Pi]\ 
                \((m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) + 
                    2\ m\ n\ \((1 + \[Nu])\))\)\ A\&_\_1[T\_1]\ 
                A\&_\_2[T\_1]\)\/\(32\ \((m - n)\)\)\)\), 
      c\_24 \[Rule] 
        \(-\(\(\[Pi]\ 
                \((m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) + 
                    2\ m\ n\ \((1 + \[Nu])\))\)\ A\_1[T\_1]\ 
                A\_2[T\_1]\)\/\(32\ \((m - n)\)\)\)\), 
      c\_29 \[Rule] 
        \(-\(\(\[Pi]\ 
                \((m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) - 
                    2\ m\ n\ \((1 + \[Nu])\))\)\ A\&_\_1[T\_1]\ 
                A\&_\_2[T\_1]\)\/\(32\ \((m + n)\)\)\)\), 
      c\_30 \[Rule] 
        \(-\(\(\[Pi]\ 
                \((m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) - 
                    2\ m\ n\ \((1 + \[Nu])\))\)\ A\_1[T\_1]\ 
                A\_2[T\_1]\)\/\(32\ \((m + n)\)\)\)\), 
      c\_41 \[Rule] 
        \(-\(\(\[Pi]\ 
                \((m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) - 
                    2\ m\ n\ \((1 + \[Nu])\))\)\ 
                \((A\_2[T\_1]\ A\&_\_1[T\_1] + A\_1[T\_1]\ A\&_\_2[T\_1])
                  \)\)\/\(32\ \((m + n)\)\)\)\), 
      c\_42 \[Rule] 
        \(-\(\(\[Pi]\ 
                \((m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) + 
                    2\ m\ n\ \((1 + \[Nu])\))\)\ 
                \((A\_2[T\_1]\ A\&_\_1[T\_1] + A\_1[T\_1]\ A\&_\_2[T\_1])
                  \)\)\/\(32\ \((m - n)\)\)\)\), 
      c\_44 \[Rule] 
        \(-\(\(\[Pi]\ 
                \((m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) + 
                    2\ m\ n\ \((1 + \[Nu])\))\)\ 
                \((A\_2[T\_1]\ A\&_\_1[T\_1] + A\_1[T\_1]\ A\&_\_2[T\_1])
                  \)\)\/\(32\ \((m - n)\)\)\)\), 
      c\_47 \[Rule] 
        \(-\(\(\[Pi]\ 
                \((m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) - 
                    2\ m\ n\ \((1 + \[Nu])\))\)\ 
                \((A\_2[T\_1]\ A\&_\_1[T\_1] + A\_1[T\_1]\ A\&_\_2[T\_1])
                  \)\)\/\(32\ \((m + n)\)\)\)\), 
      d\_17 \[Rule] 
        \(\[Pi]\ \((
              m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) - 
                2\ m\ n\ \((1 + \[Nu])\))\)\ A\&_\_1[T\_1]\ 
            A\&_\_2[T\_1]\)\/\(32\ \((m + n)\)\), 
      d\_18 \[Rule] 
        \(\[Pi]\ \((
              m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) - 
                2\ m\ n\ \((1 + \[Nu])\))\)\ A\_1[T\_1]\ A\_2[T\_1]\)\/\(32\ 
            \((m + n)\)\), 
      d\_19 \[Rule] 
        \(-\(\(\[Pi]\ 
                \((m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) + 
                    2\ m\ n\ \((1 + \[Nu])\))\)\ A\&_\_1[T\_1]\ 
                A\&_\_2[T\_1]\)\/\(32\ \((m - n)\)\)\)\), 
      d\_20 \[Rule] 
        \(-\(\(\[Pi]\ 
                \((m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) + 
                    2\ m\ n\ \((1 + \[Nu])\))\)\ A\_1[T\_1]\ 
                A\_2[T\_1]\)\/\(32\ \((m - n)\)\)\)\), 
      d\_23 \[Rule] 
        \(\[Pi]\ \((
              m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) + 
                2\ m\ n\ \((1 + \[Nu])\))\)\ A\&_\_1[T\_1]\ 
            A\&_\_2[T\_1]\)\/\(32\ \((m - n)\)\), 
      d\_24 \[Rule] 
        \(\[Pi]\ \((
              m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) + 
                2\ m\ n\ \((1 + \[Nu])\))\)\ A\_1[T\_1]\ A\_2[T\_1]\)\/\(32\ 
            \((m - n)\)\), 
      d\_29 \[Rule] 
        \(-\(\(\[Pi]\ 
                \((m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) - 
                    2\ m\ n\ \((1 + \[Nu])\))\)\ A\&_\_1[T\_1]\ 
                A\&_\_2[T\_1]\)\/\(32\ \((m + n)\)\)\)\), 
      d\_30 \[Rule] 
        \(-\(\(\[Pi]\ 
                \((m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) - 
                    2\ m\ n\ \((1 + \[Nu])\))\)\ A\_1[T\_1]\ 
                A\_2[T\_1]\)\/\(32\ \((m + n)\)\)\)\), 
      d\_41 \[Rule] 
        \(\[Pi]\ \((
              m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) - 
                2\ m\ n\ \((1 + \[Nu])\))\)\ 
            \((A\_2[T\_1]\ A\&_\_1[T\_1] + A\_1[T\_1]\ A\&_\_2[T\_1])
              \)\)\/\(32\ \((m + n)\)\), 
      d\_42 \[Rule] 
        \(-\(\(\[Pi]\ 
                \((m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) + 
                    2\ m\ n\ \((1 + \[Nu])\))\)\ 
                \((A\_2[T\_1]\ A\&_\_1[T\_1] + A\_1[T\_1]\ A\&_\_2[T\_1])
                  \)\)\/\(32\ \((m - n)\)\)\)\), 
      d\_44 \[Rule] 
        \(\[Pi]\ \((
              m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) + 
                2\ m\ n\ \((1 + \[Nu])\))\)\ 
            \((A\_2[T\_1]\ A\&_\_1[T\_1] + A\_1[T\_1]\ A\&_\_2[T\_1])
              \)\)\/\(32\ \((m - n)\)\), 
      d\_47 \[Rule] 
        \(-\(\(\[Pi]\ 
                \((m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) - 
                    2\ m\ n\ \((1 + \[Nu])\))\)\ 
                \((A\_2[T\_1]\ A\&_\_1[T\_1] + A\_1[T\_1]\ A\&_\_2[T\_1])
                  \)\)\/\(32\ \((m + n)\)\)\)\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["8.3.3 Solvability Conditions", "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting the ",
  StyleBox["w1sol",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", ",
  StyleBox["u1sol",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", and ",
  StyleBox["v1sol",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["eqEpsm[[3]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(order3Eq = 
        \((order1Eq[\([1]\)] /. w\_1 -> w\_2)\) == 
          \((\(\(\(\((order1Eq[\([1]\)] /. w\_1 -> w\_2)\) - 
                        eqEpsm[\([3, 1]\)] /. w1sol\) /. u1sol\) /. v1sol
                  \) /. modeshapes // Expand)\); \) // Timing\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({4.17999999999994997`\ Second, Null}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["The associated boundary conditions are", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order3BC = bcEpsw[\([2]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(w\_2[0, y, T\_0, T\_1] == 0\), ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["w", "2", 
              TagBox[\((2, 0, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(0, y, T\_0, T\_1\), "]"}], 
          "==", "0"}], ",", \(w\_2[1, y, T\_0, T\_1] == 0\), ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["w", "2", 
              TagBox[\((2, 0, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(1, y, T\_0, T\_1\), "]"}], 
          "==", "0"}], ",", \(w\_2[x, 0, T\_0, T\_1] == 0\), ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["w", "2", 
              TagBox[\((0, 2, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(x, 0, T\_0, T\_1\), "]"}], 
          "==", "0"}], ",", \(w\_2[x, 1, T\_0, T\_1] == 0\), ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["w", "2", 
              TagBox[\((0, 2, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(x, 1, T\_0, T\_1\), "]"}], 
          "==", "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To describe the nearness of the primary resonance, we introduce the \
detuning parameter ",
  Cell[BoxData[
      \(TraditionalForm\`\[Sigma]\_2\)]],
  " defined by"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ResonanceCond = {
        \[CapitalOmega] -> \[Omega]\_\(m, n\) + \[Epsilon]\ \[Sigma]\_2}; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Because the homogeneous parts of ",
  StyleBox["order3Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  StyleBox[" ",
    FontColor->RGBColor[1, 0, 1]],
  "and",
  StyleBox[" ",
    FontColor->RGBColor[1, 0, 1]],
  StyleBox["order3BC",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  StyleBox[" ",
    FontColor->RGBColor[1, 0, 1]],
  "have a nontrivial solution, the corresponding nonhomogeneous problem has a \
solution only if solvability conditions are satisfied. These conditions \
demand that the right-hand side of ",
  StyleBox["order3Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " be orthogonal to every solution of the adjoint homogeneous problem. In \
this case, the problem is self-adjoint."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "To determine the solvability conditions, we collect the terms that may \
lead to secular terms, the terms proportional to ",
  Cell[BoxData[
      \(E\^\(I\ \[Omega]\_\(m, n\)\ T\_0\)\)]],
  ". To this end, we use the rule"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(expRule = 
      Exp[a_] :> Exp[Expand[a /. ResonanceCond] /. \[Epsilon]\ T\_0 -> T\_1]; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["and obtain", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(ST = 
        Coefficient[order3Eq[\([2]\)] /. expRule, 
            E\^\(I\ \[Omega]\_\(m, n\)\ T\_0\)] // TrigReduce; \) // Timing
      \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({132.529999999999992`\ Second, Null}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Among all the spatial-dependent non-forcing terms in ",
  StyleBox["ST",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", only the following terms are not orthogonal to the ",
  StyleBox["modeshapes",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ":"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(terms1 = 
      Cases[ST, Cos[m\ \[Pi]\ x + a_. \ n\ \[Pi]\ y /; Abs[a] == 1], 
          \[Infinity]] // Union\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({Cos[m\ \[Pi]\ x - n\ \[Pi]\ y], Cos[m\ \[Pi]\ x + n\ \[Pi]\ y]}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(terms2 = 
      Cases[ST, Cos[n\ \[Pi]\ x + a_. \ m\ \[Pi]\ y /; Abs[a] == 1], 
          \[Infinity]] // Union\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({Cos[n\ \[Pi]\ x - m\ \[Pi]\ y], Cos[n\ \[Pi]\ x + m\ \[Pi]\ y]}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The coefficients of these terms in ",
  StyleBox["ST",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " are"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(terms1Coef = Coefficient[ST, terms1]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(terms2Coef = Coefficient[ST, terms2]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "The projection of ",
  StyleBox["terms1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " onto ",
  Cell[BoxData[
      \(TraditionalForm\`Sin[m\ \[Pi]\ x]\ Sin[n\ \[Pi]\ y]\)]],
  " can be obtained as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(trigRule = {Cos[2\ _. \ \[Pi]] -> 1, Sin[_. \ \[Pi]] -> 0}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(proj1Coef = 
      \(\[Integral]\_0\%1
                \(\[Integral]\_0\%1 Sin[m\ \[Pi]\ x]\ Sin[n\ \[Pi]\ y]\ #\ 
                  \[DifferentialD]x \[DifferentialD]y\)\ &\)\ /@\ terms1 /. 
        trigRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({1\/4, \(-\(1\/4\)\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "and the projection of ",
  StyleBox["terms2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " onto ",
  Cell[BoxData[
      \(TraditionalForm\`Sin[n\ \[Pi]\ x]\ Sin[m\ \[Pi]\ y]\)]],
  " can be obtained as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(proj2Coef = 
      \(\[Integral]\_0\%1
                \(\[Integral]\_0\%1 Sin[n\ \[Pi]\ x]\ Sin[m\ \[Pi]\ y]\ #\ 
                  \[DifferentialD]x \[DifferentialD]y\)\ &\)\ /@\ terms2 /. 
        trigRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({1\/4, \(-\(1\/4\)\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The forcing term in ",
  StyleBox["ST",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " is"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(f0 = ST /. Cos[_] -> 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(1\/24\ E\^\(I\ T\_1\ \[Sigma]\_2\)\ F[x, y]\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Then, the solvability conditions demand that ",
  StyleBox["ST",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " be orthogonal to every linear eigenfunction, ",
  StyleBox["modeshapes",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  "; that is,"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(intRule3 = {
        int[a_\ fun_, a1_, b1_] :> 
          a\ int[fun, a1, b1] /; FreeQ[a, First[a1] | First[b1]]}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(SCond1 = 
      Expand[24 
            \((\(\(int[Sin[m\ \[Pi]\ x]\ Sin[n\ \[Pi]\ y]\ f0, {x, 0, 1}, {y, 
                          0, 1}] + proj1Coef . terms1Coef /. symbolRule1
                    \) //. intRule3\) /. int -> Integrate)\)] == 0; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(SCond2 = 
      Expand[24 
            \((\(\(int[Sin[n\ \[Pi]\ x]\ Sin[m\ \[Pi]\ y]\ f0, {x, 0, 1}, {y, 
                          0, 1}] + proj2Coef . terms2Coef /. symbolRule1
                    \) //. intRule3\) /. int -> Integrate)\)] == 0; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
To express these solvability conditions in a more readable form, we define\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(basicTerms = {A\_1[T\_1]\ E\^\(I\ \[Omega]\_\(m, n\)\ T\_0\), 
        A\&_\_1[T\_1]\ E\^\(\(-I\)\ \[Omega]\_\(m, n\)\ T\_0\), 
        A\_2[T\_1]\ E\^\(I\ \[Omega]\_\(m, n\)\ T\_0\), 
        A\&_\_2[T\_1]\ E\^\(\(-I\)\ \[Omega]\_\(m, n\)\ T\_0\)}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(cubicTerms = 
      \(\((E\^\(\(-I\)\ \[Omega]\_\(m, n\)\ T\_0\)\ 
                  Nest[Outer[Times, basicTerms, #]\ &, basicTerms, 2] // 
                Flatten)\) /. Exp[_\ T\_0] -> 0 // Union\) // Rest\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({A\_1[T\_1]\^2\ A\&_\_1[T\_1], A\_1[T\_1]\ A\_2[T\_1]\ A\&_\_1[T\_1], 
      A\_2[T\_1]\^2\ A\&_\_1[T\_1], A\_1[T\_1]\^2\ A\&_\_2[T\_1], 
      A\_1[T\_1]\ A\_2[T\_1]\ A\&_\_2[T\_1], A\_2[T\_1]\^2\ A\&_\_2[T\_1]}
      \)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Then, we collect the coefficients of ",
  StyleBox["cubicTerms",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " from ",
  StyleBox["SCond1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["SCond2 ",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  "and obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(coef1 = Coefficient[SCond1[\([1]\)], cubicTerms] // Simplify\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({9\/8\ \[Pi]\^4\ 
        \((\(-4\)\ m\^2\ n\^2\ \[Nu] + m\^4\ \((\(-3\) + \[Nu]\^2)\) + 
            n\^4\ \((\(-3\) + \[Nu]\^2)\))\), 0, 
      3\/8\ \[Pi]\^4\ 
        \((4\ m\^2\ n\^2\ \((\(-2\) + \[Nu]\^2)\) + 
            m\^4\ \((\(-1\) - 2\ \[Nu] + \[Nu]\^2)\) + 
            n\^4\ \((\(-1\) - 2\ \[Nu] + \[Nu]\^2)\))\), 0, 
      3\/4\ \[Pi]\^4\ 
        \((4\ m\^2\ n\^2\ \((\(-2\) + \[Nu]\^2)\) + 
            m\^4\ \((\(-1\) - 2\ \[Nu] + \[Nu]\^2)\) + 
            n\^4\ \((\(-1\) - 2\ \[Nu] + \[Nu]\^2)\))\), 0}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(coef2 = Coefficient[SCond2[\([1]\)], cubicTerms] // Simplify\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({0, 
      3\/4\ \[Pi]\^4\ 
        \((4\ m\^2\ n\^2\ \((\(-2\) + \[Nu]\^2)\) + 
            m\^4\ \((\(-1\) - 2\ \[Nu] + \[Nu]\^2)\) + 
            n\^4\ \((\(-1\) - 2\ \[Nu] + \[Nu]\^2)\))\), 0, 
      3\/8\ \[Pi]\^4\ 
        \((4\ m\^2\ n\^2\ \((\(-2\) + \[Nu]\^2)\) + 
            m\^4\ \((\(-1\) - 2\ \[Nu] + \[Nu]\^2)\) + 
            n\^4\ \((\(-1\) - 2\ \[Nu] + \[Nu]\^2)\))\), 0, 
      9\/8\ \[Pi]\^4\ 
        \((\(-4\)\ m\^2\ n\^2\ \[Nu] + m\^4\ \((\(-3\) + \[Nu]\^2)\) + 
            n\^4\ \((\(-3\) + \[Nu]\^2)\))\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Hence, ",
  StyleBox["SCond1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["SCond2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " can be rewritten and combined as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(SCond = {
        Collect[SCond1[\([1]\)] /. Thread[cubicTerms -> 0], 
              \[Sigma]\_1\ A\_1[T\_1]] + coef1 . cubicTerms == 0, 
        Collect[SCond2[\([1]\)] /. Thread[cubicTerms -> 0], 
              \[Sigma]\_1\ A\_2[T\_1]] + coef2 . cubicTerms == 0}; \n
    SCond /. displayRule1\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
          \(E\^\(I\ T\_1\ \[Sigma]\_2\)\ 
              \(\[Integral]\_0\%1
                  \(\[Integral]\_0\%1
                    \( F[x, y]\ Sin[m\ \[Pi]\ x]\ Sin[n\ \[Pi]\ y]\) 
                    \[DifferentialD]y \[DifferentialD]x\)\)\), "+", 
            \(\((\(-2\)\ m\^2\ n\^2\ \[Pi]\^4 - 2\ n\^4\ \[Pi]\^4)\)\ A\_1\ 
              \[Sigma]\_1\), "+", 
            \(9\/8\ \[Pi]\^4\ 
              \((\(-4\)\ m\^2\ n\^2\ \[Nu] + m\^4\ \((\(-3\) + \[Nu]\^2)\) + 
                  n\^4\ \((\(-3\) + \[Nu]\^2)\))\)\ A\_1\%2\ A\&_\_1\), "+", 
            \(3\/8\ \[Pi]\^4\ 
              \((4\ m\^2\ n\^2\ \((\(-2\) + \[Nu]\^2)\) + 
                  m\^4\ \((\(-1\) - 2\ \[Nu] + \[Nu]\^2)\) + 
                  n\^4\ \((\(-1\) - 2\ \[Nu] + \[Nu]\^2)\))\)\ A\_2\%2\ 
              A\&_\_1\), "+", 
            \(3\/4\ \[Pi]\^4\ 
              \((4\ m\^2\ n\^2\ \((\(-2\) + \[Nu]\^2)\) + 
                  m\^4\ \((\(-1\) - 2\ \[Nu] + \[Nu]\^2)\) + 
                  n\^4\ \((\(-1\) - 2\ \[Nu] + \[Nu]\^2)\))\)\ A\_1\ A\_2\ 
              A\&_\_2\), "-", 
            RowBox[{"I", " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\_1\),
                  SequenceForm[ 
                    Subscript[ D, 1], 
                    Subscript[ A, 1]],
                  Editable->False], ")"}], " ", \(\[Omega]\_\(m, n\)\)}]}], 
          "==", "0"}], ",", 
        RowBox[{
          RowBox[{
          \(E\^\(I\ T\_1\ \[Sigma]\_2\)\ 
              \(\[Integral]\_0\%1
                  \(\[Integral]\_0\%1
                    \( F[x, y]\ Sin[n\ \[Pi]\ x]\ Sin[m\ \[Pi]\ y]\) 
                    \[DifferentialD]y \[DifferentialD]x\)\)\), "+", 
            \(\((\(-2\)\ m\^4\ \[Pi]\^4 - 2\ m\^2\ n\^2\ \[Pi]\^4)\)\ A\_2\ 
              \[Sigma]\_1\), "+", 
            \(3\/4\ \[Pi]\^4\ 
              \((4\ m\^2\ n\^2\ \((\(-2\) + \[Nu]\^2)\) + 
                  m\^4\ \((\(-1\) - 2\ \[Nu] + \[Nu]\^2)\) + 
                  n\^4\ \((\(-1\) - 2\ \[Nu] + \[Nu]\^2)\))\)\ A\_1\ A\_2\ 
              A\&_\_1\), "+", 
            \(3\/8\ \[Pi]\^4\ 
              \((4\ m\^2\ n\^2\ \((\(-2\) + \[Nu]\^2)\) + 
                  m\^4\ \((\(-1\) - 2\ \[Nu] + \[Nu]\^2)\) + 
                  n\^4\ \((\(-1\) - 2\ \[Nu] + \[Nu]\^2)\))\)\ A\_1\%2\ 
              A\&_\_2\), "+", 
            \(9\/8\ \[Pi]\^4\ 
              \((\(-4\)\ m\^2\ n\^2\ \[Nu] + m\^4\ \((\(-3\) + \[Nu]\^2)\) + 
                  n\^4\ \((\(-3\) + \[Nu]\^2)\))\)\ A\_2\%2\ A\&_\_2\), "-", 
            RowBox[{"I", " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\_2\),
                  SequenceForm[ 
                    Subscript[ D, 1], 
                    Subscript[ A, 2]],
                  Editable->False], ")"}], " ", \(\[Omega]\_\(m, n\)\)}]}], 
          "==", "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["8.3.4 Mixed Approach", "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Alternatively, we can use a mixed approach, a combination of \
discretization and direct approaches, to attack the problem, ",
  StyleBox["EOM1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["BC1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ". We first define"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(scaleRule3 = {
        \((dv : u | v)\) -> \((\[Epsilon]\^2\ dv[#1, #2, #3]\ &)\), 
        w -> \((\[Epsilon]\ w[#1, #2, #3]\ &)\)}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting ",
  StyleBox["scaleRule2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["scaleRule3",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["EOM1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", expanding the result for small \[Epsilon], and discarding terms of order \
higher than ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\^3\)]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(eq83b = 
      \((\(\(EOM1 /. scaleRule2\) /. scaleRule3 // TrigToExp\) // ExpandAll)
          \) /. \[Epsilon]\^\(n_ /; n > 3\) -> 0; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "To write ",
  StyleBox["eq83b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in a more concise way, we introduce the dispaly rule"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(display2[expr_] := 
      expr /. {\(\(Derivative[a_, b_, c_]\)[w_]\)[x, y, t] :> 
            Subscript[w, 
              Sequence\ @@\ 
                Join[Table[x, {a}], Table[y, {b}], Table[t, {c}]]]}\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["Hence,", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((uEOM = Coefficient[eq83b[\([1, 1]\)], \[Epsilon]\^2] == 0)\) // 
      display2\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(u\_\(x, x\) + u\_\(y, y\)\/2 - 1\/2\ \[Nu]\ u\_\(y, y\) + 
        v\_\(x, y\)\/2 + 1\/2\ \[Nu]\ v\_\(x, y\) + w\_x\ w\_\(x, x\) + 
        1\/2\ w\_y\ w\_\(x, y\) + 1\/2\ \[Nu]\ w\_y\ w\_\(x, y\) + 
        1\/2\ w\_x\ w\_\(y, y\) - 1\/2\ \[Nu]\ w\_x\ w\_\(y, y\) == 0\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((vEOM = Coefficient[eq83b[\([2, 1]\)], \[Epsilon]\^2] == 0)\) // 
      display2\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(u\_\(x, y\)\/2 + 1\/2\ \[Nu]\ u\_\(x, y\) + v\_\(x, x\)\/2 - 
        1\/2\ \[Nu]\ v\_\(x, x\) + v\_\(y, y\) + 1\/2\ w\_y\ w\_\(x, x\) - 
        1\/2\ \[Nu]\ w\_y\ w\_\(x, x\) + 1\/2\ w\_x\ w\_\(x, y\) + 
        1\/2\ \[Nu]\ w\_x\ w\_\(x, y\) + w\_y\ w\_\(y, y\) == 0\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((wEOM = 
          \(eq83b[\([3]\)] // Thread[\(\[Epsilon]\^\(-2\)\) #, Equal]\ &\) // 
            ExpandAll)\) // display2\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(-\(1\/24\)\)\ E\^\(\(-I\)\ T\_0\ \[CapitalOmega]\)\ \[Epsilon]\ 
          F[x, y] - 
        1\/24\ E\^\(I\ T\_0\ \[CapitalOmega]\)\ \[Epsilon]\ F[x, y] + 
        w\_\(t, t\)\/12 - \[Epsilon]\ u\_x\ w\_\(x, x\) - 
        \[Epsilon]\ \[Nu]\ v\_y\ w\_\(x, x\) - 
        1\/2\ \[Epsilon]\ w\_x\%2\ w\_\(x, x\) - 
        1\/2\ \[Epsilon]\ \[Nu]\ w\_y\%2\ w\_\(x, x\) - 
        \[Epsilon]\ u\_y\ w\_\(x, y\) + 
        \[Epsilon]\ \[Nu]\ u\_y\ w\_\(x, y\) - 
        \[Epsilon]\ v\_x\ w\_\(x, y\) + 
        \[Epsilon]\ \[Nu]\ v\_x\ w\_\(x, y\) - 
        \[Epsilon]\ w\_x\ w\_y\ w\_\(x, y\) + 
        \[Epsilon]\ \[Nu]\ w\_x\ w\_y\ w\_\(x, y\) - 
        \[Epsilon]\ \[Nu]\ u\_x\ w\_\(y, y\) - 
        \[Epsilon]\ v\_y\ w\_\(y, y\) - 
        1\/2\ \[Epsilon]\ \[Nu]\ w\_x\%2\ w\_\(y, y\) - 
        1\/2\ \[Epsilon]\ w\_y\%2\ w\_\(y, y\) + 1\/12\ w\_\(x, x, x, x\) + 
        1\/6\ w\_\(x, x, y, y\) + 
        1\/3\ \[Epsilon]\ \[Sigma]\_1\ w\_\(x, x, y, y\) + 
        1\/12\ w\_\(y, y, y, y\) + 
        1\/3\ \[Epsilon]\ \[Sigma]\_1\ w\_\(y, y, y, y\) == 0\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Linear vibrations are governed by ",
  StyleBox["wEOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " with ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon] = 0\)]],
  "; that is,"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((order1Eqw = wEOM /. \[Epsilon] -> 0)\) // display2\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(w\_\(t, t\)\/12 + 1\/12\ w\_\(x, x, x, x\) + 1\/6\ w\_\(x, x, y, y\) + 
        1\/12\ w\_\(y, y, y, y\) == 0\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "subject to the boundary conditions ",
  StyleBox["BC1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ". It follows from ",
  StyleBox["order1Eqw",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["BC1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " that the linear modeshapes and associated frequencies are given by"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\[Phi]\_\(m, n\)[x, y] /. modeshapes\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(Sin[m\ \[Pi]\ x]\ Sin[n\ \[Pi]\ y]\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(\[Omega]\_\(m, n\) /. frequencies\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\((m\^2 + n\^2)\)\ \[Pi]\^2\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To investigate the case of one-to-one internal resonance between the ",
  StyleBox["mn",
    FontSlant->"Italic"],
  "th and ",
  StyleBox["nm",
    FontSlant->"Italic"],
  "th modes (",
  Cell[BoxData[
      \(TraditionalForm\`m \[NotEqual] n\)]],
  "), we let"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(w1sol = {
        w -> Function[{x, y, t}, 
            \[Eta][t]\ Sin[m\ \[Pi]\ x]\ Sin[n\ \[Pi]\ y] + 
              \[Zeta][t]\ Sin[n\ \[Pi]\ x]\ Sin[m\ \[Pi]\ y]]}; \)\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting the ",
  StyleBox["w1sol",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["uEOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["vEOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order2Equ = 
      \((uEOM[\([1]\)] /. w -> \((0\ &)\))\) == 
        \((\((uEOM[\([1]\)] /. w -> \((0\ &)\))\) - uEOM[\([1]\)] /. w1sol // 
            TrigReduce)\); \norder2Equ // display2\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(u\_\(x, x\) + u\_\(y, y\)\/2 - 1\/2\ \[Nu]\ u\_\(y, y\) + 
        v\_\(x, y\)\/2 + 1\/2\ \[Nu]\ v\_\(x, y\) == 
      1\/16\ \((
          4\ n\^3\ \[Pi]\^3\ Sin[2\ n\ \[Pi]\ x]\ \[Zeta][t]\^2 - 
            4\ m\^2\ n\ \[Pi]\^3\ \[Nu]\ Sin[2\ n\ \[Pi]\ x]\ \[Zeta][t]\^2 - 
            2\ m\^2\ n\ \[Pi]\^3\ Sin[2\ n\ \[Pi]\ x - 2\ m\ \[Pi]\ y]\ 
              \[Zeta][t]\^2 - 
            2\ n\^3\ \[Pi]\^3\ Sin[2\ n\ \[Pi]\ x - 2\ m\ \[Pi]\ y]\ 
              \[Zeta][t]\^2 - 
            2\ m\^2\ n\ \[Pi]\^3\ Sin[2\ n\ \[Pi]\ x + 2\ m\ \[Pi]\ y]\ 
              \[Zeta][t]\^2 - 
            2\ n\^3\ \[Pi]\^3\ Sin[2\ n\ \[Pi]\ x + 2\ m\ \[Pi]\ y]\ 
              \[Zeta][t]\^2 + 
            m\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^2\ n\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\ n\^2\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            n\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^2\ n\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\ n\^2\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            n\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            3\ m\^2\ n\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            3\ m\ n\^2\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            n\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^2\ n\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\ n\^2\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            n\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            3\ m\^2\ n\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            3\ m\ n\^2\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            n\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^2\ n\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\ n\^2\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            n\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^2\ n\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\ n\^2\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            n\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^2\ n\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\ n\^2\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            n\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            3\ m\^2\ n\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            3\ m\ n\^2\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            n\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^2\ n\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\ n\^2\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            n\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^2\ n\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\ n\^2\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            n\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^2\ n\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\ n\^2\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            n\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^2\ n\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\ n\^2\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            n\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^2\ n\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\ n\^2\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            n\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            3\ m\^2\ n\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            3\ m\ n\^2\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            n\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^2\ n\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\ n\^2\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            n\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            4\ m\^3\ \[Pi]\^3\ Sin[2\ m\ \[Pi]\ x]\ \[Eta][t]\^2 - 
            4\ m\ n\^2\ \[Pi]\^3\ \[Nu]\ Sin[2\ m\ \[Pi]\ x]\ \[Eta][t]\^2 - 
            2\ m\^3\ \[Pi]\^3\ Sin[2\ m\ \[Pi]\ x - 2\ n\ \[Pi]\ y]\ 
              \[Eta][t]\^2 - 
            2\ m\ n\^2\ \[Pi]\^3\ Sin[2\ m\ \[Pi]\ x - 2\ n\ \[Pi]\ y]\ 
              \[Eta][t]\^2 - 
            2\ m\^3\ \[Pi]\^3\ Sin[2\ m\ \[Pi]\ x + 2\ n\ \[Pi]\ y]\ 
              \[Eta][t]\^2 - 
            2\ m\ n\^2\ \[Pi]\^3\ Sin[2\ m\ \[Pi]\ x + 2\ n\ \[Pi]\ y]\ 
              \[Eta][t]\^2)\)\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(order2Eqv = 
      \((vEOM[\([1]\)] /. w -> \((0\ &)\))\) == 
        \((\((vEOM[\([1]\)] /. w -> \((0\ &)\))\) - vEOM[\([1]\)] /. w1sol // 
            TrigReduce)\); \norder2Eqv // display2\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(u\_\(x, y\)\/2 + 1\/2\ \[Nu]\ u\_\(x, y\) + v\_\(x, x\)\/2 - 
        1\/2\ \[Nu]\ v\_\(x, x\) + v\_\(y, y\) == 
      1\/16\ \((
          4\ m\^3\ \[Pi]\^3\ Sin[2\ m\ \[Pi]\ y]\ \[Zeta][t]\^2 - 
            4\ m\ n\^2\ \[Pi]\^3\ \[Nu]\ Sin[2\ m\ \[Pi]\ y]\ \[Zeta][t]\^2 + 
            2\ m\^3\ \[Pi]\^3\ Sin[2\ n\ \[Pi]\ x - 2\ m\ \[Pi]\ y]\ 
              \[Zeta][t]\^2 + 
            2\ m\ n\^2\ \[Pi]\^3\ Sin[2\ n\ \[Pi]\ x - 2\ m\ \[Pi]\ y]\ 
              \[Zeta][t]\^2 - 
            2\ m\^3\ \[Pi]\^3\ Sin[2\ n\ \[Pi]\ x + 2\ m\ \[Pi]\ y]\ 
              \[Zeta][t]\^2 - 
            2\ m\ n\^2\ \[Pi]\^3\ Sin[2\ n\ \[Pi]\ x + 2\ m\ \[Pi]\ y]\ 
              \[Zeta][t]\^2 - 
            m\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^2\ n\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\ n\^2\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            n\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^2\ n\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\ n\^2\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            n\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            3\ m\^2\ n\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            3\ m\ n\^2\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            n\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^2\ n\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\ n\^2\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            n\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            3\ m\^2\ n\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            3\ m\ n\^2\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            n\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^2\ n\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\ n\^2\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            n\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^2\ n\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\ n\^2\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            n\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^2\ n\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\ n\^2\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            n\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            3\ m\^2\ n\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            3\ m\ n\^2\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            n\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^2\ n\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\ n\^2\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            n\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^2\ n\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\ n\^2\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            n\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^2\ n\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\ n\^2\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            n\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^2\ n\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\ n\^2\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            n\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^2\ n\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\ n\^2\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            n\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            3\ m\^2\ n\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            3\ m\ n\^2\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            n\^3\ \[Pi]\^3\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            m\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\^2\ n\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] - 
            m\ n\^2\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            n\^3\ \[Pi]\^3\ \[Nu]\ 
              Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ 
              \[Zeta][t]\ \[Eta][t] + 
            4\ n\^3\ \[Pi]\^3\ Sin[2\ n\ \[Pi]\ y]\ \[Eta][t]\^2 - 
            4\ m\^2\ n\ \[Pi]\^3\ \[Nu]\ Sin[2\ n\ \[Pi]\ y]\ \[Eta][t]\^2 + 
            2\ m\^2\ n\ \[Pi]\^3\ Sin[2\ m\ \[Pi]\ x - 2\ n\ \[Pi]\ y]\ 
              \[Eta][t]\^2 + 
            2\ n\^3\ \[Pi]\^3\ Sin[2\ m\ \[Pi]\ x - 2\ n\ \[Pi]\ y]\ 
              \[Eta][t]\^2 - 
            2\ m\^2\ n\ \[Pi]\^3\ Sin[2\ m\ \[Pi]\ x + 2\ n\ \[Pi]\ y]\ 
              \[Eta][t]\^2 - 
            2\ n\^3\ \[Pi]\^3\ Sin[2\ m\ \[Pi]\ x + 2\ n\ \[Pi]\ y]\ 
              \[Eta][t]\^2)\)\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To obtain the particular solutions for ",
  Cell[BoxData[
      \(TraditionalForm\`u\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`v\)]],
  ", we use the method of undetermined coefficients. To accomplish this, we \
first look for all possible terms that appear on the right-hand sides of ",
  StyleBox["order2Equ",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["order2Eqv",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " as follows:"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(possibleTerms = 
      \(\(Cases[{order2Equ[\([2]\)], order2Eqv[\([2]\)]}, #, \[Infinity]]\ &
              \)\ /@\ {_\ Sin[a_]\ h1_[t]\^2 -> 
                Sin[a]\ h1[t]\^2, _\ Sin[a_]\ h1_[t]\ h2_[t] -> 
                Sin[a]\ h1[t]\ h2[t]} // Flatten\) // Union\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({Sin[2\ n\ \[Pi]\ x]\ \[Zeta][t]\^2, 
      Sin[2\ m\ \[Pi]\ y]\ \[Zeta][t]\^2, 
      Sin[2\ n\ \[Pi]\ x - 2\ m\ \[Pi]\ y]\ \[Zeta][t]\^2, 
      Sin[2\ n\ \[Pi]\ x + 2\ m\ \[Pi]\ y]\ \[Zeta][t]\^2, 
      Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ \[Zeta][t]\ 
        \[Eta][t], 
      Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y - n\ \[Pi]\ y]\ \[Zeta][t]\ 
        \[Eta][t], 
      Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ \[Zeta][t]\ 
        \[Eta][t], 
      Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y - n\ \[Pi]\ y]\ \[Zeta][t]\ 
        \[Eta][t], 
      Sin[m\ \[Pi]\ x - n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ \[Zeta][t]\ 
        \[Eta][t], 
      Sin[m\ \[Pi]\ x + n\ \[Pi]\ x - m\ \[Pi]\ y + n\ \[Pi]\ y]\ \[Zeta][t]\ 
        \[Eta][t], 
      Sin[m\ \[Pi]\ x - n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ \[Zeta][t]\ 
        \[Eta][t], 
      Sin[m\ \[Pi]\ x + n\ \[Pi]\ x + m\ \[Pi]\ y + n\ \[Pi]\ y]\ \[Zeta][t]\ 
        \[Eta][t], Sin[2\ m\ \[Pi]\ x]\ \[Eta][t]\^2, 
      Sin[2\ n\ \[Pi]\ y]\ \[Eta][t]\^2, 
      Sin[2\ m\ \[Pi]\ x - 2\ n\ \[Pi]\ y]\ \[Eta][t]\^2, 
      Sin[2\ m\ \[Pi]\ x + 2\ n\ \[Pi]\ y]\ \[Eta][t]\^2}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "which satisfy ",
  StyleBox["BC1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ". We then define the following symbol lists to represent the undetermined \
coefficients."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(usymbolList = Table[c\_i, {i, Length[possibleTerms]}]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({c\_1, c\_2, c\_3, c\_4, c\_5, c\_6, c\_7, c\_8, c\_9, c\_10, c\_11, 
      c\_12, c\_13, c\_14, c\_15, c\_16}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(vsymbolList = usymbolList /. c -> d\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({d\_1, d\_2, d\_3, d\_4, d\_5, d\_6, d\_7, d\_8, d\_9, d\_10, d\_11, 
      d\_12, d\_13, d\_14, d\_15, d\_16}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["Then, we express the solution as", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(usol = 
      u -> Function[{x, y, t}, usymbolList . possibleTerms // Evaluate]; \n
    vsol = v -> Function[{x, y, t}, vsymbolList . possibleTerms // Evaluate]; 
    \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting ",
  StyleBox["usol",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["vsol",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["order2Equ",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["order2Eqv",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", collecting coefficients of ",
  StyleBox["possibleTerms",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", and solving for ",
  StyleBox["usymbolList",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["vsymbolList",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(algEq1 = 
      Coefficient[\(Subtract\ @@\ order2Equ /. usol\) /. vsol, 
            possibleTerms] == 0 // Thread; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(algEq2 = 
      Coefficient[\(Subtract\ @@\ order2Eqv /. usol\) /. vsol, 
            possibleTerms] == 0 // Thread; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((symbolRule2 = 
          \(Solve[Join[algEq1, algEq2], Join[usymbolList, vsymbolList]]\)[
              \([1]\)] // Simplify)\) // Timing\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({2.48000000000001818`\ Second, {
        c\_1 \[Rule] \(-\(\(\[Pi]\ \((n\^2 - m\^2\ \[Nu])\)\)\/\(16\ n\)\)\), 
        c\_13 \[Rule] 
          \(-\(\(\[Pi]\ \((m\^2 - n\^2\ \[Nu])\)\)\/\(16\ m\)\)\), 
        d\_2 \[Rule] \(-\(\(\[Pi]\ \((m\^2 - n\^2\ \[Nu])\)\)\/\(16\ m\)\)\), 
        d\_14 \[Rule] 
          \(-\(\(\[Pi]\ \((n\^2 - m\^2\ \[Nu])\)\)\/\(16\ n\)\)\), 
        c\_2 \[Rule] 0, c\_14 \[Rule] 0, d\_1 \[Rule] 0, d\_13 \[Rule] 0, 
        c\_3 \[Rule] \(n\ \[Pi]\)\/32, c\_4 \[Rule] \(n\ \[Pi]\)\/32, 
        c\_15 \[Rule] \(m\ \[Pi]\)\/32, c\_16 \[Rule] \(m\ \[Pi]\)\/32, 
        d\_3 \[Rule] \(-\(\(m\ \[Pi]\)\/32\)\), 
        d\_4 \[Rule] \(m\ \[Pi]\)\/32, 
        d\_15 \[Rule] \(-\(\(n\ \[Pi]\)\/32\)\), 
        d\_16 \[Rule] \(n\ \[Pi]\)\/32, 
        c\_7 \[Rule] 
          \(-\(\(\[Pi]\ 
                  \((m\^2\ \((\(-1\) + \[Nu])\) + 
                      n\^2\ \((\(-1\) + \[Nu])\) + 2\ m\ n\ \((1 + \[Nu])\))
                    \)\)\/\(32\ \((m - n)\)\)\)\), 
        c\_12 \[Rule] 
          \(-\(\(\[Pi]\ 
                  \((m\^2\ \((\(-1\) + \[Nu])\) + 
                      n\^2\ \((\(-1\) + \[Nu])\) - 2\ m\ n\ \((1 + \[Nu])\))
                    \)\)\/\(32\ \((m + n)\)\)\)\), 
        d\_7 \[Rule] 
          \(-\(\(\[Pi]\ 
                  \((m\^2\ \((\(-1\) + \[Nu])\) + 
                      n\^2\ \((\(-1\) + \[Nu])\) + 2\ m\ n\ \((1 + \[Nu])\))
                    \)\)\/\(32\ \((m - n)\)\)\)\), 
        d\_12 \[Rule] 
          \(-\(\(\[Pi]\ 
                  \((m\^2\ \((\(-1\) + \[Nu])\) + 
                      n\^2\ \((\(-1\) + \[Nu])\) - 2\ m\ n\ \((1 + \[Nu])\))
                    \)\)\/\(32\ \((m + n)\)\)\)\), 
        c\_5 \[Rule] 1\/32\ \((m - n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\), 
        c\_6 \[Rule] 
          \(-\(\(\[Pi]\ 
                  \((m\^2\ \((\(-1\) + \[Nu])\) + 
                      n\^2\ \((\(-1\) + \[Nu])\) - 2\ m\ n\ \((1 + \[Nu])\))
                    \)\)\/\(32\ \((m + n)\)\)\)\), 
        c\_8 \[Rule] 1\/32\ \((m + n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\), 
        c\_9 \[Rule] 
          \(-\(\(\[Pi]\ 
                  \((m\^2\ \((\(-1\) + \[Nu])\) + 
                      n\^2\ \((\(-1\) + \[Nu])\) + 2\ m\ n\ \((1 + \[Nu])\))
                    \)\)\/\(32\ \((m - n)\)\)\)\), 
        c\_10 \[Rule] 1\/32\ \((m + n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\), 
        c\_11 \[Rule] 1\/32\ \((m - n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\), 
        d\_5 \[Rule] 
          \(-\(1\/32\)\)\ \((m + n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\), 
        d\_6 \[Rule] 
          \(\[Pi]\ \((
                m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) - 
                  2\ m\ n\ \((1 + \[Nu])\))\)\)\/\(32\ \((m + n)\)\), 
        d\_8 \[Rule] 1\/32\ \((m - n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\), 
        d\_9 \[Rule] 
          \(\[Pi]\ \((
                m\^2\ \((\(-1\) + \[Nu])\) + n\^2\ \((\(-1\) + \[Nu])\) + 
                  2\ m\ n\ \((1 + \[Nu])\))\)\)\/\(32\ \((m - n)\)\), 
        d\_10 \[Rule] 
          \(-\(1\/32\)\)\ \((m - n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\), 
        d\_11 \[Rule] 1\/32\ \((m + n)\)\ \[Pi]\ \((\(-1\) + \[Nu])\)}}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["Solvability Conditions", "Subsubsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "We introduce two time scales ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_1\)]],
  " and seek a first-order uniform asymptotic expansion of the solution of ",
  StyleBox["wEOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in the form"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(solRule = {
        w -> Function[{x, y, T\_0, T\_1}, 
            Sum[\[Epsilon]\^\(j - 1\)\ w\_j[x, y, T\_0, T\_1], {j, 2}]]}; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Transforming the time derivatives in ",
  StyleBox["wEOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in terms of the scales ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_1\)]],
  ", substituting ",
  StyleBox["solRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["wEOM",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", expanding the result for small \[Epsilon], and discarding terms of order \
higher than ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\)]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(eq83c = 
      \((\(wEOM /. {\((s : u | v | w)\)[x, y, t] -> s[x, y, T\_0, T\_1], 
                  \(\(Derivative[a_, b_, c_]\)[h_]\)[x, y, t] -> 
                    \(dt[c]\)[D[h[x, y, T\_0, T\_1], {x, a}, {y, b}]]}\) /. 
              solRule // ExpandAll)\) /. \[Epsilon]\^\(n_ /; n > 1\) -> 0; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[
"Equating coefficients of like powers of \[Epsilon], we have"], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(eqEps = CoefficientList[eq83c[\([1]\)], \[Epsilon]] == 0 // Thread; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Accounting for the two time scales ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_1\)]],
  ", we create a rule for ",
  Cell[BoxData[
      \(TraditionalForm\`\[Eta][t]\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Zeta][t]\)]]
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ampRule = {
        \[Eta][t] -> 
          A\_1[T\_1]\ E\^\(I\ \[Omega]\_\(m, n\)\ T\_0\) + 
            A\&_\_1[T\_1]\ E\^\(\(-I\)\ \[Omega]\_\(m, n\)\ T\_0\), 
        \[Zeta][t] -> 
          A\_2[T\_1]\ E\^\(I\ \[Omega]\_\(m, n\)\ T\_0\) + 
            A\&_\_2[T\_1]\ E\^\(\(-I\)\ \[Omega]\_\(m, n\)\ T\_0\)}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "and rewrite ",
  StyleBox["w1sol",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", ",
  StyleBox["usol",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", and ",
  StyleBox["vsol",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(sol1 = {
        w\_1 -> Function[{x, y, T\_0, T\_1}, 
            \(w[x, y, t] /. w1sol\) /. ampRule // Evaluate], 
        u -> Function[{x, y, T\_0, T\_1}, 
            \(u[x, y, t] /. usol\) /. ampRule // Evaluate], 
        v -> Function[{x, y, T\_0, T\_1}, 
            \(v[x, y, t] /. vsol\) /. ampRule // Evaluate]}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting the ",
  StyleBox["sol1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  StyleBox[" ",
    FontColor->RGBColor[1, 0, 1]],
  "into the second-order problem, ",
  StyleBox["eqEps[[2]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(order2Eqw = 
        \((eqEps[\([1, 1]\)] /. w\_1 -> w\_2)\) == 
              \((eqEps[\([1, 1]\)] /. w\_1 -> w\_2)\) - eqEps[\([2, 1]\)] /. 
            sol1 // ExpandAll; \) // Timing\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({11.0900000000000331`\ Second, Null}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Collecting the terms that may lead to secular terms, the terms \
proportional to ",
  Cell[BoxData[
      \(E\^\(I\ \[Omega]\_\(m, n\)\ T\_0\)\)]],
  ", we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(ST = 
        Coefficient[order2Eqw[\([2]\)] /. expRule, 
            E\^\(I\ \[Omega]\_\(m, n\)\ T\_0\)] // TrigReduce; \) // Timing
      \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({156.76000000000001`\ Second, Null}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Among all the spatial-dependent non-forcing terms in ",
  StyleBox["ST",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", only the following terms are not orthogonal to the ",
  StyleBox["modeshapes",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ":"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(terms1 = 
      Cases[ST, Cos[m\ \[Pi]\ x + a_. \ n\ \[Pi]\ y /; Abs[a] == 1], 
          \[Infinity]] // Union\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({Cos[m\ \[Pi]\ x - n\ \[Pi]\ y], Cos[m\ \[Pi]\ x + n\ \[Pi]\ y]}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(terms2 = 
      Cases[ST, Cos[n\ \[Pi]\ x + a_. \ m\ \[Pi]\ y /; Abs[a] == 1], 
          \[Infinity]] // Union\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({Cos[n\ \[Pi]\ x - m\ \[Pi]\ y], Cos[n\ \[Pi]\ x + m\ \[Pi]\ y]}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The coefficients of these terms in ",
  StyleBox["ST",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " are"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(terms1Coef = Coefficient[ST, terms1]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(terms2Coef = Coefficient[ST, terms2]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "The projection of ",
  StyleBox["terms1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " onto ",
  Cell[BoxData[
      \(TraditionalForm\`Sin[m\ \[Pi]\ x]\ Sin[n\ \[Pi]\ y]\)]],
  " can be obtained as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(p1Coef = 
      \(\[Integral]\_0\%1
                \(\[Integral]\_0\%1 Sin[m\ \[Pi]\ x]\ Sin[n\ \[Pi]\ y]\ #\ 
                  \[DifferentialD]x \[DifferentialD]y\)\ &\)\ /@\ terms1 /. 
        trigRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({1\/4, \(-\(1\/4\)\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "and the projection of ",
  StyleBox["terms2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " onto ",
  Cell[BoxData[
      \(TraditionalForm\`Sin[n\ \[Pi]\ x]\ Sin[m\ \[Pi]\ y]\)]],
  " can be obtained as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(p2Coef = 
      \(\[Integral]\_0\%1
                \(\[Integral]\_0\%1 Sin[n\ \[Pi]\ x]\ Sin[m\ \[Pi]\ y]\ #\ 
                  \[DifferentialD]x \[DifferentialD]y\)\ &\)\ /@\ terms2 /. 
        trigRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({1\/4, \(-\(1\/4\)\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The forcing term in ",
  StyleBox["ST",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " is"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(f0 = ST /. Cos[_] -> 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(1\/24\ E\^\(I\ T\_1\ \[Sigma]\_2\)\ F[x, y]\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Then, the solvability conditions demand that ",
  StyleBox["ST",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " be orthogonal to every linear eigenfunction, ",
  StyleBox["modeshapes",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  "; that is,"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(SC1 = 
      Expand[24 
            \((\(\(int[Sin[m\ \[Pi]\ x]\ Sin[n\ \[Pi]\ y]\ f0, {x, 0, 1}, {y, 
                          0, 1}] + p1Coef . terms1Coef /. symbolRule2\) //. 
                  intRule3\) /. int -> Integrate)\)] == 0; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(SC2 = 
      Expand[24 
            \((\(\(int[Sin[n\ \[Pi]\ x]\ Sin[m\ \[Pi]\ y]\ f0, {x, 0, 1}, {y, 
                          0, 1}] + p2Coef . terms2Coef /. symbolRule2\) //. 
                  intRule3\) /. int -> Integrate)\)] == 0; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
They agree with the results obtained in the previous section:\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \({SC1, SC2} === {SCond1, SCond2}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(True\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"Microsoft Windows 3.0",
ScreenRectangle->{{0, 1024}, {0, 712}},
WindowToolbars->{"RulerBar", "EditBar"},
WindowSize->{705, 565},
WindowMargins->{{2, Automatic}, {Automatic, 5}},
PrintingCopies->1,
PrintingStartingPageNumber->247,
PrintingPageRange->{Automatic, Automatic},
PageHeaders->{{Cell[ 
        TextData[ {
          CounterBox[ "Page"]}], "PageNumber"], Inherited, Cell[ 
        TextData[ {
          ValueBox[ "FileName"]}], "Header"]}, {Cell[ 
        TextData[ {
          ValueBox[ "FileName"]}], "Header"], Inherited, Cell[ 
        TextData[ {
          CounterBox[ "Page"]}], "PageNumber"]}},
PrintingOptions->{"FirstPageHeader"->False,
"FacingPages"->True},
Magnification->1
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1731, 51, 140, 5, 195, "Title"],
Cell[1874, 58, 937, 21, 109, "Text"],

Cell[CellGroupData[{
Cell[2836, 83, 79, 2, 53, "Section"],
Cell[2918, 87, 118, 3, 30, "Input"],
Cell[3039, 92, 112, 3, 30, "Input"],
Cell[3154, 97, 678, 17, 71, "Text"],
Cell[3835, 116, 456, 15, 30, "Input"],
Cell[4294, 133, 117, 3, 30, "Input"],
Cell[4414, 138, 328, 13, 26, "Text"],
Cell[4745, 153, 324, 8, 34, "Input"],
Cell[5072, 163, 321, 8, 40, "Text"],
Cell[5396, 173, 172, 4, 22, "Input"],
Cell[5571, 179, 247, 8, 26, "Text"],
Cell[5821, 189, 499, 10, 72, "Input"],
Cell[6323, 201, 144, 4, 26, "Text"],
Cell[6470, 207, 900, 18, 107, "Input"]
}, Open  ]],

Cell[CellGroupData[{
Cell[7407, 230, 83, 2, 46, "Section"],
Cell[7493, 234, 334, 7, 54, "Text"],
Cell[7830, 243, 199, 5, 40, "Text"],
Cell[8032, 250, 443, 9, 50, "Input"],
Cell[8478, 261, 1700, 60, 83, "Text"],
Cell[10181, 323, 283, 6, 22, "Input"],
Cell[10467, 331, 179, 7, 26, "Text"],
Cell[10649, 340, 233, 6, 40, "Text"],
Cell[10885, 348, 134, 3, 38, "Input"],
Cell[11022, 353, 445, 9, 40, "Input"],
Cell[11470, 364, 361, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[11856, 380, 407, 8, 75, "Input"],
Cell[12266, 390, 1742, 49, 61, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[14045, 444, 98, 2, 43, "Subsection"],
Cell[14146, 448, 448, 14, 40, "Text"],
Cell[14597, 464, 208, 7, 22, "Input"],
Cell[14808, 473, 287, 9, 40, "Text"],

Cell[CellGroupData[{
Cell[15120, 486, 189, 5, 22, "Input"],
Cell[15312, 493, 467, 14, 22, "Output"]
}, Open  ]],
Cell[15794, 510, 413, 16, 26, "Text"],

Cell[CellGroupData[{
Cell[16232, 530, 412, 12, 22, "Input"],
Cell[16647, 544, 472, 14, 22, "Output"]
}, Open  ]],
Cell[17134, 561, 838, 29, 55, "Text"],

Cell[CellGroupData[{
Cell[17997, 594, 215, 5, 22, "Input"],
Cell[18215, 601, 713, 18, 22, "Output"]
}, Open  ]],
Cell[18943, 622, 336, 14, 26, "Text"],

Cell[CellGroupData[{
Cell[19304, 640, 169, 4, 22, "Input"],
Cell[19476, 646, 469, 12, 35, "Output"]
}, Open  ]],
Cell[19960, 661, 240, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[20225, 676, 142, 4, 22, "Input"],
Cell[20370, 682, 522, 13, 34, "Output"]
}, Open  ]],
Cell[20907, 698, 396, 17, 26, "Text"],
Cell[21306, 717, 469, 14, 40, "Text"],
Cell[21778, 733, 138, 3, 22, "Input"],
Cell[21919, 738, 748, 29, 40, "Text"],

Cell[CellGroupData[{
Cell[22692, 771, 218, 5, 34, "Input"],
Cell[22913, 778, 4113, 103, 87, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[27063, 886, 119, 3, 22, "Input"],
Cell[27185, 891, 239, 5, 22, "Output"]
}, Open  ]],
Cell[27439, 899, 411, 13, 40, "Text"],
Cell[27853, 914, 931, 26, 22, "Input"],
Cell[28787, 942, 212, 7, 26, "Text"],

Cell[CellGroupData[{
Cell[29024, 953, 215, 5, 22, "Input"],
Cell[29242, 960, 593, 18, 22, "Output"]
}, Open  ]],
Cell[29850, 981, 422, 14, 40, "Text"],
Cell[30275, 997, 171, 4, 34, "Input"],
Cell[30449, 1003, 203, 7, 26, "Text"],

Cell[CellGroupData[{
Cell[30677, 1014, 89, 2, 34, "Subsubsection"],
Cell[30769, 1018, 469, 13, 54, "Text"],

Cell[CellGroupData[{
Cell[31263, 1035, 306, 7, 34, "Input"],
Cell[31572, 1044, 1434, 38, 34, "Output"]
}, Open  ]],
Cell[33021, 1085, 226, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[33272, 1097, 94, 3, 22, "Input"],
Cell[33369, 1102, 593, 18, 22, "Output"]
}, Open  ]],
Cell[33977, 1123, 333, 12, 26, "Text"],
Cell[34313, 1137, 158, 4, 22, "Input"],
Cell[34474, 1143, 897, 34, 40, "Text"],

Cell[CellGroupData[{
Cell[35396, 1181, 260, 6, 34, "Input"],
Cell[35659, 1189, 1210, 29, 34, "Output"]
}, Open  ]],
Cell[36884, 1221, 93, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[37002, 1227, 144, 4, 22, "Input"],
Cell[37149, 1233, 535, 14, 22, "Output"]
}, Open  ]],
Cell[37699, 1250, 1193, 36, 69, "Text"],
Cell[38895, 1288, 113, 3, 22, "Input"],
Cell[39011, 1293, 240, 5, 22, "Input"],

Cell[CellGroupData[{
Cell[39276, 1302, 187, 5, 22, "Input"],
Cell[39466, 1309, 301, 7, 22, "Output"]
}, Open  ]],
Cell[39782, 1319, 190, 7, 26, "Text"],

Cell[CellGroupData[{
Cell[39997, 1330, 288, 7, 34, "Input"],
Cell[40288, 1339, 137, 3, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[40462, 1347, 224, 5, 22, "Input"],
Cell[40689, 1354, 284, 7, 28, "Output"]
}, Open  ]],
Cell[40988, 1364, 804, 32, 40, "Text"],

Cell[CellGroupData[{
Cell[41817, 1400, 340, 7, 34, "Input"],
Cell[42160, 1409, 2427, 44, 117, "Output"]
}, Open  ]],
Cell[44602, 1456, 298, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[44925, 1472, 122, 3, 22, "Input"],
Cell[45050, 1477, 2292, 40, 146, "Output"]
}, Open  ]],
Cell[47357, 1520, 386, 15, 26, "Text"],

Cell[CellGroupData[{
Cell[47768, 1539, 146, 4, 22, "Input"],
Cell[47917, 1545, 1122, 21, 54, "Output"]
}, Open  ]],
Cell[49054, 1569, 283, 9, 26, "Text"],

Cell[CellGroupData[{
Cell[49362, 1582, 105, 3, 22, "Input"],
Cell[49470, 1587, 241, 5, 23, "Output"]
}, Open  ]],
Cell[49726, 1595, 430, 15, 26, "Text"],

Cell[CellGroupData[{
Cell[50181, 1614, 184, 5, 22, "Input"],
Cell[50368, 1621, 123, 3, 24, "Output"]
}, Open  ]],
Cell[50506, 1627, 65, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[50596, 1633, 205, 5, 22, "Input"],
Cell[50804, 1640, 132, 3, 24, "Output"]
}, Open  ]],
Cell[50951, 1646, 182, 7, 26, "Text"],

Cell[CellGroupData[{
Cell[51158, 1657, 424, 8, 34, "Input"],
Cell[51585, 1667, 308, 7, 24, "Output"]
}, Open  ]],
Cell[51908, 1677, 143, 4, 26, "Text"],

Cell[CellGroupData[{
Cell[52076, 1685, 638, 13, 74, "Input"],
Cell[52717, 1700, 32035, 1115, 148, 16467, 917, "GraphicsData", 
"PostScript", "Graphics"]
}, Open  ]],
Cell[84767, 2818, 1796, 47, 126, "Text"],
Cell[86566, 2867, 251, 6, 40, "Text"],
Cell[86820, 2875, 286, 7, 34, "Input"],
Cell[87109, 2884, 402, 12, 40, "Text"],

Cell[CellGroupData[{
Cell[87536, 2900, 164, 5, 22, "Input"],
Cell[87703, 2907, 115, 3, 35, "Output"]
}, Open  ]],
Cell[87833, 2913, 166, 7, 26, "Text"],

Cell[CellGroupData[{
Cell[88024, 2924, 279, 6, 22, "Input"],
Cell[88306, 2932, 148, 4, 24, "Output"]
}, Open  ]],
Cell[88469, 2939, 212, 5, 40, "Text"],
Cell[88684, 2946, 156, 4, 22, "Input"],
Cell[88843, 2952, 311, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[89179, 2967, 193, 4, 22, "Input"],
Cell[89375, 2973, 109, 3, 22, "Output"]
}, Open  ]],
Cell[89499, 2979, 84, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[89608, 2985, 234, 6, 34, "Input"],
Cell[89845, 2993, 99, 3, 22, "Output"]
}, Open  ]],
Cell[89959, 2999, 280, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[90264, 3014, 240, 6, 22, "Input"],
Cell[90507, 3022, 97, 3, 22, "Output"]
}, Open  ]],
Cell[90619, 3028, 89, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[90733, 3034, 240, 6, 34, "Input"],
Cell[90976, 3042, 102, 3, 22, "Output"]
}, Open  ]],
Cell[91093, 3048, 146, 4, 26, "Text"],
Cell[91242, 3054, 116, 3, 24, "Input"],
Cell[91361, 3059, 116, 3, 24, "Input"],
Cell[91480, 3064, 116, 3, 24, "Input"],
Cell[91599, 3069, 282, 8, 40, "Text"],
Cell[91884, 3079, 243, 6, 34, "Input"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[92176, 3091, 96, 2, 43, "Subsection"],
Cell[92275, 3095, 639, 25, 40, "Text"],
Cell[92917, 3122, 257, 6, 34, "Input"],
Cell[93177, 3130, 132, 4, 26, "Text"],

Cell[CellGroupData[{
Cell[93334, 3138, 180, 5, 22, "Input"],
Cell[93517, 3145, 239, 6, 22, "Output"]
}, Open  ]],
Cell[93771, 3154, 536, 25, 26, "Text"],
Cell[94310, 3181, 708, 28, 40, "Text"],
Cell[95021, 3211, 299, 6, 47, "Input"],
Cell[95323, 3219, 135, 3, 26, "Text"],

Cell[CellGroupData[{
Cell[95483, 3226, 232, 5, 34, "Input"],
Cell[95718, 3233, 27001, 574, 157, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[122756, 3812, 92, 2, 34, "Subsubsection"],
Cell[122851, 3816, 398, 10, 40, "Text"],
Cell[123252, 3828, 388, 9, 35, "Input"],
Cell[123643, 3839, 377, 12, 27, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[124057, 3856, 93, 2, 34, "Subsubsection"],
Cell[124153, 3860, 473, 20, 26, "Text"],

Cell[CellGroupData[{
Cell[124651, 3884, 335, 7, 60, "Input"],
Cell[124989, 3893, 21648, 487, 289, "Output"]
}, Open  ]],
Cell[146652, 4383, 273, 9, 26, "Text"],

Cell[CellGroupData[{
Cell[146950, 4396, 226, 6, 22, "Input"],
Cell[147179, 4404, 442, 9, 22, "Output"]
}, Open  ]],
Cell[147636, 4416, 203, 6, 22, "Input"],
Cell[147842, 4424, 232, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[148099, 4436, 226, 6, 34, "Input"],
Cell[148328, 4444, 1631, 39, 22, "Output"]
}, Open  ]],
Cell[149974, 4486, 1125, 33, 83, "Text"],

Cell[CellGroupData[{
Cell[151124, 4523, 216, 5, 34, "Input"],
Cell[151343, 4530, 733, 20, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[152113, 4555, 750, 20, 22, "Input"],
Cell[152866, 4577, 591, 18, 22, "Output"]
}, Open  ]],
Cell[153472, 4598, 89, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[153586, 4604, 122, 3, 22, "Input"],
Cell[153711, 4609, 601, 18, 24, "Output"]
}, Open  ]],
Cell[154327, 4630, 483, 20, 26, "Text"],

Cell[CellGroupData[{
Cell[154835, 4654, 251, 6, 47, "Input"],
Cell[155089, 4662, 3639, 83, 101, "Output"]
}, Open  ]],
Cell[158743, 4748, 101, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[158869, 4754, 113, 3, 22, "Input"],
Cell[158985, 4759, 693, 18, 35, "Output"]
}, Open  ]],
Cell[159693, 4780, 565, 18, 55, "Text"],

Cell[CellGroupData[{
Cell[160283, 4802, 334, 8, 47, "Input"],
Cell[160620, 4812, 773, 15, 45, "Output"]
}, Open  ]],
Cell[161408, 4830, 512, 16, 40, "Text"],

Cell[CellGroupData[{
Cell[161945, 4850, 134, 3, 22, "Input"],
Cell[162082, 4855, 240, 5, 22, "Output"]
}, Open  ]],
Cell[162337, 4863, 189, 5, 22, "Input"],
Cell[162529, 4870, 433, 16, 26, "Text"],

Cell[CellGroupData[{
Cell[162987, 4890, 359, 8, 72, "Input"],
Cell[163349, 4900, 2702, 53, 156, "Output"]
}, Open  ]],
Cell[166066, 4956, 217, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[166308, 4968, 215, 5, 22, "Input"],
Cell[166526, 4975, 1252, 23, 122, "Output"]
}, Open  ]],
Cell[167793, 5001, 284, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[168102, 5017, 110, 3, 22, "Input"],
Cell[168215, 5022, 2043, 38, 233, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[170307, 5066, 94, 2, 34, "Subsubsection"],
Cell[170404, 5070, 562, 24, 26, "Text"],

Cell[CellGroupData[{
Cell[170991, 5098, 418, 8, 60, "Input"],
Cell[171412, 5108, 117, 3, 22, "Output"]
}, Open  ]],
Cell[171544, 5114, 232, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[171801, 5126, 226, 6, 34, "Input"],
Cell[172030, 5134, 7726, 144, 579, "Output"]
}, Open  ]],
Cell[179771, 5281, 364, 13, 26, "Text"],

Cell[CellGroupData[{
Cell[180160, 5298, 365, 8, 60, "Input"],
Cell[180528, 5308, 1670, 37, 78, "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[182259, 5352, 165, 6, 43, "Subsection"],
Cell[182427, 5360, 1015, 31, 69, "Text"],
Cell[183445, 5393, 6549, 126, 574, "Input"]
}, Open  ]],

Cell[CellGroupData[{
Cell[190031, 5524, 143, 4, 43, "Subsection"],
Cell[190177, 5530, 455, 14, 40, "Text"],
Cell[190635, 5546, 251, 6, 22, "Input"],
Cell[190889, 5554, 214, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[191128, 5566, 146, 4, 22, "Input"],
Cell[191277, 5572, 1462, 31, 98, "Output"]
}, Open  ]],
Cell[192754, 5606, 146, 4, 26, "Text"],
Cell[192903, 5612, 488, 13, 54, "Text"],

Cell[CellGroupData[{
Cell[193416, 5629, 215, 5, 22, "Input"],
Cell[193634, 5636, 171, 4, 24, "Output"]
}, Open  ]],
Cell[193820, 5643, 112, 2, 26, "Text"],
Cell[193935, 5647, 244, 6, 34, "Input"],
Cell[194182, 5655, 241, 8, 26, "Text"],
Cell[194426, 5665, 251, 6, 22, "Input"],
Cell[194680, 5673, 214, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[194919, 5685, 148, 4, 22, "Input"],
Cell[195070, 5691, 1431, 30, 84, "Output"]
}, Open  ]],
Cell[196516, 5724, 132, 4, 26, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[196685, 5733, 166, 5, 43, "Subsection"],
Cell[196854, 5740, 174, 5, 26, "Text"],
Cell[197031, 5747, 32274, 1114, 148, 16326, 911, "GraphicsData", 
"PostScript", "Graphics"],
Cell[229308, 6863, 791, 23, 54, "Text"],

Cell[CellGroupData[{
Cell[230124, 6890, 101, 2, 34, "Subsubsection"],
Cell[230228, 6894, 261, 10, 26, "Text"],

Cell[CellGroupData[{
Cell[230514, 6908, 215, 5, 22, "Input"],
Cell[230732, 6915, 171, 4, 24, "Output"]
}, Open  ]],
Cell[230918, 6922, 219, 7, 26, "Text"],
Cell[231140, 6931, 244, 6, 34, "Input"],
Cell[231387, 6939, 234, 6, 40, "Text"],
Cell[231624, 6947, 248, 6, 22, "Input"],
Cell[231875, 6955, 214, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[232114, 6967, 148, 4, 22, "Input"],
Cell[232265, 6973, 1840, 37, 70, "Output"]
}, Open  ]],
Cell[234120, 7013, 175, 5, 26, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[234332, 7023, 104, 2, 34, "Subsubsection"],
Cell[234439, 7027, 470, 15, 40, "Text"],

Cell[CellGroupData[{
Cell[234934, 7046, 215, 5, 22, "Input"],
Cell[235152, 7053, 171, 4, 24, "Output"]
}, Open  ]],
Cell[235338, 7060, 219, 7, 26, "Text"],
Cell[235560, 7069, 244, 6, 34, "Input"],
Cell[235807, 7077, 236, 6, 40, "Text"],
Cell[236046, 7085, 251, 6, 22, "Input"],
Cell[236300, 7093, 214, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[236539, 7105, 148, 4, 22, "Input"],
Cell[236690, 7111, 1466, 31, 98, "Output"]
}, Open  ]],
Cell[238171, 7145, 182, 5, 26, "Text"],
Cell[238356, 7152, 187, 5, 26, "Text"],

Cell[CellGroupData[{
Cell[238568, 7161, 215, 5, 22, "Input"],
Cell[238786, 7168, 171, 4, 24, "Output"]
}, Open  ]],
Cell[238972, 7175, 219, 7, 26, "Text"],
Cell[239194, 7184, 244, 6, 34, "Input"],
Cell[239441, 7192, 236, 6, 40, "Text"],
Cell[239680, 7200, 251, 6, 22, "Input"],
Cell[239934, 7208, 214, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[240173, 7220, 148, 4, 22, "Input"],
Cell[240324, 7226, 1434, 30, 84, "Output"]
}, Open  ]],
Cell[241773, 7259, 182, 5, 26, "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[242004, 7270, 167, 5, 43, "Subsection"],
Cell[242174, 7277, 175, 5, 26, "Text"],
Cell[242352, 7284, 31139, 1043, 148, 15580, 845, "GraphicsData", 
"PostScript", "Graphics"],
Cell[273494, 8329, 347, 10, 40, "Text"],

Cell[CellGroupData[{
Cell[273866, 8343, 101, 2, 34, "Subsubsection"],
Cell[273970, 8347, 302, 11, 40, "Text"],

Cell[CellGroupData[{
Cell[274297, 8362, 215, 5, 22, "Input"],
Cell[274515, 8369, 172, 4, 24, "Output"]
}, Open  ]],
Cell[274702, 8376, 220, 7, 26, "Text"],
Cell[274925, 8385, 244, 6, 34, "Input"],
Cell[275172, 8393, 236, 6, 40, "Text"],
Cell[275411, 8401, 248, 6, 22, "Input"],
Cell[275662, 8409, 214, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[275901, 8421, 148, 4, 22, "Input"],
Cell[276052, 8427, 1858, 37, 98, "Output"]
}, Open  ]],
Cell[277925, 8467, 177, 5, 26, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[278139, 8477, 104, 2, 34, "Subsubsection"],
Cell[278246, 8481, 320, 11, 40, "Text"],

Cell[CellGroupData[{
Cell[278591, 8496, 215, 5, 22, "Input"],
Cell[278809, 8503, 172, 4, 24, "Output"]
}, Open  ]],
Cell[278996, 8510, 220, 7, 26, "Text"],
Cell[279219, 8519, 244, 6, 34, "Input"],
Cell[279466, 8527, 238, 6, 40, "Text"],
Cell[279707, 8535, 251, 6, 22, "Input"],
Cell[279961, 8543, 214, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[280200, 8555, 148, 4, 22, "Input"],
Cell[280351, 8561, 1466, 31, 98, "Output"]
}, Open  ]],
Cell[281832, 8595, 183, 5, 26, "Text"],
Cell[282018, 8602, 320, 11, 40, "Text"],

Cell[CellGroupData[{
Cell[282363, 8617, 215, 5, 22, "Input"],
Cell[282581, 8624, 172, 4, 24, "Output"]
}, Open  ]],
Cell[282768, 8631, 220, 7, 26, "Text"],
Cell[282991, 8640, 244, 6, 34, "Input"],
Cell[283238, 8648, 238, 6, 40, "Text"],
Cell[283479, 8656, 251, 6, 22, "Input"],
Cell[283733, 8664, 214, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[283972, 8676, 148, 4, 22, "Input"],
Cell[284123, 8682, 1454, 31, 84, "Output"]
}, Open  ]],
Cell[285592, 8716, 189, 5, 26, "Text"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[285842, 8728, 97, 2, 46, "Section"],
Cell[285942, 8732, 314, 7, 40, "Text"],
Cell[286259, 8741, 3451, 61, 194, "Input"],
Cell[289713, 8804, 245, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[289983, 8816, 298, 7, 34, "Input"],
Cell[290284, 8825, 440, 10, 33, "Output"]
}, Open  ]],
Cell[290739, 8838, 1224, 50, 55, "Text"],
Cell[291966, 8890, 445, 9, 72, "Input"],
Cell[292414, 8901, 133, 3, 26, "Text"],

Cell[CellGroupData[{
Cell[292572, 8908, 224, 5, 34, "Input"],
Cell[292799, 8915, 126069, 2671, 336, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[418905, 11591, 95, 2, 43, "Subsection"],
Cell[419003, 11595, 277, 8, 26, "Text"],
Cell[419283, 11605, 251, 6, 22, "Input"],
Cell[419537, 11613, 318, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[419880, 11629, 224, 5, 22, "Input"],
Cell[420107, 11636, 235, 5, 22, "Output"]
}, Open  ]],
Cell[420357, 11644, 380, 12, 40, "Text"],

Cell[CellGroupData[{
Cell[420762, 11660, 168, 4, 22, "Input"],
Cell[420933, 11666, 285, 6, 22, "Output"]
}, Open  ]],
Cell[421233, 11675, 439, 17, 26, "Text"],

Cell[CellGroupData[{
Cell[421697, 11696, 208, 6, 30, "Input"],
Cell[421908, 11704, 208, 5, 29, "Output"]
}, Open  ]],
Cell[422131, 11712, 253, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[422409, 11724, 232, 6, 22, "Input"],
Cell[422644, 11732, 220, 5, 22, "Output"]
}, Open  ]],
Cell[422879, 11740, 65, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[422969, 11746, 288, 7, 35, "Input"],
Cell[423260, 11755, 924, 18, 45, "Output"]
}, Open  ]],
Cell[424199, 11776, 1135, 37, 40, "Text"],
Cell[425337, 11815, 1697, 46, 111, "Text"],
Cell[427037, 11863, 311, 8, 34, "Input"],
Cell[427351, 11873, 182, 4, 26, "Text"],
Cell[427536, 11879, 172, 5, 22, "Input"],
Cell[427711, 11886, 290, 10, 26, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[428038, 11901, 96, 2, 43, "Subsection"],
Cell[428137, 11905, 363, 14, 26, "Text"],
Cell[428503, 11921, 371, 8, 47, "Input"],
Cell[428877, 11931, 232, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[429134, 11943, 175, 5, 22, "Input"],
Cell[429312, 11950, 1525, 40, 35, "Output"]
}, Open  ]],
Cell[430852, 11993, 1108, 32, 83, "Text"],

Cell[CellGroupData[{
Cell[431985, 12029, 1054, 25, 60, "Input"],
Cell[433042, 12056, 601, 18, 22, "Output"]
}, Open  ]],
Cell[433658, 12077, 584, 16, 55, "Text"],

Cell[CellGroupData[{
Cell[434267, 12097, 362, 8, 47, "Input"],
Cell[434632, 12107, 783, 15, 45, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[435452, 12127, 385, 8, 72, "Input"],
Cell[435840, 12137, 2115, 38, 103, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[437992, 12180, 103, 3, 22, "Input"],
Cell[438098, 12185, 83, 3, 22, "Output"]
}, Open  ]],
Cell[438196, 12191, 922, 24, 83, "Text"],
Cell[439121, 12217, 401, 8, 88, "Input"],
Cell[439525, 12227, 385, 14, 26, "Text"],
Cell[439913, 12243, 269, 6, 34, "Input"],
Cell[440185, 12251, 167, 7, 26, "Text"],

Cell[CellGroupData[{
Cell[440377, 12262, 188, 5, 22, "Input"],
Cell[440568, 12269, 478, 8, 45, "Output"]
}, Open  ]],
Cell[441061, 12280, 135, 3, 22, "Input"],
Cell[441199, 12285, 1115, 43, 40, "Text"],
Cell[442317, 12330, 341, 8, 34, "Input"],
Cell[442661, 12340, 774, 28, 40, "Text"],

Cell[CellGroupData[{
Cell[443460, 12372, 435, 9, 60, "Input"],
Cell[443898, 12383, 180, 5, 20, "Message"],
Cell[444081, 12390, 4358, 82, 244, "Output"]
}, Open  ]],
Cell[448454, 12475, 156, 5, 26, "Text"],

Cell[CellGroupData[{
Cell[448635, 12484, 735, 14, 74, "Input"],
Cell[449373, 12500, 908, 15, 72, "Output"]
}, Open  ]],
Cell[450296, 12518, 108, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[450429, 12524, 190, 5, 34, "Input"],
Cell[450622, 12531, 1392, 23, 131, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[452051, 12559, 225, 5, 35, "Input"],
Cell[452279, 12566, 1569, 31, 109, "Output"]
}, Open  ]],
Cell[453863, 12600, 112, 2, 26, "Text"],
Cell[453978, 12604, 364, 9, 34, "Input"]
}, Open  ]],

Cell[CellGroupData[{
Cell[454379, 12618, 97, 2, 43, "Subsection"],
Cell[454479, 12622, 372, 16, 26, "Text"],

Cell[CellGroupData[{
Cell[454876, 12642, 465, 11, 60, "Input"],
Cell[455344, 12655, 117, 3, 22, "Output"]
}, Open  ]],
Cell[455476, 12661, 315, 9, 40, "Text"],
Cell[455794, 12672, 226, 6, 22, "Input"],
Cell[456023, 12680, 358, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[456406, 12696, 536, 12, 72, "Input"],
Cell[456945, 12710, 117, 3, 22, "Output"]
}, Open  ]],
Cell[457077, 12716, 254, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[457356, 12728, 638, 12, 85, "Input"],
Cell[457997, 12742, 798, 14, 46, "Output"]
}, Open  ]],
Cell[458810, 12759, 193, 8, 26, "Text"],
Cell[459006, 12769, 129, 3, 22, "Input"],
Cell[459138, 12774, 81, 2, 26, "Text"],
Cell[459222, 12778, 179, 5, 22, "Input"],

Cell[CellGroupData[{
Cell[459426, 12787, 282, 6, 47, "Input"],
Cell[459711, 12795, 448, 9, 33, "Output"]
}, Open  ]],
Cell[460174, 12807, 209, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[460408, 12819, 255, 6, 47, "Input"],
Cell[460666, 12827, 2688, 55, 79, "Output"]
}, Open  ]],
Cell[463369, 12885, 446, 16, 40, "Text"],

Cell[CellGroupData[{
Cell[463840, 12905, 398, 8, 74, "Input"],
Cell[464241, 12915, 2583, 53, 79, "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[466885, 12975, 88, 2, 46, "Section"],
Cell[466976, 12979, 987, 35, 69, "Text"],
Cell[467966, 13016, 117, 3, 22, "Input"],

Cell[CellGroupData[{
Cell[468108, 13023, 130, 3, 22, "Input"],
Cell[468241, 13028, 621, 19, 22, "Output"]
}, Open  ]],
Cell[468877, 13050, 1424, 26, 163, "Input"],
Cell[470304, 13078, 94, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[470423, 13084, 354, 7, 47, "Input"],
Cell[470780, 13093, 1225, 32, 45, "Output"]
}, Open  ]],
Cell[472020, 13128, 1235, 42, 69, "Text"],
Cell[473258, 13172, 367, 13, 26, "Text"],
Cell[473628, 13187, 136, 3, 22, "Input"],
Cell[473767, 13192, 101, 2, 26, "Text"],
Cell[473871, 13196, 185, 4, 24, "Input"],
Cell[474059, 13202, 273, 6, 34, "Input"],
Cell[474335, 13210, 515, 20, 26, "Text"],
Cell[474853, 13232, 233, 5, 23, "Input"],

Cell[CellGroupData[{
Cell[475111, 13241, 135, 3, 22, "Input"],
Cell[475249, 13246, 1225, 32, 45, "Output"]
}, Open  ]],
Cell[476489, 13281, 225, 8, 26, "Text"],
Cell[476717, 13291, 288, 8, 34, "Input"],
Cell[477008, 13301, 73, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[477106, 13307, 179, 5, 47, "Input"],
Cell[477288, 13314, 467, 9, 61, "Output"],
Cell[477758, 13325, 434, 8, 61, "Output"],
Cell[478195, 13335, 899, 16, 73, "Output"]
}, Open  ]],
Cell[479109, 13354, 581, 22, 40, "Text"],
Cell[479693, 13378, 286, 6, 34, "Input"],
Cell[479982, 13386, 132, 4, 26, "Text"],

Cell[CellGroupData[{
Cell[480139, 13394, 332, 9, 34, "Input"],
Cell[480474, 13405, 381, 9, 33, "Output"]
}, Open  ]],
Cell[480870, 13417, 256, 10, 26, "Text"],
Cell[481129, 13429, 383, 13, 40, "Text"],
Cell[481515, 13444, 202, 5, 22, "Input"],
Cell[481720, 13451, 712, 28, 40, "Text"],
Cell[482435, 13481, 243, 6, 34, "Input"],
Cell[482681, 13489, 127, 3, 22, "Input"],
Cell[482811, 13494, 315, 12, 26, "Text"],
Cell[483129, 13508, 228, 5, 34, "Input"],

Cell[CellGroupData[{
Cell[483382, 13517, 257, 6, 47, "Input"],
Cell[483642, 13525, 347, 7, 33, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[484026, 13537, 242, 6, 47, "Input"],
Cell[484271, 13545, 4037, 94, 124, "Output"]
}, Open  ]],
Cell[488323, 13642, 98, 2, 26, "Text"],
Cell[488424, 13646, 810, 16, 92, "Input"],
Cell[489237, 13664, 199, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[489461, 13676, 186, 4, 22, "Input"],
Cell[489650, 13682, 19930, 441, 218, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[509617, 14128, 95, 2, 43, "Subsection"],
Cell[509715, 14132, 310, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[510050, 14148, 128, 3, 22, "Input"],
Cell[510181, 14153, 1181, 30, 28, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[511399, 14188, 106, 3, 22, "Input"],
Cell[511508, 14193, 1255, 32, 47, "Output"]
}, Open  ]],
Cell[512778, 14228, 343, 12, 26, "Text"],
Cell[513124, 14242, 253, 6, 35, "Input"],
Cell[513380, 14250, 340, 13, 26, "Text"],
Cell[513723, 14265, 574, 12, 35, "Input"]
}, Open  ]],

Cell[CellGroupData[{
Cell[514334, 14282, 96, 2, 43, "Subsection"],
Cell[514433, 14286, 289, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[514747, 14302, 355, 7, 60, "Input"],
Cell[515105, 14311, 117, 3, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[515259, 14319, 355, 7, 60, "Input"],
Cell[515617, 14328, 117, 3, 22, "Output"]
}, Open  ]],
Cell[515749, 14334, 101, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[515875, 14340, 107, 3, 22, "Input"],
Cell[515985, 14345, 204, 4, 22, "Output"]
}, Open  ]],
Cell[516204, 14352, 576, 19, 40, "Text"],
Cell[516783, 14373, 227, 5, 34, "Input"],
Cell[517013, 14380, 144, 4, 22, "Input"],

Cell[CellGroupData[{
Cell[517182, 14388, 132, 3, 22, "Input"],
Cell[517317, 14393, 4081, 71, 212, "Output"]
}, Open  ]],
Cell[521413, 14467, 270, 9, 26, "Text"],

Cell[CellGroupData[{
Cell[521708, 14480, 133, 3, 22, "Input"],
Cell[521844, 14485, 436, 7, 45, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[522317, 14497, 115, 3, 22, "Input"],
Cell[522435, 14502, 436, 7, 45, "Output"]
}, Open  ]],
Cell[522886, 14512, 95, 2, 26, "Text"],
Cell[522984, 14516, 299, 8, 34, "Input"],
Cell[523286, 14526, 798, 32, 40, "Text"],

Cell[CellGroupData[{
Cell[524109, 14562, 247, 6, 47, "Input"],
Cell[524359, 14570, 117, 3, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[524513, 14578, 247, 6, 47, "Input"],
Cell[524763, 14586, 117, 3, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[524917, 14594, 196, 5, 34, "Input"],
Cell[525116, 14601, 12479, 265, 1156, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[537644, 14872, 97, 2, 43, "Subsection"],
Cell[537744, 14876, 467, 20, 26, "Text"],

Cell[CellGroupData[{
Cell[538236, 14900, 343, 7, 60, "Input"],
Cell[538582, 14909, 117, 3, 22, "Output"]
}, Open  ]],
Cell[538714, 14915, 101, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[538840, 14921, 106, 3, 22, "Input"],
Cell[538949, 14926, 1255, 32, 47, "Output"]
}, Open  ]],
Cell[540219, 14961, 243, 8, 26, "Text"],
Cell[540465, 14971, 181, 5, 22, "Input"],
Cell[540649, 14978, 831, 25, 55, "Text"],
Cell[541483, 15005, 301, 8, 40, "Text"],
Cell[541787, 15015, 178, 5, 22, "Input"],
Cell[541968, 15022, 73, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[542066, 15028, 221, 6, 22, "Input"],
Cell[542290, 15036, 117, 3, 22, "Output"]
}, Open  ]],
Cell[542422, 15042, 358, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[542805, 15058, 191, 5, 22, "Input"],
Cell[542999, 15065, 148, 4, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[543184, 15074, 191, 5, 22, "Input"],
Cell[543378, 15081, 148, 4, 22, "Output"]
}, Open  ]],
Cell[543541, 15088, 200, 8, 26, "Text"],
Cell[543744, 15098, 122, 3, 22, "Input"],
Cell[543869, 15103, 122, 3, 22, "Input"],
Cell[543994, 15108, 295, 11, 26, "Text"],
Cell[544292, 15121, 147, 4, 22, "Input"],

Cell[CellGroupData[{
Cell[544464, 15129, 288, 7, 34, "Input"],
Cell[544755, 15138, 102, 3, 28, "Output"]
}, Open  ]],
Cell[544872, 15144, 299, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[545196, 15159, 288, 7, 34, "Input"],
Cell[545487, 15168, 102, 3, 28, "Output"]
}, Open  ]],
Cell[545604, 15174, 184, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[545813, 15186, 102, 3, 22, "Input"],
Cell[545918, 15191, 124, 3, 28, "Output"]
}, Open  ]],
Cell[546057, 15197, 352, 12, 26, "Text"],
Cell[546412, 15211, 203, 6, 22, "Input"],
Cell[546618, 15219, 334, 8, 60, "Input"],
Cell[546955, 15229, 334, 8, 60, "Input"],
Cell[547292, 15239, 145, 4, 26, "Text"],
Cell[547440, 15245, 337, 7, 22, "Input"],

Cell[CellGroupData[{
Cell[547802, 15256, 289, 7, 47, "Input"],
Cell[548094, 15265, 300, 6, 34, "Output"]
}, Open  ]],
Cell[548409, 15274, 400, 16, 26, "Text"],

Cell[CellGroupData[{
Cell[548834, 15294, 142, 4, 22, "Input"],
Cell[548979, 15300, 605, 13, 69, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[549621, 15318, 142, 4, 22, "Input"],
Cell[549766, 15324, 612, 14, 69, "Output"]
}, Open  ]],
Cell[550393, 15341, 296, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[550714, 15357, 374, 8, 47, "Input"],
Cell[551091, 15367, 2995, 64, 199, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[554135, 15437, 89, 2, 43, "Subsection"],
Cell[554227, 15441, 392, 13, 40, "Text"],
Cell[554622, 15456, 218, 5, 22, "Input"],
Cell[554843, 15463, 533, 20, 40, "Text"],
Cell[555379, 15485, 220, 5, 22, "Input"],
Cell[555602, 15492, 226, 8, 26, "Text"],
Cell[555831, 15502, 288, 8, 34, "Input"],
Cell[556122, 15512, 69, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[556216, 15518, 162, 4, 22, "Input"],
Cell[556381, 15524, 353, 7, 28, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[556771, 15536, 162, 4, 22, "Input"],
Cell[556936, 15542, 350, 6, 28, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[557323, 15553, 206, 5, 22, "Input"],
Cell[557532, 15560, 1148, 23, 73, "Output"]
}, Open  ]],
Cell[558695, 15586, 280, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[559000, 15601, 133, 3, 22, "Input"],
Cell[559136, 15606, 190, 4, 28, "Output"]
}, Open  ]],
Cell[559341, 15613, 459, 16, 40, "Text"],

Cell[CellGroupData[{
Cell[559825, 15633, 116, 3, 22, "Input"],
Cell[559944, 15638, 115, 3, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[560096, 15646, 113, 3, 22, "Input"],
Cell[560212, 15651, 108, 3, 22, "Output"]
}, Open  ]],
Cell[560335, 15657, 340, 13, 26, "Text"],
Cell[560678, 15672, 255, 7, 22, "Input"],
Cell[560936, 15681, 371, 16, 26, "Text"],

Cell[CellGroupData[{
Cell[561332, 15701, 271, 6, 47, "Input"],
Cell[561606, 15709, 10814, 217, 783, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[572457, 15931, 271, 6, 47, "Input"],
Cell[572731, 15939, 10814, 217, 792, "Output"]
}, Open  ]],
Cell[583560, 16159, 574, 19, 40, "Text"],

Cell[CellGroupData[{
Cell[584159, 16182, 347, 7, 60, "Input"],
Cell[584509, 16191, 1267, 25, 82, "Output"]
}, Open  ]],
Cell[585791, 16219, 265, 9, 26, "Text"],

Cell[CellGroupData[{
Cell[586081, 16232, 133, 3, 22, "Input"],
Cell[586217, 16237, 191, 4, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[586445, 16246, 115, 3, 22, "Input"],
Cell[586563, 16251, 191, 4, 22, "Output"]
}, Open  ]],
Cell[586769, 16258, 95, 2, 26, "Text"],
Cell[586867, 16262, 247, 6, 34, "Input"],
Cell[587117, 16270, 792, 32, 40, "Text"],
Cell[587912, 16304, 201, 5, 22, "Input"],
Cell[588116, 16311, 201, 5, 22, "Input"],

Cell[CellGroupData[{
Cell[588342, 16320, 220, 5, 47, "Input"],
Cell[588565, 16327, 3213, 65, 282, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[591815, 16397, 94, 2, 34, "Subsubsection"],
Cell[591912, 16401, 395, 14, 40, "Text"],
Cell[592310, 16417, 218, 6, 22, "Input"],
Cell[592531, 16425, 698, 26, 40, "Text"],
Cell[593232, 16453, 390, 8, 72, "Input"],
Cell[593625, 16463, 133, 3, 26, "Text"],
Cell[593761, 16468, 157, 4, 22, "Input"],
Cell[593921, 16474, 380, 15, 26, "Text"],
Cell[594304, 16491, 402, 10, 22, "Input"],
Cell[594709, 16503, 355, 16, 26, "Text"],
Cell[595067, 16521, 409, 9, 47, "Input"],
Cell[595479, 16532, 363, 14, 26, "Text"],

Cell[CellGroupData[{
Cell[595867, 16550, 268, 6, 47, "Input"],
Cell[596138, 16558, 117, 3, 22, "Output"]
}, Open  ]],
Cell[596270, 16564, 239, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[596534, 16576, 222, 6, 22, "Input"],
Cell[596759, 16584, 116, 3, 22, "Output"]
}, Open  ]],
Cell[596890, 16590, 358, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[597273, 16606, 191, 5, 22, "Input"],
Cell[597467, 16613, 148, 4, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[597652, 16622, 191, 5, 22, "Input"],
Cell[597846, 16629, 148, 4, 22, "Output"]
}, Open  ]],
Cell[598009, 16636, 200, 8, 26, "Text"],
Cell[598212, 16646, 122, 3, 22, "Input"],
Cell[598337, 16651, 122, 3, 22, "Input"],
Cell[598462, 16656, 295, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[598782, 16671, 285, 7, 34, "Input"],
Cell[599070, 16680, 102, 3, 28, "Output"]
}, Open  ]],
Cell[599187, 16686, 299, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[599511, 16701, 285, 7, 34, "Input"],
Cell[599799, 16710, 102, 3, 28, "Output"]
}, Open  ]],
Cell[599916, 16716, 184, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[600125, 16728, 102, 3, 22, "Input"],
Cell[600230, 16733, 124, 3, 28, "Output"]
}, Open  ]],
Cell[600369, 16739, 352, 12, 26, "Text"],
Cell[600724, 16753, 323, 7, 47, "Input"],
Cell[601050, 16762, 323, 7, 47, "Input"],
Cell[601376, 16771, 132, 4, 26, "Text"],

Cell[CellGroupData[{
Cell[601533, 16779, 111, 3, 22, "Input"],
Cell[601647, 16784, 85, 3, 22, "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

